/* assembler
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "parser.h"
#include "regalloc.h"
#include "cstringf.h"
#include "varlist.h"

FILE *assembler_file;
int location;
int last_location;
int *image;
boolean *location_used;
tree *assertions;
loc_t last_loc = NULL;
int assembled_code_size;
int assembled_page_instructions = 0;
int assembled_bank_instructions = 0;

void show_line(loc_t loc)
{
    stack_guard;
    if (assembler_file == NULL)
        return;
    if ((loc != NULL)
        && ((last_loc == NULL) || (loc->line_nr != last_loc->line_nr))) {
        if (verbose_assembler) {
            log((m, "l=%d ln=%d s=%s", loc, loc->line_nr, loc->line));
        }
        if (loc == nowhere) {
            fprintf(assembler_file, "\n");
        } else {
            fprintf(assembler_file, "\n;; %03d : %s\n", loc->line_nr, loc->line);
            if (result_gpsim_info) {
                fprintf(assembler_file, ";#CSRC %s %d\n", loc->file_name, loc->line_nr);
            }
        }
        last_loc = loc;
    }
}

void emit_assembley(tree p, char *line, int phase)
{
    stack_guard;
    if (assembler_file == NULL)
        return;
    if (line[0] != '\0') {
        if (phase == 2) {
            show_line(p->loc);
            fprintf(assembler_file, "%s\n", line);
#ifdef __DEBUG__
            fprintf(assembler_file, "%s\n", line);
#endif
        }
#ifdef __DEBUG__
        printf("%s\n", line);
#endif
    }
}

void assemble_test(tree p, int phase)
{
    string line = "";
    stack_guard;
    assert_kind(NULL, p, node_test);

#ifdef __DEBUG__
    sprintf(line, "; cassert");
#endif
    cassert(p->next == NULL);
    if (phase == 2) {
        if (assertions[location] == NULL) {
            assertions[location] = p;
        } else {
            tree q = assertions[location];
            while (q->next != NULL) {
                q = q->next;
            }
            q->next = p;
        }
    }

    emit_assembley(p, line, phase);
}


void assemble_type(tree p, int phase)
{
    string line = "";
    stack_guard;
    assert_kind(NULL, p, node_type);

    sprintf(line, "; type  %s", p->name);

    emit_assembley(p, line, phase);
}

void assemble_var(tree p, int phase)
{
    string line = "";
    stack_guard;
    assert_kind(NULL, p, node_var);

    if (p->indirect) {
        sprintf(line, "; var (virtual) %s ", p->name);
    } else {

        if (p->address == NULL) {
            trace show_subtree(p);
        trace}

        assert_kind(p->loc, p->address, node_chain);
        sprintf(line, "; var H'%03X:%03X' %s", p->address->first->x, p->address->next->x, p->name);
        if (phase == 2) {
            add_variable(p);
        }
    }

    emit_assembley(p, line, phase);
}

void assemble_const(tree p, int phase)
{
    string line = "";
    stack_guard;
    assert_kind(NULL, p, node_const);

    assert_pointer(p->loc, p->value);
    sprintf(line, "; const H'%03X' %s", p->value->x, p->name);

    emit_assembley(p, line, phase);
}

void assemble_label(tree p, int phase)
{
    string line = "";
    stack_guard;
    assert_kind(NULL, p, node_label);

    p->x = location;
    sprintf(line, "%s: ; %04X", p->label, p->x);

    emit_assembley(p, line, phase);
}

void assemble_org(tree p, int phase)
{
    string line = "";
    int n;
    stack_guard;
    assert_kind(NULL, p, node_org);
    n = p->x;
    if (target_cpu == pic_16) {
        n = n * 2;
    }

    sprintf(line, " ORG %04X", n);
    location = p->x;

    emit_assembley(p, line, phase);
}

void assemble_asm(tree p, int phase)
{
    string label, value, file, comment, dest, temp;
    int opcode, n, label_value, file_address;
    string line = "";
    char *op_string;
    //boolean fizzle = false;
    stack_guard;
    assert_kind(NULL, p, node_asm);

    label_value = file_address = 0;

    if (p->kind == node_w) {
        return;
    }

    if (p->opcode == opcode_movfw) {
        p->opcode = opcode_movf;
        p->dest = dest_w;
    }
    opcode = opcode_value[p->opcode];
    if (opcode == -1) {
        switch (p->opcode) {
        case opcode_page:
            if ((target_cpu == pic_16)) {
                p->fizzle = true;
            } else if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)      /* Added 16f876 & 16f873 by Javi 2003-03-01 */
                       ||(target_chip == t_16f88)
                ) {
                if (phase == 1) {
                    location += 2;
                    return;
                } else {
                    opcode_t op1, op2;
                    tree asm1, asm2;
                    int x = -1;
                    char *name = NULL;
                    tree a = p->first;
                    assert_pointer(p->loc, a);
                    if (a->kind == node_ref) {
                        a = a->first;
                        assert_pointer(p->loc, a);
                    }
                    assert_pointer(p->loc, p->first->name);
                    switch (a->kind) {
                    case node_const:{
                            assert_pointer(a->loc, a->value);
                            x = a->value->x;
                            break;
                        }
                    case node_label:{
                            x = a->x;
                            name = a->label;
                            break;
                        }
                    case node_procedure:{
                            assert_pointer(a->loc, a->address);
                            assert_pointer(a->loc, a->call_label);
                            x = a->call_label->x;
                            name = a->call_label->name;
                            break;
                        }
                    default:{
                            snark_node(a->loc, a);
                            break;
                        }
                    }
                    assert_pointer(p->loc, p->first->name);
                    op1 = ((x & 0x0800) == 0) ? opcode_bcf : opcode_bsf;
                    op2 = ((x & 0x1000) == 0) ? opcode_bcf : opcode_bsf;
                    asm1 = new_node(p->loc, node_asm);
                    asm1->opcode = op1;
                    asm1->first = pic_pclath;
                    asm1->next = new_value(type_universal, 3);
                    assemble_asm(asm1, phase);
                    asm2 = new_node(p->loc, node_asm);
                    asm2->opcode = op2;
                    asm2->first = pic_pclath;
                    asm2->next = new_value(type_universal, 4);
                    assemble_asm(asm2, phase);
                    return;
                }
                /* could previously  be ignored */
                return;
            }
        case opcode_bank:{
                if (target_cpu == pic_14) { 
                
               int x = 0;

                    opcode_t op3, op4;
                    tree asm3, asm4;

                    tree q = p->first;
                    assert_pointer(p->loc, q);
  
               while (q->kind == node_ref) {
                    q = q->first;
                   }
                switch (q->kind) {
                case node_var:{
            
                        if (q->address == NULL) {
                            snark_node(q->loc, q);
                        }
                        x = q->address->first->x;
                        break;
                    }
                case node_value:{
                       /* x = q->x;  */
                        /*  break; */
                       return;
                    }
                case node_const:{
                       /* x = q->value->x; */
                         /*  break; */
                       return;
                    }
                default:{
                       /* snark_node(q->loc, q); */
                       /*  break; */
                       return;
                    }
                }
                if (phase == 1) {
                    if (!(p->fizzle))
                    {
                       location += 2; 
                    }
                    return;
                } else {

                    if (!(p->fizzle))
                    {
                    op3 = ((x & 0x0080) == 0) ? opcode_bcf : opcode_bsf;
                    op4 = ((x & 0x0100) == 0) ? opcode_bcf : opcode_bsf;
                    asm3 = new_node(p->loc, node_asm);
                    asm3->opcode = op3;
                    asm3->first = pic_flags;
                    asm3->next = new_value(type_universal, 5);
                    assemble_asm(asm3, phase);
                    asm4 = new_node(p->loc, node_asm);
                    asm4->opcode = op4;
                    asm4->first = pic_flags;
                    asm4->next = new_value(type_universal, 6);
                    assemble_asm(asm4, phase);                    
                    }
                    }
                    return;
                }
                if ( (target_chip == t_12f675)
                    || (target_chip == t_12f629)
                    ) {
                    /* can be ignored */
                    return;
                }
                if (target_cpu == pic_16) {
                    /* can be ignored */

               int xx = 0;

                    opcode_t op5;
                    tree asm5;

                    tree qq = p->first;
                    assert_pointer(p->loc, qq);
  
               while (qq->kind == node_ref) {
                    qq = qq->first;
                   }
                switch (qq->kind) {
                case node_var:{
            
                        if (qq->address == NULL) {
                            snark_node(qq->loc, qq);
                        }
                        xx = qq->address->first->x;
                        break;
                    }
                case node_value:{
                       /* xx = qq->x; */
                       /* break; */
                       return ;
                    }
                case node_const:{
                       /* xx = qq->value->x;  */
                       /*  break;  */
                       return ;
                    }
                default:{
                        /* snark_node(qq->loc, qq); */
                        /*  break;  */
                       return ;
                    }
                }
                if (phase == 1) {
                    if (!(p->fizzle) && ( (xx > 0x07F) && (xx < 0xF80) ))
                    {
                       location += 1; 
                    }
                    return;
                } else {
                    if (!(p->fizzle))
                    {
                      if ( ( (xx > 0x07F) && (xx < 0xF80) )  )
                       {
                        op5 = opcode_movlb ; 
                        xx = (xx>>8)&0x0F;
                        asm5 = new_node(p->loc, node_asm);
                        asm5->opcode = op5;
                        asm5->first = new_value(type_universal, xx);
                        assemble_asm(asm5, phase);
                       }
                      }
                    }


                    return;
                }
                break;
            }
        case opcode_hpage:{
                if (phase == 2) {
                    int x = 0;
                    tree a = p->first;
                    assert_pointer(p->loc, a);
                    if (a->kind == node_ref) {
                        a = a->first;
                        assert_pointer(p->loc, a);
                    }
                    assert_pointer(p->loc, p->first->name);
                    switch (a->kind) {
                    case node_const:{
                            assert_pointer(a->loc, a->value);
                            x = a->value->x;
                            break;
                        }
                    case node_label:{
                            x = a->x;
                            break;
                        }
                    case node_procedure:{
                            assert_pointer(a->loc, a->address);
                            assert_pointer(a->loc, a->call_label);
                            x = a->call_label->x;
                            break;
                        }
                    default:{
                            snark_node(a->loc, a);
                            break;
                        }
                    }
                    if (x == -1) {
                        trace_subtree(a);
                    }
                    jal_assert(p->loc, x != -1);
                    p->opcode = opcode_movlw;
                    n = x >> 8;
                    p->first = new_const(new_value(type_universal, n));
                }
                break;
            }
        default:{
                log((m, "nr=%d opcode '%s'", p->nr, opcode_name[p->opcode]));
                jal_assert(p->loc, false);
                break;
            }
        }
    }
    sprintf(label, "");
    sprintf(value, "");
    sprintf(file, "");
    sprintf(dest, "");
    sprintf(comment, "");

    opcode = opcode_value[p->opcode];
    if (code_has(p->opcode, field_const)) {
        tree q = p->first;
        tree v = NULL;
        assert_pointer(NULL, q);
        if (q->kind == node_ref)
            q = q->first;
        assert_pointer(NULL, q);
        if (q->kind == node_const)
            q = q->value;
        switch (q->kind) {
        case node_value:{
                v = q;
                break;
            }
        case node_var:{
                assert_kind(NULL, q->address, node_chain);
                assert_kind(NULL, q->address->first, node_value);
                v = q->address->first;
                break;
            }
        default:{
                snark_node(p->loc, q);
            }
        }
        assert_kind(NULL, v, node_value);
        /* p->first->type = type_byte; */
        /* chop( p->first ); */
	if ((target_cpu == pic_16) && (p->opcode == opcode_movlb)) {
           sprintf(value, "H'%01X'", (v->x &0x0F));
           sprintf(comment, "%d", (v->x &0x0F));
           opcode |= (v->x & 0x0F);
	} else {
           sprintf(value, "H'%02X'", v->x);
           sprintf(comment, "%d", v->x);
           opcode |= (v->x & 0xFF);
	}
    }

    if (code_has(p->opcode, field_tris)) {
        assert_kind(p->loc, p->first, node_ref);
        assert_kind(p->loc, p->first->first, node_const);
        /* p->first->type = type_byte; */
        /* chop( p->first ); */
        sprintf(value, "H'%03X'", p->first->first->value->x);
        sprintf(comment, "%d", p->first->first->value->x);
        opcode |= (p->first->first->value->x & 0xFF);
    }

    if (code_has(p->opcode, field_file)) {
        int x, y, z;
        tree q = p->first;

        x = y = z = 0;

        assert_pointer(NULL, q);
        if (q->kind == node_ref)
            q = q->first;
        assert_pointer(NULL, q);
        /* chop( q ); */
        switch (q->kind) {
        case node_var:{
                assert_pointer(q->loc, q->address);
                assert_pointer(q->loc, q->address->first);
                x = q->address->first->x;
                jal_assert(q->loc, x < 4096);
                break;
            }
        case node_value:{
                x = q->x;
                break;
            }
        case node_const:{
                assert_pointer(NULL, q->value);
                x = q->value->x;
                break;
            }
        default:{
                snark_node(q->loc, q);
            }
        }
        y = x;
        z = x;
        if ((target_cpu == sx_12) || (target_cpu == pic_12)) {
            /* calculate banked address y from linear x */
/*            if (x < 32) { */
                y = x;
/*            } else { */
/*                y = (x % 16) + ((x / 16) * 32) - 16;  */
/*            }  */
            z = y;
            /* calculate bank bits */
            if (p->opcode == opcode_bank) {
                if (x < 32) {
                  /*  p->fizzle = true; */
#ifdef __DEBUG__
                    snark_node(p->loc, p);
#endif
                }
                z = ((x >> 5) & 0x7);    /* was 4 */
/*                if (z > 0)
                    z = z - 1;  */
            } else {  
                z = y % 32;
            }  
        }
        file_address = y;
        if ((target_cpu == sx_12) && (p->opcode != opcode_bank)) {
            sprintf(value, "(H'1F' & H'%02X')", y);
        } else {
            sprintf(file, "H'%03X'", y);
        }
        sprintf(comment, "0x%03X %s", x, q->name);
        if (target_cpu != pic_16) {
            opcode |= (z & 0x7F);
        } else {
            if ((z > 0x7F) && (z < 0xF80)) {  /* was (z<0xF60) ?? */
                opcode |= (z & 0xFF) | 0x100;   /* BSR selection a=1 */
            } else {
                opcode |= (z & 0xFF);          /* Access selection a=0 */
            }
        }
    }

    if (code_has(p->opcode, field_label)) {
        int x = -1;
        char *name = NULL;
        tree a = p->first;
        assert_pointer(p->loc, a);
        if (a->kind == node_ref) {
            a = a->first;
            assert_pointer(p->loc, a);
        }
        if ((a->kind == node_const)
            && (a->value != NULL)
            && (a->value->kind == node_label)
            ) {
            a = a->value;
            assert_pointer(p->loc, a);
        }
        assert_pointer(p->loc, p->first->name);
        switch (a->kind) {
        case node_const:{
                assert_pointer(a->loc, a->value);
                x = a->value->x;
                break;
            }
        case node_label:{
                if (phase == 1) {
                    x = 0;
                    name = NULL;
                } else {
                    x = a->x;
                    name = a->label;
                    if (x == -1) {
                        trace_subtree(a);
                    }
                    jal_assert(p->loc, x != -1);
                }
                break;
            }
        case node_procedure:{
                if (phase == 1) {
                    x = 0;
                    name = NULL;
                    a->used++;
                } else {
                    assert_pointer(a->loc, a->address);
                    assert_pointer(a->loc, a->call_label);
                    x = a->call_label->x;
                    name = a->call_label->name;
                    jal_assert(p->loc, x != -1);
                }
                break;
            }
        default:{
                snark_node(a->loc, a);
                break;
            }
        }
        assert_pointer(p->loc, p->first->name);
        if (name == NULL) {
            sprintf(label, "H'%05X'", x);
            sprintf(comment, "%s", p->first->name);
        } else {
            sprintf(label, "%s", name);
            sprintf(comment, "H'%05X'", x);
        }
        label_value = x;

        if ((target_cpu == sx_12) || (target_cpu == pic_12)) {
            if (p->opcode == opcode_page) {
                x = (x >> 9) & 0x07;
            } else if (p->opcode == opcode_call) {
                x = x & 0x0FF;
            } else {
                x = x & 0x1FF;
            }
        }
        sprintf(comment, "%s [%04X]", p->first->name, x);
        if (target_cpu != pic_16) {
            opcode |= (x & 0x7FF);
        } else {
            if (p->opcode == opcode_a2nd) {
                opcode |= ((x & 0xFFF00) >> 8);
            } else if ((p->opcode == opcode_bra) || (p->opcode == opcode_rcall)) {
                opcode |= ((x-location-1) & 0x007FF) ;
            } else if ((p->opcode == opcode_bc) || (p->opcode == opcode_bn) ||
	               (p->opcode == opcode_bnc) || (p->opcode == opcode_bnn) ||
	               (p->opcode == opcode_bnov) || (p->opcode == opcode_bnz) ||
		       (p->opcode == opcode_bov) || (p->opcode == opcode_bz) ) {
                opcode |= ((x-location-1) & 0x000FF) ;
            } else {
                opcode |= (x & 0xFF);
            }

        }
    }

    if (code_has(p->opcode, field_dest)) {
        sprintf(dest, ",%c", p->dest);
        if ((target_cpu == sx_12) || (target_cpu == pic_12)) {
            opcode |= ((p->dest == dest_f) ? 1 : 0) << 5;
        } else if (target_cpu == pic_14) {
            opcode |= ((p->dest == dest_f) ? 1 : 0) << 7;
        } else {
            opcode |= ((p->dest == dest_f) ? 1 : 0) << 9;
        }
    }

    if (code_has(p->opcode, field_bit)) {
        tree q = p->next;
        assert_pointer(p->loc, q);
        if (q->kind == node_ref)
            q = q->first;
        assert_pointer(p->loc, q);
        if (q->kind == node_const)
            q = q->value;
        switch (q->kind) {
        case node_value:{
                break;
            }
        case node_var:{
                assert_kind(NULL, q->address, node_chain);
                assert_kind(NULL, q->address->next, node_value);
                q = q->address->next;
                break;
            }
        default:{
                snark_node(p->loc, q);
                break;
            }
        }
        assert_kind(p->loc, q, node_value);
        sprintf(dest, ",%d", q->x);
        if ((target_cpu == sx_12) || (target_cpu == pic_12)) {
            opcode |= (q->x & 0x07) << 5;
        } else if (target_cpu == pic_14) {
            opcode |= (q->x & 0x07) << 7;
        } else {
            opcode |= (q->x & 0x07) << 9;
        }
    }

    if ( (code_has(p->opcode, field_fsr) )) {
        int x, y, z;
        tree q = p->first;

        x = y = z = 0;

        assert_pointer(NULL, q);
        if (q->kind == node_ref)
            q = q->first;
        assert_pointer(NULL, q);
        /* chop( q ); */
        switch (q->kind) {
        case node_value:{
                x = q->x;
                break;
            }
        case node_const:{
                assert_pointer(NULL, q->value);
                x = q->value->x;
                break;
            }
        default:{
                snark_node(q->loc, q);
            }
        }
        y = x;
        z = x;
        if (p->opcode == opcode_lfsr) {
           sprintf(file, "%d", y);
           opcode |= ((y & 0x03) << 4) ;
	}
    }

    if ( (code_has(p->opcode, field_flabel) )) {
        int xx, yy, zz;
	tree qq = p->next;

        xx = yy = zz = 0;

        assert_pointer(NULL, qq);
        if (qq->kind == node_ref)
            qq = qq->first;
        assert_pointer(NULL, qq);
        switch (qq->kind) {
        case node_value:{
                xx = qq->x;
                break;
            }
        case node_const:{
                assert_pointer(NULL, qq->value);
                xx = qq->value->x;
                break;
            }
        default:{
                snark_node(qq->loc, qq);
            }
        }
        yy = xx;
        zz = xx;

        if (target_cpu != pic_16) {
            opcode |= (yy & 0x7FF);
        } else {
            if (p->opcode == opcode_f2nd) {
                opcode |= (yy & 0x00FF ) ;
            } else if (p->opcode == opcode_lfsr) {
	        sprintf(dest, ",H'%03X'", yy);
                opcode |= ((yy & 0x0F00) >> 8)  ;
	    }
        }
    }


    op_string = opcode_name[p->opcode];
  /*  if ((target_chip != t_16f84) && (target_chip != t_16c84))  */
        switch (p->opcode) {
        case opcode_page:{
                if (target_cpu == pic_12) {
                    if ((label_value & 0x200) == 0) {
                        op_string = "bcf 3,5 ;";
                        opcode = 0x4A3;
                    } else {
                        op_string = "bsf 3,5 ;";
                        opcode = 0x5A3;
                    }
                } else {
                    op_string = "CPAGE";
                }
                if ((phase == 1) && !(p->fizzle)) {
                    assembled_page_instructions++;
                }
                break;
            }
        case opcode_bank:{
                if (target_cpu == pic_16) {
/*                    p->fizzle = true; */ /* temporary! */
                   int xx = (file_address>>8)&0x0F;
		             if ((file_address >= 0xF80) || (file_address < 0x080)) {
                      p->fizzle = true;
		             }
		             op_string = "movlb";
		             sprintf(file, "H'%02X'", xx);
                   opcode = 0x0100 + xx;
                } else if (target_cpu == pic_12) {
                    if ((file_address & 0x20) == 0) {
                        op_string = "bcf 4,5 ;";
                        opcode = 0x4A4;
                    } else {
                        op_string = "bsf 4,5 ;";
                        opcode = 0x5A4;
                    }
                } else {
                    op_string = "RBANK";
                }
                if ((phase == 1) && !(p->fizzle)) {
                    assembled_bank_instructions++;
                }
                break;
            }
        case opcode_return:{
                if (target_cpu == sx_12)
                    op_string = "RET";
                break;
            }
        case opcode_retfie:{
                op_string = "RETFIE";
                break;
            }
        default:{
                break;
            }
        }

    sprintf(temp, "%c%-8s%s%s%s%s", ((p->fizzle) ? ';' : ' '), op_string, label, value, file, dest);

    if (result_hex_code) {
        sprintf(line, " %-40s ; %04X: %04X ; %s", temp, location, opcode, comment);
    } else {
        sprintf(line, " %s", temp);
    }

    if ((!p->fizzle)) {
        if (phase == 2) {
#ifdef __DEBUG__
            log((m, "p=%d a=%d", phase, location));
#endif
            if (location <= last_location) {
                image[location] = opcode;
                if (!location_used[location]) {
                    location_used[location] = true;
                    assembled_code_size++;
                } else {
                    snark(NULL);
                }
            }
        }
        location++;
    }
    if ((target_cpu == pic_16) && (p->fizzle)) {
        /* do not emit */
    } else {
        emit_assembley(p, line, phase);
    }

}

void assemble_statement(tree p, int phase)
{
    stack_guard;
    if (p == NULL)
        return;

    if (verbose_assembler) {
        log((m, "assemble_statement node loc=%03X nr=%04d kind=%d %s", location, p->nr, p->kind,
             node_name[p->kind]));
    }

    switch (p->kind) {

    case node_w:
    case node_precall:{
            /* ignore */
            break;
        }

    case node_test:{
            assemble_test(p, phase);
            break;
        }

    case node_decl:{
            assemble_statement(p->first, phase);
            break;
        }

    case node_type:{
            assemble_type(p, phase);
            break;
        }

    case node_var:{
            assemble_var(p, phase);
            break;
        }

    case node_const:{
            assemble_const(p, phase);
            break;
        }

    case node_label:{
            assemble_label(p, phase);
            break;
        }

    case node_org:{
            assemble_org(p, phase);
            break;
        }

    case node_procedure:{
            assemble_statement(p->first, phase);
            break;
        }

    case node_chain:{
            assemble_statement(p->first, phase);
            assemble_statement(p->next, phase);
            break;
        }

    case node_asm:{
            assemble_asm(p, phase);
            break;
        }

    default:{
            snark_node(p->loc, p);
            break;
        }
    }
}

#define eeprom_factor(i) \
   ((( i >= eeprom_start ) && ( target_cpu == pic_16 )) ? 1 : 2 )

int n_to_write(int i)
{
    int n = 0;
    if (i < eeprom_start) {
        while ((n < 8)
               && (i <= target_last_rom)
               && (location_used[i])
            ) {
            n++;
            i++;
            if ((n > 0) && (i % 8 == 0))
                return n;
        }
    } else {
        int nn;
        if (1 == eeprom_factor(i)) {
            nn = 16;
        } else {
            nn = 8;
        }
        while ((n < nn)
               && (i <= eeprom_start + eeprom_index)
            ) {
            n++;
            i++;
            if ((n > 0) && (i % nn == 0))
                return n;
        }
    }
    return n;
}

int low(int x)
{
    return x & 0xFF;
}

int high(int x)
{
    return (x >> 8) & 0xFF;
}

void checksum_add(int *x, int n)
{
    *x = (*x + n) & 0xFF;
}


/* write the hex result file */
void write_hex(char *file_name, int config)
{
    string s;
    FILE *f;
    int i, j, c;
    int ignore = 0;
    stack_guard;

    sprintf(s, "%s.hex", file_name);
    f = fopen(s, "w");
    if (f == NULL) {
        fatal(NULL, (m, "could not create %s", s));
    }

    fprintf(f, ":020000040000FA\n");

    for (i = 0; i != -1;) {
        int k, n = n_to_write(i);
        int addr;
        if (n > 0) {
            int c = 0;
            fprintf(f, ":");

            /* number of bytes */
			k = n *eeprom_factor( i );
            fprintf(f, "%02X", k);
            c += (2 * n) & 0xFF;

            /* address of first byte */
            addr = eeprom_factor(i) * i;
            fprintf(f, "%04X", (addr % 0x10000));
            c += addr & 0xFF;
            c += (addr >> 8) & 0xFF;

            /* record type */
            fprintf(f, "00");

            /* data bytes */
            for (j = 0; j < n; j++) {
                int x = ((i <= last_location) ? (image[i + j])
                         : (eeprom_data[i + j - eeprom_start])
                    );
                int x1 = x & 0xFF;
                int x2 = (x >> 8) & 0xFF;
                c = (c + x1 + x2) & 0xFF;
                fprintf(f, "%02X", x1);
                if (eeprom_factor(i) > 1) {
                    fprintf(f, "%02X", x2);
                }
            }

            /* checksum */
            fprintf(f, "%02X", (256 - c) & 0xFF);
            fprintf(f, "\n");

            i += n;
        } else {
            i++;
        }
        if (i > last_location) {
            if (i < eeprom_start) {
                i = eeprom_start;
                if (target_cpu == pic_16) {
                    fprintf(f, ":0200000400F00A\n");
                }
            } else if (i > eeprom_start + eeprom_index) {
                i = -1;
            }
        }
    }

    c = 0;
    /*if( target_cpu == sx_12 ){
       fprintf( f, ":021FFE00" );
       checksum_add( &c, 0x02 );
       checksum_add( &c, 0x1F );
       checksum_add( &c, 0xFE );
       checksum_add( &c, 0x00 ); */
    if (target_cpu == sx_12) {
        fprintf(f, ":04202000");
        checksum_add(&c, 0x04);
        checksum_add(&c, 0x20);
        checksum_add(&c, 0x20);
        checksum_add(&c, 0x00);
    } else if (target_cpu == pic_12) {
        fprintf(f, ":021FFE00");
        checksum_add(&c, 0x02);
        checksum_add(&c, 0x1F);
        checksum_add(&c, 0xFE);
        checksum_add(&c, 0x00);
    } else if (target_cpu == pic_14) {
        fprintf(f, ":02400E00");
        checksum_add(&c, 0x02);
        checksum_add(&c, 0x40);
        checksum_add(&c, 0x0E);
        checksum_add(&c, 0x00);
    } else {
        ignore = 1;
    }
    checksum_add(&c, low(config));
    checksum_add(&c, high(config));
    if (!ignore)
        fprintf(f, "%02X%02X", low(config), high(config)
            );
    if (target_cpu == sx_12) {
        fprintf(f, "6E00");
        checksum_add(&c, 0x6E);
        checksum_add(&c, 0x00);
    }
    if (!ignore)
        fprintf(f, "%02X\n", low(256 - c)
            );
    if (target_cpu == pic_16) {
       /* Wouter's Dwarf patch update (config words in PIC16) */ 
       int n_bytes = 14;
       fprintf(f, ":020000040030CA\n");
       fprintf(f, ":%02X000000", n_bytes);
       c = n_bytes;
       for( i = 0; i < n_bytes; i++ ){
          fprintf(f, "%02X", target_fuses_array[ i ] );
          checksum_add(&c, target_fuses_array[ i ] );
       }
       fprintf(f, "%02X\n", low( 256 - c ));
       /* fprintf(f, ":0E00000000220C0E000180000FC00FE00F4028\n"); */
    }
    fprintf(f, ":00000001FF\n");

    fclose(f);
}

/* write the statistics part of the assembler result file */
void assembler_statistics(void)
{
    int lines;
    float compilation_seconds;
    stack_guard if (assembler_file == NULL)
         return;

    compilation_seconds = current_time() - start_time;
    if (compilation_seconds == 0.0) {
        lines = 0;
    } else {
        lines = (int) (scanner_total_lines / compilation_seconds);
        if (lines < 0)
            lines = 0;
    }

    fprintf(assembler_file, "; compiler          : jal %s\n", VERSION);
    fprintf(assembler_file, "; date              : %s\n", date_and_time());
    fprintf(assembler_file, "; main source       : %s\n", main_source_file);
    fprintf(assembler_file, "; command line      : %s\n", command_line);
    fprintf(assembler_file, "; target  chip      : %s\n", chip_name[target_chip]);
    fprintf(assembler_file, ";         cpu       : %s\n", cpu_name[target_cpu]);
    fprintf(assembler_file, ";         clock     : %d\n", target_clock);
    fprintf(assembler_file, "; input   files     : %d\n", scanner_total_files);
    fprintf(assembler_file, ";         lines     : %d\n", scanner_total_lines);
    fprintf(assembler_file, ";         chars     : %d\n", scanner_total_chars);
    fprintf(assembler_file, "; compilation nodes : %d\n", node_nr);
    fprintf(assembler_file, ";             stack : %dKb\n", max_stack_size / 1024);
    fprintf(assembler_file, ";              heap : %dKb\n", total_memory / 1024);
    fprintf(assembler_file, ";           seconds : %2.3f (%d lines/second)\n", compilation_seconds,
            lines);
    fprintf(assembler_file, "; output       code : %4d\n", total_code_size);
    fprintf(assembler_file, ";              page : %4d (%3.1f%%)\n", assembled_page_instructions,
            100.0 * assembled_page_instructions / total_code_size);
    fprintf(assembler_file, ";              bank : %4d (%3.1f%%)\n", assembled_bank_instructions,
            100.0 * assembled_bank_instructions / total_code_size);
    fprintf(assembler_file, ";         page+bank : %4d (%3.1f%%)\n",
            assembled_bank_instructions + assembled_page_instructions,
            100.0 * (assembled_bank_instructions + assembled_page_instructions)
            / total_code_size);
    fprintf(assembler_file, ";              file : %4d\n", used_byte + 1 - target_first_ram);
    fprintf(assembler_file, ";              stack: %4d (%d,%d,%d)\n", total_stack_usage,
            main_stack_usage, indirect_stack_usage, int_stack_usage);
}

/* write the assembler header */
void assembler_header(void)
{
    //char *osc_string = NULL;
    char *target_string = "";
    stack_guard
        /* list setting */
        fprintf(assembler_file, "\n");
    fprintf(assembler_file, " errorlevel -306\n");
    if (target_cpu == sx_12)
        target_string = "16C57 ; ";
    if (target_chip == t_12c509a)
        target_string = "12C509 ; ";
    if (target_chip == t_12c508)
        target_string = "12C508 ; ";
/*   if( target_chip == pic_16f877 ) target_string = "PIC16F877 ; ";   */
    fprintf(assembler_file, " list p=PIC%s%s\n", target_string, target_name);

    /* marco's etc */
    fprintf(assembler_file, "\n");
    if ((target_chip == t_12c508) || (target_chip == t_12c509a)) {
        fprintf(assembler_file, "_CP_OFF    equ H'0008' ;code protect off\n");
        fprintf(assembler_file, "_CP_ON     equ H'0000' ;code protect on\n");
        fprintf(assembler_file, "_MCLR_INT  equ H'0000' ;internal MCLR\n");
        fprintf(assembler_file, "_MCLR_EXT  equ H'0010' ;external MCLR\n");
    } else if (target_chip == t_16f84) {
        fprintf(assembler_file, "_CP_OFF    equ H'3FF0' ;code protect on\n");
        fprintf(assembler_file, "_CP_ON     equ H'0000' ;code protect off\n");
        fprintf(assembler_file, "_PWRTE_ON  equ H'0000' ;Power on timer on\n");
        fprintf(assembler_file, "_PWRTE_OFF equ H'0008' ;Power on timer off\n");

    } else if (target_chip == t_16c84) {
        fprintf(assembler_file, "_CP_OFF    equ H'3FF0' ;code protect on\n");
        fprintf(assembler_file, "_CP_ON     equ H'3FE0' ;code protect off\n");
        fprintf(assembler_file, "_PWRTE_ON  equ H'0008' ;Power on timer on\n");
        fprintf(assembler_file, "_PWRTE_OFF equ H'0000' ;Power on timer off\n");

    } else if (target_chip == t_16f877) {
        fprintf(assembler_file, "; note: the f877 config is still fixed!\n");

    } else if (target_chip == t_16f876) {       /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        fprintf(assembler_file, "; note: the f876 config is still fixed!\n");

    } else if (target_chip == t_16f873) {       /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        fprintf(assembler_file, "; note: the f873 config is still fixed!\n");

    } else if (target_chip == t_16f88) {
        fprintf(assembler_file, "; note: the f88 config is still fixed!\n");

    } else if (target_cpu == pic_16) {
       /* Wouter's Dwarf patch update (config words in PIC16) */ 
       /* fprintf(assembler_file, "; note: the 18fxxx config is still fixed!\n"); */
    	  int i;
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, "_CONFIG1L        EQU    H'300000'\n");
        fprintf(assembler_file, "_CONFIG1H        EQU    H'300001'\n");
        fprintf(assembler_file, "_CONFIG2L        EQU    H'300002'\n");
        fprintf(assembler_file, "_CONFIG2H        EQU    H'300003'\n");
        fprintf(assembler_file, "_CONFIG3L        EQU    H'300004'\n");
        fprintf(assembler_file, "_CONFIG3H        EQU    H'300005'\n");
        fprintf(assembler_file, "_CONFIG4L        EQU    H'300006'\n");
        fprintf(assembler_file, "_CONFIG4H        EQU    H'300007'\n");
        fprintf(assembler_file, "_CONFIG5L        EQU    H'300008'\n");
        fprintf(assembler_file, "_CONFIG5H        EQU    H'300009'\n");
        fprintf(assembler_file, "_CONFIG6L        EQU    H'30000A'\n");
        fprintf(assembler_file, "_CONFIG6H        EQU    H'30000B'\n");
        fprintf(assembler_file, "_CONFIG7L        EQU    H'30000C'\n");
        fprintf(assembler_file, "_CONFIG7H        EQU    H'30000D'\n");
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, " __CONFIG    _CONFIG1L, H'%02X'\n", target_fuses_array[  0 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG1H, H'%02X'\n", target_fuses_array[  1 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG2L, H'%02X'\n", target_fuses_array[  2 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG2H, H'%02X'\n", target_fuses_array[  3 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG3L, H'%02X'\n", target_fuses_array[  4 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG3H, H'%02X'\n", target_fuses_array[  5 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG4L, H'%02X'\n", target_fuses_array[  6 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG4H, H'%02X'\n", target_fuses_array[  7 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG5L, H'%02X'\n", target_fuses_array[  8 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG5H, H'%02X'\n", target_fuses_array[  9 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG6L, H'%02X'\n", target_fuses_array[ 10 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG6H, H'%02X'\n", target_fuses_array[ 11 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG7L, H'%02X'\n", target_fuses_array[ 12 ] );
        fprintf(assembler_file, " __CONFIG    _CONFIG7H, H'%02X'\n", target_fuses_array[ 13 ] );
/*        fprintf(assembler_file, " __CONFIG    _CONFIG1L, H'00'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG1H, H'22'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG2L, H'0C'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG2H, H'0E'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG3L, H'00'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG3H, H'01'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG4L, H'80'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG4H, H'00'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG5L, H'0F'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG5H, H'C0'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG6L, H'0F'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG6H, H'E0'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG7L, H'0F'\n"); */
/*        fprintf(assembler_file, " __CONFIG    _CONFIG7H, H'40'\n"); */
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, "#define skpnz btfsc H'FD8', 2\n");
        fprintf(assembler_file, "#define skpz btfss H'FD8', 2\n");
        fprintf(assembler_file, "#define skpnc btfsc H'FD8', 0\n");
        fprintf(assembler_file, "#define skpc btfss H'FD8', 0\n");
        fprintf(assembler_file, "#define clrc bcf H'FD8', 0\n");
        fprintf(assembler_file, "#define clrz bcf H'FD8', 2\n");
        fprintf(assembler_file, "#define rrf rrcf\n");
        fprintf(assembler_file, "#define rlf rlcf\n");

    } else if (target_chip == t_16f628) {
        fprintf(assembler_file, "; note: the f628 config is still fixed!\n");

    } else if ((target_chip == t_12f629) || (target_chip == t_12f675)) {
        fprintf(assembler_file, "; note: the f629/675 config is still fixed!\n");

    } else if (target_cpu == sx_12) {
        fprintf(assembler_file, "CPAGE MACRO X ; SX code page instruction\n");
        fprintf(assembler_file, "   DATA 10h|((X)>>d'9')\n");
        fprintf(assembler_file, " ENDM\n");
        fprintf(assembler_file, " \n");
        fprintf(assembler_file, "RBANK MACRO X ; SX register bank instruction\n");
        fprintf(assembler_file, "   DATA 18h|((X)>>5)\n");
        fprintf(assembler_file, " ENDM\n");
        fprintf(assembler_file, " \n");
        fprintf(assembler_file, "RET MACRO ; SX return instruction\n");
        fprintf(assembler_file, "   DATA 0Ch\n");
        fprintf(assembler_file, " ENDM\n");
        fprintf(assembler_file, " \n");
        fprintf(assembler_file, "RETFIE MACRO ; SX return instruction\n");
        fprintf(assembler_file, "   DATA 0Eh\n");
        fprintf(assembler_file, " ENDM\n");
        fprintf(assembler_file, " \n");
        fprintf(assembler_file, "SX_CONFIG MACRO X; SX configuration\n");
        fprintf(assembler_file, "   ORG H'1010'\n");
        fprintf(assembler_file, "   DATA (X)\n");
        fprintf(assembler_file, "   DATA H'006E'\n");
        fprintf(assembler_file, "   ORG H'0'\n");
        fprintf(assembler_file, " ENDM\n");
        fprintf(assembler_file, " \n");
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, "_XT_OSC      equ H'0001' ;XT crystal oscillator\n");
        fprintf(assembler_file, "_RC_OSC      equ H'0003' ;RC oscillator\n");
        fprintf(assembler_file, "_LP_OSC      equ H'0000' ;LP crystal oscillator\n");
        fprintf(assembler_file, "_HS_OSC      equ H'0002' ;HS crystal oscillator\n");
        fprintf(assembler_file, "_WDT_OFF     equ H'0000' ;watch dog timer off\n");
        fprintf(assembler_file, "_WDT_ON      equ H'0004' ;watch dog timer on\n");
        fprintf(assembler_file, "_CP_OFF      equ H'0008' ;code protect on\n");
        fprintf(assembler_file, "_CP_ON       equ H'0000' ;code protect off\n");
        fprintf(assembler_file, "_RC_D_4M     equ H'0000' ;internal RC divider 4MHz\n");
        fprintf(assembler_file, "_RC_D_2M     equ H'0010' ;internal RC divider 2MHz\n");
        fprintf(assembler_file, "_RC_D_1M     equ H'0020' ;internal RC divider 1MHz\n");
        fprintf(assembler_file, "_RC_D_500k   equ H'0030' ;internal RC divider 500kHz\n");
        fprintf(assembler_file, "_RC_D_250k   equ H'0040' ;internal RC divider 250kHz\n");
        fprintf(assembler_file, "_RC_D_125k   equ H'0050' ;internal RC divider 125kHz\n");
        fprintf(assembler_file, "_RC_D_62k    equ H'0060' ;internal RC divider 62.5kHz\n");
        fprintf(assembler_file, "_RC_D_31k    equ H'0070' ;internal RC divider 31.25kHz\n");
        fprintf(assembler_file, "_IRC_ON      equ H'0000' ;internal RC mode\n");
        fprintf(assembler_file, "_IRC_OFF     equ H'0080' ;mode according to _...OSC\n");
        fprintf(assembler_file, "_STACKX_ON   equ H'0000' ;8-level stack\n");
        fprintf(assembler_file, "_STACKX_OFF  equ H'0100' ;2-level stack\n");
        fprintf(assembler_file, "_OPTX_ON     equ H'0000' ;6-bit option register\n");
        fprintf(assembler_file, "_OPTX_OFF    equ H'0200' ;8-bit option register\n");
        fprintf(assembler_file, "_SYNC_ON     equ H'0000' ;synchronous input enable\n");
        fprintf(assembler_file, "_SYNC_OFF    equ H'0400' ;synchronous input disable\n");
        fprintf(assembler_file, "_TURBO_ON    equ H'0000' ;turbo mode enable\n");
        fprintf(assembler_file, "_TURBO_OFF   equ H'0800' ;trubo mode disable\n");
        fprintf(assembler_file, "\n");

    } else {
        jal_assert(NULL, false);
    }
}

/* calculate the configuration value (and osc text for the SX) */
void calc_config(int *value, char **osc)
{
    int config = 0;
    char *osc_divider_string = NULL;
    int osc_divider_value = -1;
    stack_guard;

    if (target_cpu == sx_12) {

        if (config_osc != osc_int) {
            osc_divider_string = " _IRC_OFF ";
            osc_divider_value = 0x0080;
        } else if (target_clock == 4000000) {
            osc_divider_string = " _IRC_ON + _RC_D_4M ";
            osc_divider_value = 0x0003;
        } else if (target_clock == 2000000) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0013;
        } else if (target_clock == 1000000) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0023;
        } else if (target_clock == 500000) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0033;
        } else if (target_clock == 250000) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0043;
        } else if (target_clock == 125000) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0053;
        } else if (target_clock == 62500) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0063;
        } else if (target_clock == 31250) {
            osc_divider_string = " _IRC_ON + _RC_D_2M ";
            osc_divider_value = 0x0073;
        } else {
            fatal(NULL, (m, "the internal oscillator does not support %d Hz", target_clock));
        }
        config = osc_divider_value + +(config_watchdog ? 0x0004 : 0x0000)
            + (config_protection ? 0x0000 : 0x0008)
            + (config_osc == osc_rc ? 0x0003 : 0x0000)
            + (config_osc == osc_hs ? 0x0002 : 0x0000)
            + (config_osc == osc_xt ? 0x0001 : 0x0000)
            + (config_osc == osc_lp ? 0x0000 : 0x0000);

    } else if (target_cpu == pic_12) {

        config = (config_protection ? 0x0000 : 0x0008)
            + (config_int_mclr ? 0x0000 : 0x0010)
            + (config_watchdog ? 0x0004 : 0x0000)
            + (config_osc == osc_rc ? 0x0003 : 0x0000)
            + (config_osc == osc_int ? 0x0002 : 0x0000)
            + (config_osc == osc_xt ? 0x0001 : 0x0000)
            + (config_osc == osc_lp ? 0x0000 : 0x0000);

    } else if (target_cpu == pic_14) {

        if (target_chip == t_16f84) {
            config = (config_protection ? 0x0000 : 0x3FF0)
                + (config_powerup ? 0x0000 : 0x0008);
        } else {
            config = (config_protection ? 0x3FE0 : 0x3FF0)
                + (config_powerup ? 0x0008 : 0x0000);
        }
        config = config + (config_watchdog ? 0x0004 : 0x0000)
            + (config_osc == osc_rc ? 0x0003 : 0x0000)
            + (config_osc == osc_hs ? 0x0002 : 0x0000)
            + (config_osc == osc_xt ? 0x0001 : 0x0000)
            + (config_osc == osc_lp ? 0x0000 : 0x0000);

        if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)) {      /* Added 16f876 & 16f873 by Javi 2003-03-01 */
            config = 0x3F72;
        }
        if (target_chip == t_16f88) {
            config = 0x3F72;
        }
        if (target_chip == t_16f628) {
            config = 0x3F62;
        }
        if ((target_chip == t_12f675) || (target_chip == t_12f629)) {
            config = 0x3FE2;
        }
    }

    if (target_fuses != -1) {
        config = target_fuses;
    }

    *value = config;
    *osc = osc_divider_string;
}

/* write the assembler configuration part */
void assembler_config(char *osc_string)
{

    if (target_cpu == sx_12) {

        fprintf(assembler_file, " SX_CONFIG ( %s + %s + %s + %s + %s )\n",
                "_TURBO_ON + _SYNC_ON + _OPTX_ON + _STACKX_ON ", osc_string,
                (config_protection ? "_CP_ON" : "_CP_OFF"),
                (config_watchdog ? "_WDT_ON" : "_WDT_OFF"), (osc_name[config_osc]));

    } else if (target_fuses != -1) {

        fprintf(assembler_file, " __CONFIG H'%04X' \n", target_fuses);

    } else if (target_cpu == pic_12) {

        fprintf(assembler_file, "_WDT_OFF    equ H'0000' ;watch dog timer off\n");
        fprintf(assembler_file, "_WDT_ON     equ H'0004' ;watch dog timer on\n");
        fprintf(assembler_file, "_RC_EXT_OSC equ H'0003' ;external RC oscillator\n");
        fprintf(assembler_file, "_RC_INT_OSC equ H'0002' ;internal RC oscillator\n");
        fprintf(assembler_file, "_XT_OSC     equ H'0001' ;XT crystal oscillator\n");
        fprintf(assembler_file, "_LP_OSC     equ H'0000' ;LP crystal oscillator\n");
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, " __CONFIG %s + %s + %s + %s \n",
                (config_int_mclr ? "_MCLR_INT" : "_MCLR_EXT"),
                (config_protection ? "_CP_ON" : "_CP_OFF"),
                (config_watchdog ? "_WDT_ON" : "_WDT_OFF"), (osc_name_509[config_osc])
            );
        fprintf(assembler_file, "\n");

    } else if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)) {   /* Added 16f876 & 16f873 by Javi 2003-03-01 */

        fprintf(assembler_file, " __CONFIG H'3F72' \n");

    } else if (target_cpu == pic_16) {

        /* fprintf( assembler_file, "; __CONFIG H'3F72' \n" ); */

    } else if (target_chip == t_16f628) {

        fprintf(assembler_file, " __CONFIG H'3F62' \n");

    } else if ((target_chip == t_12f675) || (target_chip == t_12f629)) {

        fprintf(assembler_file, " __CONFIG H'3FE2' \n");

    } else if (target_cpu == pic_14) {

        fprintf(assembler_file, "_WDT_OFF   equ H'0000' ;watch dog timer off\n");
        fprintf(assembler_file, "_WDT_ON    equ H'0004' ;watch dog timer on\n");
        fprintf(assembler_file, "_RC_OSC    equ H'0003' ;RC oscillator\n");
        fprintf(assembler_file, "_HS_OSC    equ H'0002' ;HS crystal oscillator\n");
        fprintf(assembler_file, "_XT_OSC    equ H'0001' ;XT crystal oscillator\n");
        fprintf(assembler_file, "_LP_OSC    equ H'0000' ;LP crystal oscillator\n");
        fprintf(assembler_file, "\n");
        fprintf(assembler_file, " __CONFIG %s + %s + %s + %s \n",
                (config_protection ? "_CP_ON" : "_CP_OFF"),
                (config_powerup ? "_PWRTE_ON" : "_PWRTE_OFF"),
                (config_watchdog ? "_WDT_ON" : "_WDT_OFF"), (osc_name[config_osc])
            );
        fprintf(assembler_file, "\n");

    } else {
        jal_assert(NULL, false);
    }
}


void assemble(tree * p)
{
    int i, config;
    string s;
    char *osc_string = NULL;
    stack_guard
        /* allocate */
        last_location = target_last_rom;
    image = allocate(sizeof(int) * (last_location + 1));
    location_used = allocate(sizeof(boolean) * (last_location + 1));
    assertions = allocate(sizeof(tree) * (last_location + 1));
    assembled_code_size = 0;
    for (i = 0; i <= last_location; i++) {
        image[i] = 0;
        location_used[i] = false;
        assertions[i] = NULL;
    }

    /* phase one */
    location = 0;
    assemble_statement(*p, 1);

    /* create header part of the assembler output file  */
    if (result_asm) {
        sprintf(s, "%s.asm", main_source_file);
        assembler_file = fopen(s, "w");
        if (assembler_file == NULL) {
            fatal(NULL, (m, "could not create %s", s));
        }
    } else {
        assembler_file = NULL;
    }
    assembler_statistics();
    assembler_header();
    calc_config(&config, &osc_string);
    assembler_config(osc_string);

    /* phase two */
    location = 0;
    assemble_statement(*p, 2);

    /* trailer part of the assembler output file */
    if (result_asm) {
        fprintf(assembler_file, "\n END\n");
        fprintf(assembler_file, "\n");
        show_variables(assembler_file);
        fprintf(assembler_file, "\n");
        fclose(assembler_file);
    }

    /* did the code fit? */
    if (location > last_location + 1) {
        fatal(NULL, (m, "code too large %d (%d available)", location, last_location));
    }

    /* write the hex output file */
    if (result_hex) {
        write_hex(main_source_file, config);
    }
}
