/* simulator
   Copyright (C) 1998, 1999, 2000, 2001, 2002, 2003
   Wouter van Ooijen

This file is part of jal.

jal is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

jal is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with jal; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA.  */

#include "stdhdr.h"
#include "global.h"
#include "target.h"
#include "errorlh.h"
#include "treerep.h"
#include "assemble.h"
#include "stacksg.h"
#include "scanner.h"
#include "codegen.h"
#include "regalloc.h"
#include "reswords.h"
#include "compdriver.h"
#include "parser.h"
#include "treetools.h"
#include "simulator.h"
#include "cstringf.h"

#include <limits.h>

/* uses image,assertions, last_location */
boolean sim_run = true;
int sim_pc = 0;
int sim_w;
int sim_mem[4096];
int sim_stack[9];
int sim_stack16[32];
int sim_fss;     /* Fast Register Stack for PIC16s  */
int sim_sp = 0;

int sim_instruction;
int sim_literal;
int sim_data_address;
int sim_call_label;
int sim_goto_label;
int sim_bra_address;
int sim_lfsr_address;
int sim_lfsr_dest;
int sim_bit;
int sim_bit_mask;
int sim_destination;
int sim_tris_address;

int sim_status;
int sim_reg_option;
int sim_intcon;
int sim_tmr0_prescaler;
int sim_tmr0_prescaler_start;
int sim_tmr0;

int sim_instruction_cycles;

#define intcon_gie  0x80
#define intcon_eeie 0x40
#define intcon_t0ie 0x20
#define intcon_inte 0x10
#define intcon_rbie 0x08
#define intcon_t0if 0x04
#define intcon_intf 0x02
#define intcon_rbif 0x01

#define option_psa  0x08 /* Changed by Ziya Erdemir 18.02.2003*/        /*#define option_psa  0x04 */

#define status_c    0x01
#define status_dc   0x02
#define status_z    0x04
#define status_ov   0x08  /* PIC16 cores */
#define status_n    0x10  /* PIC16 cores */
#define status_pd   0x08
#define status_to   0x10
#define status_rp0  0x20
#define status_rp1  0x40
#define status_irp  0x80
#define status_pa0  0x20
#define status_pa1  0x40
#define status_pa2  0x80

boolean sim_c_flag, sim_z_flag, sim_dc_flag, sim_ov_flag, sim_n_flag;
boolean sim_check_int = false;

#define simulation_error( loc, x ){ string m; \
   sprintf x; \
   ffatal( loc, m ); \
}

#define extra_cycles_for_jump ( ( target_cpu == sx_12 ) ? 2 : 1 )
void instruction_cycle(int n)
{
    while (n > 0) {
        n--;
        simulated_instructions++;
        sim_tmr0_prescaler--;
        if (sim_tmr0_prescaler == 0) {
            sim_tmr0_prescaler = sim_tmr0_prescaler_start;
            sim_tmr0++;                 /* it was sim_tmr0--;           */
            if (sim_tmr0 == 256)  {     /* it was sim_tmr0 == 0x00      */
                sim_tmr0 = 0x00;        /* it was sim_tmr0       = 0xFF; */
                sim_intcon |= intcon_t0if;
                sim_check_int = true;
            }
        }
    }
}

void sim_update_option(void)
{
    if ((sim_reg_option & option_psa) != 0) {
        sim_tmr0_prescaler_start = 1;
    } else
        switch (sim_reg_option & 0x07) {
        case 0:
            sim_tmr0_prescaler_start = 2;
            break;
        case 1:
            sim_tmr0_prescaler_start = 4;
            break;
        case 2:
            sim_tmr0_prescaler_start = 8;
            break;
        case 3:
            sim_tmr0_prescaler_start = 16;
            break;
        case 4:
            sim_tmr0_prescaler_start = 32;
            break;
        case 5:
            sim_tmr0_prescaler_start = 64;
            break;
        case 6:
            sim_tmr0_prescaler_start = 128;
            break;
        case 7:
            sim_tmr0_prescaler_start = 256;
            break;
        }
}

int sim_get(int a)
{
    int x;
    int addr, data;
    switch (a) {
    case 0xFEF:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
       }
       data = sim_mem[addr];
	    return data;
    }
    case 0xFEE:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          data = sim_mem[addr];
          addr = addr + 1;
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFED:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          data = sim_mem[addr];
          addr = addr - 1;
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFEC:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          addr = addr + 1;
          data = sim_mem[addr];
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFEB:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9] + sim_w;
          data = sim_mem[addr];
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFE7:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          data = sim_mem[addr];
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFE6:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          data = sim_mem[addr];
          addr = addr + 1;
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFE5:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          data = sim_mem[addr];
          addr = addr - 1;
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFE4:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          addr = addr + 1;
          data = sim_mem[addr];
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFE3:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1] + sim_w;
          data = sim_mem[addr];
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFDF:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          data = sim_mem[addr];
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFDE:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          data = sim_mem[addr];
          addr = addr + 1;
          sim_mem[0xFDA] = (addr>>8) & 0xFF;
          sim_mem[0xFD9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFDD:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          data = sim_mem[addr];
          addr = addr - 1;
          sim_mem[0xFDA] = (addr>>8) & 0xFF;
          sim_mem[0xFD9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFDC:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          addr = addr + 1;
          data = sim_mem[addr];
          sim_mem[0xFDA] = (addr>>8) & 0xFF;
          sim_mem[0xFD9] = (addr) & 0xFF;
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0xFDB:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9] + sim_w;
          data = sim_mem[addr];
       } else {
          data = sim_mem[addr];
       }
	    return data;
    }
    case 0x81:{                /* option register */
            if (target_cpu == pic_16) {
                break;
            }
            return sim_reg_option;
        }
    case 0x02:
    case 0xFF9:{               /* Program Counter */
            if (target_cpu == pic_16) {
                if (a == 0xFF9) {
                    return (sim_pc * 2) & 0xFF;
                } else {
                    break;
                }
            }
            return sim_pc & 0xFF;
        }
    case 0x03:
    case 0xFD8:{               /* flags */
            if (target_cpu == pic_16)  {
               if (a == 0xFD8) {
                  x = sim_mem[0xFD8] & (~7);
               } else {
                 break;
               }
            } else {
               x = sim_mem[3] & (~7);
            }   
            if (sim_c_flag)
                x |= 1;
            if (sim_dc_flag)
                x |= 2;
            if (sim_z_flag)
                x |= 4;
            if ((target_cpu == pic_16)) {
               if (sim_ov_flag)
                   x |= 8;
               if (sim_n_flag)
                   x |= 0x10;
            }
#ifdef __DEBUG__
            log((m, "f=%02X c=%d dc=%d z=%d ov=%d n=%d", x, sim_c_flag, sim_dc_flag, sim_z_flag,  sim_ov_flag, sim_n_flag));
#endif
            return x;
        }
    case 0x0b:{                /* intcon register */
            if (target_cpu == pic_16) {
                break;
            }
            return sim_intcon;
        }
    case 0xFE8:{               /* F18 WREG */
            if (target_cpu == pic_16) {
                return sim_w;
            } else {
                break;
            }
        }
        /*  included by Ziya Erdemir    18.02.2003    */
    case 0x00:{              /* INDF */
            if (target_cpu == pic_16) {
                break;
            } else if (target_cpu == pic_14) {
                return sim_mem[sim_mem[4] | (sim_status & 0x80) << 1];  /* IRP */
            } else {
                return sim_mem[sim_mem[4] & 0x7F];  /* pic_12 */
            }
        }
    default:{
            break;
        }
    }
#ifdef __DEBUG__
    log((m, "get a=%03X d=%02X", a, sim_mem[a] & 0xFF));
#endif
    return sim_mem[a] & 0xFF;
}

void sim_put(int a, int d)
{
    int addr;
    int high;
#ifdef __DEBUG__
    log((m, "put a=%03X d=%02X", a, d));
#endif
    sim_mem[a] = d & 0xFF;
    switch (a) {
    case 0xFEF:{
       if (target_cpu == pic_16) {
           addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0xFEE:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          sim_mem[addr] = d & 0xFF;
          addr = addr + 1;
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFED:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          sim_mem[addr] = d & 0xFF;
          addr = addr - 1;
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFEC:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9];
          addr = addr + 1;
          sim_mem[addr] = d & 0xFF;
          sim_mem[0xFEA] = (addr>>8) & 0xFF;
          sim_mem[0xFE9] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFEB:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFEA]<<8)+sim_mem[0xFE9] + sim_w;
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0xFE7:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0xFE6:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          sim_mem[addr] = d & 0xFF;
          addr = addr + 1;
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFE5:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          sim_mem[addr] = d & 0xFF;
          addr = addr - 1;
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFE4:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1];
          addr = addr + 1;
          sim_mem[addr] = d & 0xFF;
          sim_mem[0xFE2] = (addr>>8) & 0xFF;
          sim_mem[0xFE1] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFE3:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFE2]<<8)+sim_mem[0xFE1] + sim_w;
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0xFDF:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0xFDE:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          sim_mem[addr] = d & 0xFF;
          addr = addr + 1;
          sim_mem[0xFDA] = ((addr>>8) & 0xFF);
          sim_mem[0xFD9] = ((addr) & 0xFF);
       }
       break;
    }
    case 0xFDD:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          sim_mem[addr] = d & 0xFF;
          addr = addr - 1;
          sim_mem[0xFDA] = (addr>>8) & 0xFF;
          sim_mem[0xFD9] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFDC:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9];
          addr = addr + 1;
          sim_mem[addr] = d & 0xFF;
          sim_mem[0xFDA] = (addr>>8) & 0xFF;
          sim_mem[0xFD9] = (addr) & 0xFF;
       }
       break;
    }
    case 0xFDB:{
       if (target_cpu == pic_16) {
          addr = (sim_mem[0xFDA]<<8)+sim_mem[0xFD9] + sim_w;
          sim_mem[addr] = d & 0xFF;
       }
       break;
    }
    case 0x81:{                /* option register */
          if (target_cpu == pic_16) {
             break;
          }
          sim_reg_option = d & 0xFF;
          sim_update_option();
          sim_check_int = true;
          break;
        }
    case 0x02:
    case 0xFF9:{               /* Program Counter */
            if ((target_cpu == pic_16) && (a != 0xFF9)) {
                break;
            }
            if (target_cpu == sx_12) {
                high = (sim_mem[3] & 0x60) << 4;
            } else if (target_cpu == pic_12) {
                high = (sim_mem[3] & 0x20) << 4;
            } else if (target_cpu == pic_14) {
                high = (sim_mem[10] & 0x1F) << 8;
            } else {
                high = ((d & 0xFE) + (sim_mem[0xFFA] << 8)) >> 1;
                d = 0;
            }
            sim_pc = (d & 0xFF) + high;
            if (verbose_simulator) {
                printf("; new page := %03X", high);
            }
            break;
        }
    case 0x03:
    case 0xFD8:{               /* flags */
            if (target_cpu == pic_16) {
               if (a == 0xFD8) {
                  sim_mem[0xFD8] = d & 0xFF;
               } else {
                  break;
               }
            } else {
               sim_mem[0x03] = d & 0xFF;
            }
            sim_c_flag = (d & 1) > 0;
            sim_dc_flag = (d & 2) > 0;
            sim_z_flag = (d & 4) > 0;
            if ((target_cpu == pic_16)) {
               sim_ov_flag = (d & 8) > 0;
               sim_n_flag = (d & 0x10) > 0;
            }
            sim_status = d & 0xFF;
#ifdef __DEBUG__
            log((m, "f=%02X c=%d dc=%d z=%d ov=%d n=%d", x, sim_c_flag, sim_dc_flag, sim_z_flag,  sim_ov_flag, sim_n_flag));
#endif
            break;
        }
    case 0x0b:{                /* intcon register */
            if (target_cpu == pic_16) {
                break;
            }
            sim_intcon = d & 0xFF;
            sim_check_int = true;
            break;
        }
    case 0x00:{                /* INDF */
            if (target_cpu == pic_16) {
                break;
            } else if (target_cpu == pic_14) {
               sim_mem[sim_mem[4]] = d & 0xFF;
               break;
            } else if (target_cpu == pic_12) {
               sim_mem[sim_mem[4] & 0x7F] = d & 0xFF;
               break;
            }
        }
    case 0xFE8:{               /* F18 WREG */
           if (target_cpu == pic_16) {
              sim_w = d & 0xFF;
           }
        }
    default:{
            break;
        }
    }
}

int decode_address ( int a ) {
int addr ;
        if (target_cpu == pic_16) {
            addr = a ;
        } else if (target_cpu == pic_14) {

            addr = ((0x80 * ((sim_mem[0x03] & 0x60) >> 5) )+ ( a & 0x7F) );
            switch (addr) {
               case 0x0:
               case 0x80:
               case 0x100:
               case 0x180:{ addr = 0 ; 
                            break; }
               case 0x1:
               case 0x101:{ addr = 1 ; 
                            break; }
               case 0x81:
               case 0x181:{ addr = 81 ; 
                            break; }
               case 0x2:
               case 0x82:
               case 0x102:
               case 0x182:{ addr = 2 ; 
                            break; }
               case 0x3:
               case 0x83:
               case 0x103:
               case 0x183:{ addr = 3 ; 
                            break; }
               case 0x4:
               case 0x84:
               case 0x104:
               case 0x184:{ addr = 4 ; 
                            break; }
               case 0xA:
               case 0x8A:
               case 0x10A:
               case 0x18A:{ addr = 0xA; 
                            break; }
               case 0xB:
               case 0x8B:
               case 0x10B:
               case 0x18B:{ addr = 0xB; 
                            break; }
             }
        } else if (target_cpu == pic_12) {

            addr = ((0x20 * ((sim_mem[0x04] & 0x60) >> 5) )+ ( a & 0x1F) );
            switch (addr) {
               case 0x0:
               case 0x20:{ addr = 0 ; 
                            break; }
               case 0x1:
               case 0x21:{ addr = 1 ; 
                            break; }
               case 0x2:
               case 0x22:{ addr = 2 ; 
                            break; }
               case 0x3:
               case 0x23:{ addr = 3 ; 
                            break; }
               case 0x4:
               case 0x24:{ addr = 4 ; 
                            break; }
               case 0x5:
               case 0x25:{ addr = 5 ; 
                            break; }
               case 0x6:
               case 0x26:{ addr = 6 ; 
                            break; }
             }
        } else {

            addr = a ;
        }  

        return (addr) ;   
}


void sim_store_result(int d)
{
    d = d & 0xFF;
    if (sim_destination) {
        sim_put(decode_address(sim_data_address), d);
        if (verbose_simulator) {
            printf(", f; f := %02X", d);
        }
    } else {
        sim_w = d;
        if (verbose_simulator) {
            printf(", w; w := %02X", d);
        }
    }
}

void sim_stack16_get(void)
{
    sim_sp = sim_mem[0xFFC] & 0x1F;
    sim_pc = (sim_mem[0xFFF]<<16)+(sim_mem[0xFFE]<<8)+(sim_mem[0xFFD]);
    sim_sp--;
    if (((sim_mem[0xFFC] & 0x40) != 0) | (sim_sp <0)) {
       if (verbose_simulator) {
           printf("; Warning!! Stack Underflow, stack value: %06X",sim_pc);
       }
       sim_pc = 0;
       sim_mem[0xFFC] = sim_mem[0xFFC] | 0x40;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0);
    } else {
       if (verbose_simulator) {
           printf("; Returned Stack value (%02X): %06X",sim_sp,(sim_pc));
       }
       sim_mem[0xFFC] = sim_mem[0xFFC] & 0xBF;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0) + (sim_sp & 0x1F);
       sim_mem[0xFFF] = (sim_stack16[sim_sp+1]>>16) & 0xFF;
       sim_mem[0xFFE] = (sim_stack16[sim_sp+1]>>8) & 0xFF;
       sim_mem[0xFFD] = (sim_stack16[sim_sp+1]) & 0xFF;
    }
}
void sim_stack16_put(void)
{
    sim_sp = sim_mem[0xFFC] & 0x1F;
    sim_sp++;
#ifdef __DEBUG__
    log((m, "sim_sp=%d", sim_sp));
#endif
    if (((sim_mem[0xFFC] & 0x80) != 0) | (sim_sp >31)) {
       if (verbose_simulator) {
           printf("; Warning!! Stack Overflow, stack value: %06X",sim_pc);
       }
       sim_sp = 31;
       sim_pc = 0;
       sim_mem[0xFFC] = sim_mem[0xFFC] | 0x80;
    } else {
       if (verbose_simulator) {
           printf("; New Stack value (%02X): %06X",sim_sp,(sim_pc));
       }
       sim_mem[0xFFC] = sim_mem[0xFFC] & 0x7F;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0) + (sim_sp & 0x1F);
       sim_stack16[sim_sp] = (sim_mem[0xFFF]<<16)+(sim_mem[0xFFE]<<8)+(sim_mem[0xFFD]);
    }
    sim_mem[0xFFF] = ((sim_pc)>>16) & 0xFF;
    sim_mem[0xFFE] = ((sim_pc)>>8) & 0xFF;
    sim_mem[0xFFD] = (sim_pc) & 0xFF;
}


void sim_show_literal(void)
{
    if (verbose_simulator) {
        printf("%02X", sim_literal);
    }
}
void sim_show_call_label(void)
{
    if (verbose_simulator) {
        printf("%03X", sim_call_label);
    }
}
void sim_show_goto_label(void)
{
    if (verbose_simulator) {
        printf("%03X", sim_goto_label);
    }
}
void sim_show_data_address(void)
{
    if (verbose_simulator) {
        printf("%02X", decode_address(sim_data_address));
    }
}
void sim_show_bit(void)
{
    if (verbose_simulator) {
        printf(",%01X", sim_bit);
    }
}
void sim_show_tris_address(void)
{
    if (verbose_simulator) {
        printf("%01X", sim_tris_address);
    }
}
void sim_show_new_w(void)
{
    sim_w = sim_w & 0xFF;
    if (verbose_simulator) {
        printf("; w := %02X", sim_w);
    }
}

void sim_nop(void)
{
}
void sim_clrw(void)
{
    sim_w = 0;
}
void sim_return(void)
{
    if (target_cpu == pic_16) {
       sim_stack16_get();
    } else {
       sim_sp--;
    if ( verbose_simulator )  {
       log((m, "sim_sp=%d", sim_sp));
    }      
     if (sim_sp < 0)
           snark(NULL);
       sim_pc = sim_stack[sim_sp];
    }
    instruction_cycle(extra_cycles_for_jump);
}
void sim_retfie(void)
{
    if (target_cpu == pic_16) {
       sim_stack16_get();
    } else {
       sim_sp--;
       if (sim_sp < 0)
           snark(NULL);
       sim_pc = sim_stack[sim_sp];
    }
    sim_intcon += intcon_gie;
    instruction_cycle(extra_cycles_for_jump);
}
void sim_option(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! OPTION simulated as NOP.\n");
    }
}
void sim_sleep(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! SLEEP simulated as NOP.\n");
    }
}
void sim_clrwdt(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! CLRWDT simulated as NOP.\n");
    }
}
void sim_tris(void)
{
    sim_show_tris_address();
    sim_put(sim_tris_address, sim_w);
}
void sim_movwf(void)
{
    sim_show_data_address();
    sim_put(decode_address(sim_data_address), sim_w);
}
void sim_clrf(void)
{
    sim_z_flag = (0 == 0);
    sim_show_data_address();
    sim_put(decode_address(sim_data_address), 0);
}
void sim_subwf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address)) - sim_w;
    sim_show_data_address();
    /* Javi 2003-04-01. Can have carry/borrow and be zero in a byte */
    sim_c_flag = (y >= 0);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = (y == 0);
    sim_store_result(y);
}
void sim_subwfb(void)
{
    int y;
    if ( sim_c_flag ) {
       y = sim_get(decode_address(sim_data_address)) - sim_w;
    } else {
       y = sim_get(decode_address(sim_data_address)) - sim_w - 1;
    }
    sim_show_data_address();
    sim_c_flag = (y >= 0);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = (y == 0);
    sim_store_result(y);
}
void sim_subfwb(void)
{
    int y;
    if ( sim_c_flag ) {
       y = sim_w - sim_get(decode_address(sim_data_address));
    } else {
       y = sim_w - sim_get(decode_address(sim_data_address)) - 1;
    }
    sim_show_data_address();
    sim_c_flag = (y >= 0);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = (y == 0);
    sim_store_result(y);
}
void sim_decf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y - 1);
    sim_c_flag = (y >= 0);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_decfsz(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y - 1) & 0xFF;
    sim_store_result(y);
    if (y == 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_dcfsnz(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y - 1) & 0xFF;
    sim_store_result(y);
    if (y != 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_rrf(void)
{
    int y;
    boolean old_carry;

    y = sim_get(decode_address(sim_data_address));
    old_carry = sim_c_flag;
    sim_show_data_address();
    sim_c_flag = ((y & 1) == 1);
    y = y >> 1;
    if (old_carry) {
        y = y | 0x80;
    }
    sim_n_flag = (y > 127);
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_rrncf(void)
{
    int y;
    boolean old_carry;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    old_carry = ((y & 1) == 1);
    y = y >> 1;
    if (old_carry) {
        y = y | 0x80;
    }
    sim_n_flag = (y > 127);
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_rlf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = y << 1;
    if (sim_c_flag) {
        y = y | 1;
    }
    sim_c_flag = y > 0xFF;
    sim_n_flag = (y > 127);
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_rlncf(void)
{
    int y;
    boolean old_carry;
    
    y = sim_get(decode_address(sim_data_address));
    old_carry = ((y & 0x80) == 0x80);
    sim_show_data_address();
    y = y << 1;
    if (old_carry) {
        y = y | 1;
    }
    sim_c_flag = y > 0xFF;
    sim_n_flag = (y > 127);
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_iorwf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = y | sim_w ;
    sim_z_flag = y == 0;
    sim_n_flag = (y > 127);
    sim_store_result(y);
}
void sim_andwf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = y & sim_w;
    sim_z_flag = y == 0;
    sim_n_flag = (y > 127);
    sim_store_result(y);
}
void sim_xorwf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = y ^ sim_w;
    sim_z_flag = y == 0;
    sim_n_flag = (y > 127);
    sim_store_result(y);
}
void sim_addwf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = y + sim_w;
    sim_c_flag = (y > 0xFF);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_addwfc(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    if (sim_c_flag) {
       y = y + sim_w + 1;
    } else {
       y = y + sim_w;
    }
    sim_c_flag = (y > 0xFF);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_movf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_z_flag = y == 0;
    sim_n_flag = (y < 0);
    sim_store_result(y);
}
void sim_comf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y ^ 0xFF);
    sim_z_flag = y == 0;
    sim_n_flag = (y < 0);
    sim_store_result(y);
}
void sim_incf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y + 1);
    sim_c_flag = (y > 0xFF);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_store_result(y);
}
void sim_incfsz(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y + 1) & 0xFF;
    sim_store_result(y);
    if (y == 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_infsnz(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = (y + 1) & 0xFF;
    sim_store_result(y);
    if (y != 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_swapf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    y = ((y >> 4) & 0x0F) | ((y << 4) & 0xF0);
    sim_store_result(y);
}
void sim_setf(void)
{
    int y;
    y = 0xFF;
    sim_show_data_address();
    sim_put(decode_address(sim_data_address) , y ) ;
}
void sim_tstfsz(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    if (y == 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_bcf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_show_bit();
    y = y & ~sim_bit_mask;
#ifdef __DEBUG__
    if (verbose_simulator) {
        log((m, "y=%X mask=%X", y, sim_bit_mask));
    }
#endif
    sim_put(decode_address(sim_data_address), y);
}
void sim_btg(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_show_bit();
    y = y ^ sim_bit_mask;
#ifdef __DEBUG__
    if (verbose_simulator) {
        log((m, "y=%X mask=%X", y, sim_bit_mask));
    }
#endif
    sim_put(decode_address(sim_data_address), y);
}
void sim_bsf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_show_bit();
    y = y | sim_bit_mask;
#ifdef __DEBUG__
    if (verbose_simulator) {
        log((m, "y=%X mask=%X", y, sim_bit_mask));
    }
#endif
    sim_put(decode_address(sim_data_address), y);
}
void sim_btfsc(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_show_bit();
#ifdef __DEBUG__
    if (verbose_simulator) {
        log((m, "y=%X mask=%X cond=%d", y, sim_bit_mask, y & sim_bit_mask));
    }
#endif
    if ((y & sim_bit_mask) == 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_btfss(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address));
    sim_show_data_address();
    sim_show_bit();
#ifdef __DEBUG__
    if (verbose_simulator) {
        log((m, "y=%X mask=%X cond=%d", y, sim_bit_mask, y & sim_bit_mask));
    }
#endif
    if ((y & sim_bit_mask) != 0) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_call(void)
{
    if (target_cpu == pic_16) {
       sim_stack16_put();
    } else {
       sim_stack[sim_sp] = sim_pc;
       sim_sp++;
    if ( verbose_simulator )  {
       log((m, "sim_sp=%d", sim_sp));
    }
       if (sim_sp > 8)
           snark(NULL);
    }
    sim_show_call_label();
    sim_pc = sim_call_label;
    instruction_cycle(extra_cycles_for_jump);
}
void sim_bc(void)
{
    if (sim_c_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bn(void)
{
    if (sim_n_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bnc(void)
{
    if (! sim_c_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bnn(void)
{
    if (! sim_n_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bnov(void)
{
    if (! sim_ov_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bnz(void)
{
    if (! sim_z_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bov(void)
{
    if (sim_ov_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_bra(void)
{
    if ((sim_bra_address & 0x7FF) < 0x400) {
     sim_call_label = sim_pc + (sim_bra_address & 0x7FF);
    } else {
     sim_call_label = sim_pc + ( 0xFFFFF800 | (sim_bra_address & 0x7FF));
    }
    sim_show_call_label();
    sim_pc = sim_call_label;
    instruction_cycle(extra_cycles_for_jump);
}
void sim_rcall(void)
{
    sim_stack16_put();
    if ((sim_bra_address & 0x7FF) < 0x400) {
     sim_call_label = sim_pc + (sim_bra_address & 0x7FF);
    } else {
     sim_call_label = sim_pc + ( 0xFFFFF800 | (sim_bra_address & 0x7FF));
    }
    sim_show_call_label();
    sim_pc = sim_call_label;
    instruction_cycle(extra_cycles_for_jump);
}
void sim_bz(void)
{
    if (sim_z_flag) {
       if (sim_literal < 0x80) {
        sim_call_label = sim_pc + (sim_literal);
       } else {
        sim_call_label = sim_pc + ( 0xFFFFFF00 | sim_literal);
       }
       sim_show_call_label();
       sim_pc = sim_call_label;
       instruction_cycle(extra_cycles_for_jump);
    }
}
void sim_goto(void)
{
    sim_show_goto_label();
    sim_pc = sim_goto_label;
    instruction_cycle(extra_cycles_for_jump);
}
void sim_movlw(void)
{
    sim_show_literal();
    sim_w = sim_literal  & 0xFF ;
    sim_show_new_w();
}
void sim_retlw(void)
{
    sim_show_literal();
    sim_w = sim_literal  & 0xFF ;
    sim_show_new_w();
    sim_return();
}
void sim_iorlw(void)
{
    int y;
    y = sim_w | sim_literal;
    sim_show_literal();
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_w = y ;
    sim_show_new_w();
}
void sim_andlw(void)
{
    int y;
    y = sim_w & sim_literal;
    sim_show_literal();
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_w = y ;
    sim_show_new_w();
}
void sim_xorlw(void)
{
    int y;
    y = sim_w ^ sim_literal;
    sim_show_literal();
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_w = y ;
    sim_show_new_w();
}
void sim_sublw(void)
{
    int y;
    y = sim_literal - sim_w;
    sim_show_literal();
    sim_c_flag = (y >= 0);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_w = y  ;
    sim_show_new_w();
}
void sim_addlw(void)
{
    int y;
    y = sim_literal + sim_w;
    sim_show_literal();
    sim_c_flag = (y > 0xFF);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_w = y ;
    sim_show_new_w();
}
void sim_bank(void)
{
    if (target_cpu == pic_16) {
       sim_put(0xFE0, (sim_instruction & 0x0F));
       if (verbose_simulator) {
           printf("; bank := %02X", sim_instruction & 0x0F);
       }
    } else if (target_cpu == pic_12) {
    } else if (target_cpu == pic_14) {
    }
    
}
void sim_page(void)
{
    if (target_cpu == pic_16) {
       if (verbose_simulator) {
           printf("; page ignored.\n");
       }
    } else {
       sim_put(3, (sim_get(3) & 0x1F)
            + ((sim_instruction & 0x07) << 5));
       if (verbose_simulator) {
           printf("; page := %02X", sim_instruction & 0x07);
       }
    }
}

void sim_mullw(void)
{
    int y;
    y = sim_w * sim_literal;
    sim_show_literal();
    sim_mem[0xFF4] = ( y >> 8 )  & 0xFF ;
    sim_mem[0xFF3] = y & 0xFF;
}
void sim_tblrd(void)
{
    int y;
    int w;
    y = (sim_mem[0xFF8]<<16) + (sim_mem[0xFF7]<<8) + sim_mem[0xFF6];
    w = y % 2 ;
    if (w == 0) {
       sim_mem[0xFF5] = image[y] & 0xFF;
    } else {
       sim_mem[0xFF5] = (image[y]>>8) & 0xFF;
    }
    instruction_cycle(1);
}
void sim_tblrd_postinc(void)
{
    int y;
    int w;
    y = (sim_mem[0xFF8]<<16) + (sim_mem[0xFF7]<<8) + sim_mem[0xFF6];
    w = y % 2 ;
    if (w == 0) {
       sim_mem[0xFF5] = image[y] & 0xFF;
    } else {
       sim_mem[0xFF5] = (image[y]>>8) & 0xFF;
    }
    y = y + 1;
    sim_mem[0xFF8] = ( y >>16)& 0x0000FF;
    sim_mem[0xFF7] = ( y >>8)& 0x0000FF;
    sim_mem[0xFF6] = ( y )& 0x0000FF;
    instruction_cycle(1);
}
void sim_tblrd_postdec(void)
{
    int y;
    int w;
    y = (sim_mem[0xFF8]<<16) + (sim_mem[0xFF7]<<8) + sim_mem[0xFF6];
    w = y % 2 ;
    if (w == 0) {
       sim_mem[0xFF5] = image[y] & 0xFF;
    } else {
       sim_mem[0xFF5] = (image[y]>>8) & 0xFF;
    }
    y = y - 1;
    sim_mem[0xFF8] = ( y >>16)& 0x0000FF;
    sim_mem[0xFF7] = ( y >>8)& 0x0000FF;
    sim_mem[0xFF6] = ( y )& 0x0000FF;
    instruction_cycle(1);
}
void sim_tblrd_preinc(void)
{
    int y;
    int w;
    y = (sim_mem[0xFF8]<<16) + (sim_mem[0xFF7]<<8) + sim_mem[0xFF6];
    y = y + 1;
    sim_mem[0xFF8] = ( y >>16)& 0x0000FF;
    sim_mem[0xFF7] = ( y >>8)& 0x0000FF;
    sim_mem[0xFF6] = ( y )& 0x0000FF;
    w = y % 2 ;
    if (w == 0) {
       sim_mem[0xFF5] = image[y] & 0xFF;
    } else {
       sim_mem[0xFF5] = (image[y]>>8) & 0xFF;
    }
    instruction_cycle(1);
}
void sim_tblwt(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! TBLWT simulated as NOP.\n");
    }
    instruction_cycle(1);
}
void sim_tblwt_postinc(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! TBLWT*+ simulated as NOP.\n");
    }
    instruction_cycle(1);
}
void sim_tblwt_postdec(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! TBLWT*- simulated as NOP.\n");
    }
    instruction_cycle(1);
}
void sim_tblwt_preinc(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! TBLWT+* simulated as NOP.\n");
    }
    instruction_cycle(1);
}
void sim_mulwf(void)
{
    int y;
    y = sim_w * sim_get(sim_data_address);
    sim_mem[0xFF4] = ( y >> 8)  & 0xFF ;
    sim_mem[0xFF3] = y & 0xFF;
}
void sim_cpfseq(void)
{
    if (sim_get(decode_address(sim_data_address)) == sim_w) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_cpfsgt(void)
{
    if (sim_get(decode_address(sim_data_address)) > sim_w) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_cpfslt(void)
{
    if (sim_get(decode_address(sim_data_address)) < sim_w) {
        sim_pc = sim_pc + 1;
        instruction_cycle(1);
    }
}
void sim_negf(void)
{
    int y;
    y = sim_get(decode_address(sim_data_address)) ;
    y = 1 + ( y ^ 0xFF );
    sim_c_flag = (y > 0xFF);
    sim_ov_flag = (y > 127);
    sim_n_flag = (y < 0);
    y = y & 0xFF;
    sim_z_flag = y == 0;
    sim_put(decode_address(sim_data_address), y ) ;
}
void sim_daw(void)
{
    int y;
    y = sim_w;
    if ( (y & 0x0F)>9 ){
       y = (y & 0xF0) | ((y & 0x0F) + 6 );
    }
    if ( (y & 0xF0)>0x90 ){
       y = (y & 0x0F) + ((y & 0xF0) + 0x60 );
    }
    sim_c_flag = (y > 0xFF);
    y = y & 0xFF;
    sim_w = y ;
    sim_show_new_w();
}
void sim_push(void)
{
    sim_sp = sim_mem[0xFFC] & 0x1F;
    sim_sp++;
#ifdef __DEBUG__
    log((m, "sim_sp=%d", sim_sp));
#endif
    if (((sim_mem[0xFFC] & 0x80) != 0) | (sim_sp >31)) {
       if (verbose_simulator) {
           printf("; Warning!! Stack Overflow, stack value: %08X",sim_pc);
       }
       sim_sp = 31;
       sim_mem[0xFFC] = sim_mem[0xFFC] | 0x80;
    } else {
       sim_mem[0xFFC] = sim_mem[0xFFC] & 0xBF;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0) + (sim_sp & 0x1F);
       sim_stack16[sim_sp] = (sim_mem[0xFFF]<<16)+(sim_mem[0xFFE]<<8)+(sim_mem[0xFFD]);
    }
    sim_mem[0xFFF] = ((sim_pc+1)>>16) & 0xFF;
    sim_mem[0xFFE] = ((sim_pc+1)>>8) & 0xFF;
    sim_mem[0xFFD] = (sim_pc+1) & 0xFF;
}
void sim_pop(void)
{
    int pctemp;
    sim_sp = sim_mem[0xFFC] & 0x1F;
    pctemp = sim_stack16[sim_sp];
    sim_sp--;
    if (((sim_mem[0xFFC] & 0x40) != 0) | (sim_sp <0)) {
       if (verbose_simulator) {
           printf("; Warning!! Stack Underflow, stack value: %08X",pctemp);
       }
       pctemp = 0;
       sim_mem[0xFFC] = sim_mem[0xFFC] | 0x40;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0);
    } else {
       sim_mem[0xFFC] = sim_mem[0xFFC] & 0xBF;
       sim_mem[0xFFC] = (sim_mem[0xFFC] & 0xC0) & (sim_sp & 0x1F);
    }
    sim_mem[0xFFF] = (pctemp>>16) & 0xFF;
    sim_mem[0xFFE] = (pctemp>>8) & 0xFF;
    sim_mem[0xFFD] = (pctemp) & 0xFF;
}
void sim_reset(void)
{
    if (verbose_simulator) {
        printf("; WARNING!! RESET simulated as NOP.\n");
    }
}
void sim_lfsr(void)
{
   if (sim_lfsr_dest == 0){
      sim_mem[0xFEA] = (sim_lfsr_address >> 8) & 0xFF;
      sim_mem[0xFE9] = sim_lfsr_address & 0xFF;
   } else if (sim_lfsr_dest == 1){
      sim_mem[0xFE2] = (sim_lfsr_address >> 8) & 0xFF;
      sim_mem[0xFE1] = sim_lfsr_address & 0xFF;
   } else if (sim_lfsr_dest == 2){
      sim_mem[0xFDA] = (sim_lfsr_address >> 8) & 0xFF;
      sim_mem[0xFD9] = sim_lfsr_address & 0xFF;
   }
   instruction_cycle(1);
}
void sim_movlb(void)
{
    int y;
    y = sim_literal;
    y = y & 0x0F;
    if (verbose_simulator) {
        printf("; bank := %02X", y);
    }
   sim_mem[0xFE0] = y;
}

#define try_op( v, s, f ) \
case v: \
   if( verbose_simulator ){ printf( "%-7s ", s ); } \
   f(); \
   return;

void sim12_00(void)
{
#ifdef __DEBUG__
    trace log((m, "i=%03x", sim_instruction));
#endif
    switch (sim_instruction) {
        try_op(0x040, "clrw", sim_clrw);
        try_op(0x000, "nop", sim_nop);
        try_op(0x004, "clrwdt", sim_clrwdt);
        try_op(0x002, "option", sim_option);
        try_op(0x003, "sleep", sim_sleep);
        if (target_cpu == sx_12)
            try_op(0x00C, "return", sim_return);
        if (target_cpu == sx_12)
            try_op(0x00E, "retfie", sim_retfie);
    default:
        break;
    }
    if ((target_cpu == sx_12) && ((sim_instruction & 0x0F8) == 0x018)) {
        if (verbose_simulator) {
            printf("%-7s ", "bank");
        }
        sim_bank();
        return;
    }
    if ((target_cpu == sx_12) && ((sim_instruction & 0x0F8) == 0x010)) {
        if (verbose_simulator) {
            printf("%-7s ", "page");
        }
        sim_page();
        return;
    }
    if ((sim_instruction & 0x0C0) == 0x0C0) {
        if (verbose_simulator) {
            printf("%-7s ", "decf");
        }
        sim_decf();
        return;
    }
    if ((sim_instruction & 0x0C0) == 0x080) {
        if (verbose_simulator) {
            printf("%-7s ", "subwf");
        }
        sim_subwf();
        return;
    }
    if ((sim_instruction & 0x0E0) == 0x020) {
        if (verbose_simulator) {
            printf("%-7s ", "movwf");
        }
        sim_movwf();
        return;
    }
    if ((sim_instruction & 0x0E0) == 0x060) {
        if (verbose_simulator) {
            printf("%-7s ", "clrf");
        }
        sim_clrf();
        return;
    }
    if ((sim_instruction & 0x0F8) == 0x000) {
        if (verbose_simulator) {
            printf("%-7s ", "tris");
        }
        sim_tris();
        return;
    }
    snark(NULL);
}

void sim12_01(void)
{
#ifdef __DEBUG__
    trace
#endif
        switch (sim_instruction & 0x0C0) {
        try_op(0x000, "iorwf", sim_iorwf);
        try_op(0x040, "andwf", sim_andwf);
        try_op(0x080, "xorwf", sim_xorwf);
        try_op(0x0C0, "addwf", sim_addwf);
    default:
        break;
    }
    snark(NULL);
}

void sim12_02(void)
{
#ifdef __DEBUG__
    trace
#endif
        switch (sim_instruction & 0x0C0) {
        try_op(0x000, "movf", sim_movf);
        try_op(0x040, "comf", sim_comf);
        try_op(0x080, "incf", sim_incf);
        try_op(0x0C0, "decfsz", sim_decfsz);
    default:
        break;
    }
    snark(NULL);
}

void sim12_03(void)
{
#ifdef __DEBUG__
    trace
#endif
        switch (sim_instruction & 0x0C0) {
        try_op(0x000, "rrf", sim_rrf);
        try_op(0x040, "rlf", sim_rlf);
        try_op(0x080, "swapf", sim_swapf);
        try_op(0x0C0, "incfsz", sim_incfsz);
    default:
        break;
    }
    snark(NULL);
}

void sim12_rest(void)
{
#ifdef __DEBUG__
    trace
#endif
        switch (sim_instruction & 0xF00) {
        try_op(0x400, "bcf", sim_bcf);
        try_op(0x500, "bsf", sim_bsf);
        try_op(0x600, "btfsc", sim_btfsc);
        try_op(0x700, "btfss", sim_btfss);
        try_op(0x800, "retlw", sim_retlw);
        try_op(0x900, "call", sim_call);
        try_op(0xA00, "goto", sim_goto);
        try_op(0xB00, "goto", sim_goto);
        try_op(0xC00, "movlw", sim_movlw);
        try_op(0xD00, "iorlw", sim_iorlw);
        try_op(0xE00, "andlw", sim_andlw);
        try_op(0xF00, "xorlw", sim_xorlw);
    default:
        break;
    }
    snark(NULL);
}

void sim14_00(void)
{
    if ((sim_instruction & 0x3F80) == 0) {
        switch (sim_instruction) {
            try_op(0x00, "nop", sim_nop);
            try_op(0x20, "nop", sim_nop);
            try_op(0x40, "nop", sim_nop);
            try_op(0x60, "nop", sim_nop);
            try_op(0x08, "return", sim_return);
            try_op(0x09, "retfie", sim_retfie);
            try_op(0x62, "option", sim_option);
            try_op(0x63, "sleep", sim_sleep);
            try_op(0x64, "clrwdt", sim_clrwdt);
            try_op(0x65, "tris", sim_tris);
            try_op(0x66, "tris", sim_tris);
        default:
            snark(NULL);
        }
    }
    if ((sim_instruction & 0x3F80) == 0x0100) {
        if (verbose_simulator) {
            printf("%-7s ", "clrw");
        }
        sim_clrw();
        return;
    }
    switch ((sim_instruction >> 8) & 0x0F) {
        try_op(0x00, "movwf", sim_movwf);
        try_op(0x01, "clrf", sim_clrf);
        try_op(0x02, "subwf", sim_subwf);
        try_op(0x03, "decf", sim_decf);
        try_op(0x04, "iorwf", sim_iorwf);
        try_op(0x05, "andwf", sim_andwf);
        try_op(0x06, "xorwf", sim_xorwf);
        try_op(0x07, "addwf", sim_addwf);
        try_op(0x08, "movf", sim_movf);
        try_op(0x09, "comf", sim_comf);
        try_op(0x0A, "incf", sim_incf);
        try_op(0x0B, "decfsz", sim_decfsz);
        try_op(0x0C, "rrf", sim_rrf);
        try_op(0x0D, "rlf", sim_rlf);
        try_op(0x0E, "swapf", sim_swapf);
        try_op(0x0F, "incfsz", sim_incfsz);
    default:
        snark(NULL);
    }
}

void sim14_01(void)
{
    switch ((sim_instruction >> 10) & 0x03) {
        try_op(0x00, "bcf", sim_bcf);
        try_op(0x01, "bsf", sim_bsf);
        try_op(0x02, "btfsc", sim_btfsc);
        try_op(0x03, "btfss", sim_btfss);
    default:
        snark(NULL);
    }
}

void sim14_02(void)
{
    switch (sim_instruction & 0x800) {
        try_op(0x0000, "call", sim_call);
        try_op(0x0800, "goto", sim_goto);
    default:
        snark(NULL);
    }
}

void sim14_03(void)
{
    switch ((sim_instruction >> 8) & 0x0F) {
        try_op(0x00, "movlw", sim_movlw);
        try_op(0x01, "movlw", sim_movlw);
        try_op(0x02, "movlw", sim_movlw);
        try_op(0x03, "movlw", sim_movlw);
        try_op(0x04, "retlw", sim_retlw);
        try_op(0x05, "retlw", sim_retlw);
        try_op(0x06, "retlw", sim_retlw);
        try_op(0x07, "retlw", sim_retlw);
        try_op(0x08, "iorlw", sim_iorlw);
        try_op(0x09, "andlw", sim_andlw);
        try_op(0x0A, "xorlw", sim_xorlw);
        try_op(0x0C, "sublw", sim_sublw);
        try_op(0x0D, "sublw", sim_sublw);
        try_op(0x0E, "addlw", sim_addlw);
        try_op(0x0F, "addlw", sim_addlw);
    default:
        snark(NULL);
    }
}

#define sim_field( start, bits )\
   ( ( sim_instruction >> start ) & ( ( 1 << bits ) - 1 ) )

void show_location(void)
{
    if (!verbose_simulator)
        return;
    printf("%05d ", simulated_instructions);
    printf("%03X: %04X ", sim_pc, image[sim_pc]);
}

void sim_interrupt(void)
{
    sim_stack[sim_sp] = sim_pc;
    sim_sp++;
    if (sim_sp > 8)
        snark(NULL);
    sim_pc = 4;
    instruction_cycle(2);
    sim_intcon &= (!intcon_gie);
}

void simulate_one_instruction_12(void)
{
    /* also not for 12c509a ! */
#ifdef __DEBUG__
    if (sim_check_int) {
        sim_check_int = false;
        if (sim_intcon & intcon_gie != 0) {
            if ((sim_intcon & intcon_t0ie != 0)
                && (sim_intcon & intcon_t0if != 0)
                ) {
                sim_interrupt();
                return;
            }
        }
    }
#endif

    sim_instruction = image[sim_pc];
    sim_literal = sim_field(0, 8);
    sim_data_address = sim_field(0, 5);
    if (sim_data_address > 0x0F) {
        if (target_cpu == sx_12) {
            sim_data_address += (sim_mem[4] & 0xE0);
        } else {
            sim_data_address += (sim_mem[4] & 0x20);
        }
    }
    sim_call_label = sim_field(0, 8)
        + ((sim_status & (status_pa0 | status_pa1)) << 4);
    sim_goto_label = sim_field(0, 9)
        + ((sim_status & (status_pa0 | status_pa1)) << 4);
    sim_bit = sim_field(5, 3);
    sim_bit_mask = 1 << sim_bit;
    sim_destination = sim_field(5, 1);
    sim_tris_address = sim_field(0, 3);
    show_location();
    sim_pc++;

#ifdef __DEBUG__
    log((m, "%d", sim_instruction >> 8));
#endif
    switch (sim_instruction >> 8) {
    case 0x00:
        sim12_00();
        break;
    case 0x01:
        sim12_01();
        break;
    case 0x02:
        sim12_02();
        break;
    case 0x03:
        sim12_03();
        break;
    default:
        sim12_rest();
        break;
    }
    sim_w = sim_w & 0xFF;
    instruction_cycle(1);
}

void simulate_one_instruction_14(void)
{
    if (sim_check_int) {
        sim_check_int = false;
        if ((sim_intcon & intcon_gie) != 0) {
            if (((sim_intcon & intcon_t0ie) != 0)
                && ((sim_intcon & intcon_t0if) != 0)
                ) {
                sim_interrupt();
                return;
            }
        }
    }

    sim_instruction = image[sim_pc];
    sim_literal = sim_field(0, 8);
    sim_data_address = sim_field(0, 7)
        + ((sim_status & (status_rp0 | status_rp1)) << 1);
    sim_call_label = sim_field(0, 11);
    sim_goto_label = sim_field(0, 11);
    if ((target_chip == t_16f877) || (target_chip == t_16f876) || (target_chip == t_16f873)) {  /* Added 16f876 & 16f873 by Javi 2003-03-01 */
        sim_goto_label += (sim_mem[10] & 0x18) << 8;
        sim_call_label += (sim_mem[10] & 0x18) << 8; /* Javi 2003-06-25  */
    }
    sim_bit = sim_field(7, 3);
    sim_bit_mask = 1 << sim_bit;
    sim_destination = sim_field(7, 1);
    sim_tris_address = sim_field(0, 3);
    show_location();
    sim_pc++;

#ifdef __DEBUG__
    log((m, "%d", sim_instruction >> 12));
#endif
    switch (sim_instruction >> 12) {
    case 0x00:
        sim14_00();
        break;
    case 0x01:
        sim14_01();
        break;
    case 0x02:
        sim14_02();
        break;
    case 0x03:
        sim14_03();
        break;
    default:
        snark(NULL);
    }
    sim_w = sim_w & 0xFF;
    instruction_cycle(1);
}


void simulate_one_instruction_16(void)
{
    if (sim_check_int) {
        sim_check_int = false;
        if ((sim_intcon & intcon_gie) != 0) {
            if (((sim_intcon & intcon_t0ie) != 0)
                && ((sim_intcon & intcon_t0if) != 0)
                ) {
                sim_interrupt();
                return;
            }
        }
    }

    sim_instruction = image[sim_pc];
    sim_literal = sim_field(0, 8);
    sim_data_address = sim_field(0, 8);
    if (sim_field(8, 1)) {
        sim_data_address += (sim_mem[0xFE0] & 0x0F) << 8;
    } else {
        if (sim_data_address > 0x7F) {
            sim_data_address += 0xF00;
        }
    }
    sim_call_label = sim_field(0, 8) + ((image[sim_pc + 1] & 0xFFF) << 8);
    sim_goto_label = sim_call_label;
    sim_bit = sim_field(9, 3);
    sim_bit_mask = 1 << sim_bit;
    sim_destination = sim_field(9, 1);
    sim_bra_address = image[sim_pc] & 0x07FF;
    sim_lfsr_address = ((image[sim_pc] & 0x0F) << 8) + (image[sim_pc + 1] & 0x00FF);
    sim_lfsr_dest = (image[sim_pc] >> 4 ) & 0x0003;
    show_location();
    sim_pc++;

#define IFM( m, n, i ) \
      if(( sim_instruction & m ) == n ){ \
         if( verbose_simulator ){ printf( "%-7s ", #i ); } \
         i(); }

#ifdef __DEBUG__
    log((m, "%d", sim_instruction >> 12));
#endif
    sim_instruction_cycles = 1;
    IFM(0xFC00, 0x2400, sim_addwf)
        else
    IFM(0xFC00, 0x2000, sim_addwfc)
        else
    IFM(0xFC00, 0x1400, sim_andwf)
        else
    IFM(0xFE00, 0x6A00, sim_clrf)
        else
    IFM(0xFC00, 0x1C00, sim_comf)
        else
    IFM(0xFF00, 0x6200, sim_cpfseq)
        else
    IFM(0xFF00, 0x6400, sim_cpfsgt)
        else
    IFM(0xFF00, 0x6000, sim_cpfslt)
        else
    IFM(0xFC00, 0x0400, sim_decf)
        else
    IFM(0xFC00, 0x2C00, sim_decfsz)
        else
    IFM(0xFC00, 0x4C00, sim_dcfsnz)
        else
    IFM(0xFC00, 0x2800, sim_incf)
        else
    IFM(0xFC00, 0x3C00, sim_incfsz)
        else
    IFM(0xFC00, 0x4800, sim_infsnz)
        else
    IFM(0xFC00, 0x1000, sim_iorwf)
        else
    IFM(0xFC00, 0x5000, sim_movf)
        else
    IFM(0xFE00, 0x6E00, sim_movwf)
        else
    IFM(0xFE00, 0x0200, sim_mulwf)
        else
    IFM(0xFE00, 0x6C00, sim_negf)
        else
    IFM(0xFC00, 0x3400, sim_rlf)
        else
    IFM(0xFC00, 0x4400, sim_rlncf)
        else
    IFM(0xFC00, 0x3000, sim_rrf)
        else
    IFM(0xFC00, 0x4000, sim_rrncf)
        else
    IFM(0xFE00, 0x6800, sim_setf)
        else
    IFM(0xFC00, 0x5400, sim_subfwb)
        else
    IFM(0xFC00, 0x5C00, sim_subwf)
        else
    IFM(0xFC00, 0x5800, sim_subwfb)
        else
    IFM(0xFC00, 0x3800, sim_swapf)
        else
    IFM(0xFE00, 0x6600, sim_tstfsz)
        else
    IFM(0xFC00, 0x1800, sim_xorwf)
        else
    IFM(0xF000, 0x9000, sim_bcf)
        else
    IFM(0xF000, 0x8000, sim_bsf)
        else
    IFM(0xF000, 0xB000, sim_btfsc)
        else
    IFM(0xF000, 0xA000, sim_btfss)
        else
    IFM(0xF000, 0x7000, sim_btg)
        else
    IFM(0xFF00, 0xE200, sim_bc)
        else
    IFM(0xFF00, 0xE600, sim_bn)
        else
    IFM(0xFF00, 0xE300, sim_bnc)
        else
    IFM(0xFF00, 0xE700, sim_bnn)
        else
    IFM(0xFF00, 0xE500, sim_bnov)
        else
    IFM(0xFF00, 0xE100, sim_bnz)
        else
    IFM(0xFF00, 0xE400, sim_bov)
        else
    IFM(0xF800, 0xD000, sim_bra)
        else
    IFM(0xFF00, 0xE000, sim_bz)
        else
    IFM(0xFE00, 0xEC00, sim_call)
        else
    IFM(0xFFFF, 0x0004, sim_clrwdt)
        else
    IFM(0xFFFF, 0x0007, sim_daw)
        else
    IFM(0xFF00, 0xEF00, sim_goto)
        else
    IFM(0xFFFF, 0x0000, sim_nop)
        else
    IFM(0xF000, 0xF000, sim_nop)
        else
    IFM(0xFFFF, 0x0006, sim_pop)
        else
    IFM(0xFFFF, 0x0005, sim_push)
        else
    IFM(0xF800, 0xD800, sim_rcall)
        else
    IFM(0xFFFF, 0x00FF, sim_reset)
        else
    IFM(0xFFFF, 0x0010, sim_retfie)
        else
    IFM(0xFF00, 0x0C00, sim_retlw)
        else
    IFM(0xFFFE, 0x0012, sim_return)
        else
    IFM(0xFFFF, 0x0003, sim_sleep)
        else
    IFM(0xFF00, 0x0F00, sim_addlw)
        else
    IFM(0xFF00, 0x0B00, sim_andlw)
        else
    IFM(0xFF00, 0x0900, sim_iorlw)
        else
    IFM(0xFFC0, 0xEE00, sim_lfsr)
        else
    IFM(0xFFF0, 0x0100, sim_movlb)
        else
    IFM(0xFF00, 0x0E00, sim_movlw)
        else
    IFM(0xFF00, 0x0D00, sim_mullw)
        else
    IFM(0xFF00, 0x0800, sim_sublw)
        else
    IFM(0xFF00, 0x0A00, sim_xorlw)
        else
    IFM(0xFFFF, 0x0008, sim_tblrd)
        else
    IFM(0xFFFF, 0x0009, sim_tblrd_postinc)
        else
    IFM(0xFFFF, 0x000A, sim_tblrd_postdec)
        else
    IFM(0xFFFF, 0x000B, sim_tblrd_preinc)
        else
    IFM(0xFFFF, 0x000C, sim_tblwt)
        else
    IFM(0xFFFF, 0x000D, sim_tblwt_postinc)
        else
    IFM(0xFFFF, 0x000E, sim_tblwt_postdec)
        else
    IFM(0xFFFF, 0x000F, sim_tblwt_preinc)
        else {
        log((m, "pc=%04X instr=%04X", sim_pc - 1, sim_instruction));
        snark(NULL);
    }
    sim_w = sim_w & 0xFF;
    instruction_cycle(sim_instruction_cycles);
}

int mask(int n)
{
    if (n == 0)
        return 0;
    return 1 | (mask(n - 1) << 1);
}

int get_mem(tree p)
{
    int a, b, x;
    boolean debug = false;
    a = p->address->first->x;
    b = p->address->next->x;
    /* x = sim_mem[a]; */
     x = sim_get(a);
    /* if(debug) log(( m , "a=%d x=%d s=%d",a,x,p->address->value->y)); */
    x = x >> b;
    if (debug)
        log((m, "x=%d bits=%d, m=%d", x, p->type->bits, mask(p->type->bits)));
    x = x & mask(p->type->bits);
    if (debug)
        log((m, "x=%d", x));
    return x;
}

boolean evaluate_assert(tree p, char *s)
{
    int a, b, x;
    assert_pointer(p->loc, p->first);
    switch (p->first->kind) {

    case node_var:{
            assert_pointer(p->loc, p->first->address);
            assert_pointer(p->loc, p->first->address->first);
            a = p->first->address->first->x;
            b = p->first->address->next->x;
            if (verbose_simulator) {
                printf("assert location %02X:%d == %02X\n", a, b, p->x);
            }
            x = get_mem(p->first);
            if (x != p->x) {
                sprintf(s, "value of location %02X:%d is %02X instead of %02X", a, b, x, p->x);
                return false;
            }
            break;
        }

    case node_const:{
            assert_pointer(p->loc, p->first->value);
            a = p->first->value->x;
            if (verbose_simulator) {
                printf("assert constant %02X == %02X\n", a, p->x);
            }
            if (a != p->x) {
                sprintf(s, "value of const is %02X instead of %02X", a, p->x);
                return false;
            }
            break;
        }

    default:{
            snark_node(p->loc, p->first);
            break;
        }
    }
    return true;
}

void simulate_assertions(void)
{
    string s;
    tree p = assertions[sim_pc];
#ifdef __DEBUG__
    log((m, "loc=%03X ", sim_pc));
#endif
    while (p != NULL) {
        assert_kind(p->loc, p, node_test);
#ifdef __DEBUG__
        log((m, "loc=%03X n=%d k=%d op=%d", sim_pc, p->nr, p->kind, p->op));
#endif
        switch (p->op) {
        case op_test_assert:{
                if (!evaluate_assert(p, s)) {
                    simulation_error(p->loc, (m, "assert (%d) failed, %s", p->nr, s));
                }
                break;
            }
        case op_test_end:{
                sim_run = false;
                break;
            }
        default:
            snark_node(p->loc, p);
        }
        p = p->next;
    }
}

int simulation_max_steps = 1000 * 1000 * 10;
void simulate(tree * dummy)
{
    float start_time;
    float seconds;
    long kips;
    start_time = current_time();
    for (;;) {
        if ((sim_pc < 0) || (sim_pc > target_last_rom)) {
            simulation_error(NULL, (m, "pc (%03X) beyond code (%03X) ", sim_pc, location - 1)
                );
        }
        simulate_assertions();
        if (!sim_run) {
            seconds = current_time() - start_time;
            if (seconds < 1.0/(float)SHRT_MAX) {
                kips = simulated_instructions;
                printf("simlation    instructions:%d seconds:%2.8f\n",
                   simulated_instructions, seconds);
             } else {
                kips = (long)(simulated_instructions / seconds);
                printf("simlation    instructions:%d seconds:%2.3f (%ld instructions/second)\n",
                   simulated_instructions, seconds, kips);
             }
           printf("test OK\n");
            return;
        }
        if (simulated_instructions > simulation_max_steps) {
            simulation_error(NULL, (m, "max steps (%d) reached", simulation_max_steps)
                );
        }
        if ((target_cpu == sx_12) || (target_cpu == pic_12)) {
            simulate_one_instruction_12();
        } else if (target_cpu == pic_14) {
            simulate_one_instruction_14();
        } else if (target_cpu == pic_16) {
            simulate_one_instruction_16();
        } else {
            snark(NULL);
        }
        if (verbose_simulator) {
            printf("\n");
        }
    }
}
