#ifndef LIB_STR_H
#define LIB_STR_H

#include <stdarg.h>

#include "alloc.h"
#include "stddef.h"

/* mem_cpy - copy memory area
 *
 * - copies n bytes from memory area src to memory area dst
 * - returns a pointer to dst
 */
extern void *mem_cpy(void *dst, const void *src, size_t n);

/* mem_set - fill memory with a constant byte
 *
 * - fills the first n bytes of the memory area pointed to by *dst with
 *   the constant byte c
 */
extern void *mem_set(void *dst, unsigned char c, size_t n);

/* mem_cmp - compare memory areas
 *
 * - compares the first n bytes of the memory areas s1 and s2
 * - returns an integer less than, equal to, or greater than zero
 */
extern size_t mem_cmp(const void *s1, const void *s2, size_t n);

/* mem_move - copy memory area
 *
 * - copies n bytes from memory area src to memory area dst.
 * - the memoryareas may overlap
 * - returns a pointer to dst
 */
extern void *mem_move(void *dst, const void *src, size_t n);

/* str_ncmp - compare max n bytes of two strings
 *
 * - similar to memcmp()
 */
extern size_t str_ncmp(const unsigned char *s1, const unsigned char *s2, size_t n);

/* str_cmp - compare two strings
 *
 * - similar to strncmp(), but without limit!
 */
extern size_t str_cmp(const unsigned char *s1, const unsigned char *s2);

/* str_len - calculate the length of a string
 *
 * - returns the length of the string, not including the '\0'
 */
extern size_t str_len(const unsigned char *s);

/* str_cpy - copy a string
 *
 * - copy *src to string *dst
 * - returns str_len(dst)
 */
extern size_t str_cpy(unsigned char *dst, const unsigned char *src);

/* str_ncpy - copy a string
 *
 * - copy *src to string *dst, for max. n bytes
 * - allways null terminates
 * - n should allways be one less than the available size
 * - returns the copied length n
 * -> str_ncpy(foo, "bar", 3); -> writes "ba\0" to foo
 */
extern size_t str_ncpy(unsigned char *dst, const unsigned char *src, size_t n);

/* str_cat - concatenate two strings
 *
 * - appends *src to *dst
 * - returns *dst
 * - *dst _must_ have enought space !
 */
extern unsigned char *str_cat(unsigned char *dst, const unsigned char *src);

/* str_dup - duplicate a string
 *
 * - returns the new allocated string
 */
extern unsigned char *str_dup(const unsigned char *s);

/* str_concat - new space for n strings
 *
 * - allocates + fills space for str1, str2, str3, ... n
 * - returns destination of new string
 */
extern unsigned char *str_concat(const char *first, ...);

/* str_chr - locate character in string
 *
 * - finds first occurrence of the character c
 * - returns pointer to the character in the string (or NULL)
 */
extern unsigned char *str_chr(const unsigned char *str, unsigned char c);

/* str_rchr - locate character in string
 *
 * - finds last occurrence of the character c
 * - returns pointer to the character in the string (or NULL)
 */
extern unsigned char *str_rchr(const unsigned char *str, unsigned char c);

/* str_str - locate a substring
 *
 * - finds the first *needle in *haystack
 * - returns pointer to the beginning of the substring (or NULL)
 */
extern unsigned char *str_str(const unsigned char *haystack, const unsigned char *needle);

/* str_add - add *append to *dst
 *
 * - dst should be allocated || NULL
 * - be aware: there is no NULL termination in it !!
 * - returns *dst
 */
extern unsigned char *str_add(unsigned char **dst, const unsigned char *append);

/* str_split - split *str into n strings
 *
 * - splits *str into n strings that are separated by c
 * - *str is modified, but not freed
 * - allocate space for them, return n as *len
 */
extern unsigned char **str_split(unsigned char *buf, unsigned char c, size_t *len);

#endif /* LIB_STR_H */
