// Copyright 2022 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "google/cloud/bigtable/table.h"
#include "google/cloud/bigtable/testing/mock_async_failing_rpc_factory.h"
#include "google/cloud/bigtable/testing/table_test_fixture.h"
#include "google/cloud/internal/background_threads_impl.h"
#include "google/cloud/testing_util/fake_completion_queue_impl.h"

namespace google {
namespace cloud {
namespace bigtable {
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_BEGIN
namespace {

namespace btproto = ::google::bigtable::v2;
using ::google::cloud::testing_util::FakeCompletionQueueImpl;
using ::google::cloud::testing_util::StatusIs;
using ::testing::HasSubstr;
using ::testing::MockFunction;

Options NonEmptyOptions() {
  struct TestOption {
    using Type = bool;
  };
  return Options{}.set<TestOption>(true);
}

/// Define types and functions used in the tests.
class TableTest : public ::google::cloud::bigtable::testing::TableTestFixture {
 public:
  TableTest() : TableTestFixture(CompletionQueue{}) {}
};

TEST_F(TableTest, ClientProjectId) {
  EXPECT_EQ(kProjectId, client_->project_id());
}

TEST_F(TableTest, ClientInstanceId) {
  EXPECT_EQ(kInstanceId, client_->instance_id());
}

TEST_F(TableTest, StandaloneInstanceName) {
  EXPECT_EQ(kInstanceName, InstanceName(client_));
}

TEST_F(TableTest, StandaloneTableName) {
  EXPECT_EQ(kTableName, TableName(client_, kTableId));
}

TEST_F(TableTest, TableName) { EXPECT_EQ(kTableName, table_.table_name()); }

TEST_F(TableTest, WithNewTarget) {
  Table table(client_, "original-profile", kTableId);
  EXPECT_EQ(table.project_id(), kProjectId);
  EXPECT_EQ(table.instance_id(), kInstanceId);
  EXPECT_EQ(table.table_id(), kTableId);
  EXPECT_EQ(table.table_name(), TableName(kProjectId, kInstanceId, kTableId));
  EXPECT_EQ(table.app_profile_id(), "original-profile");

  std::string const other_project_id = "other-project";
  std::string const other_instance_id = "other-instance";
  std::string const other_table_id = "other-table";
  auto other_table =
      table.WithNewTarget(other_project_id, other_instance_id, other_table_id);

  EXPECT_EQ(other_table.project_id(), other_project_id);
  EXPECT_EQ(other_table.instance_id(), other_instance_id);
  EXPECT_EQ(other_table.table_id(), other_table_id);
  EXPECT_EQ(other_table.table_name(),
            TableName(other_project_id, other_instance_id, other_table_id));
  EXPECT_EQ(other_table.app_profile_id(), "original-profile");
}

TEST_F(TableTest, WithNewTargetProfile) {
  Table table(client_, "original-profile", kTableId);
  EXPECT_EQ(table.project_id(), kProjectId);
  EXPECT_EQ(table.instance_id(), kInstanceId);
  EXPECT_EQ(table.table_id(), kTableId);
  EXPECT_EQ(table.table_name(), TableName(kProjectId, kInstanceId, kTableId));
  EXPECT_EQ(table.app_profile_id(), "original-profile");

  std::string const other_project_id = "other-project";
  std::string const other_instance_id = "other-instance";
  std::string const other_table_id = "other-table";
  std::string const other_profile_id = "other-profile";
  auto other_table = table.WithNewTarget(other_project_id, other_instance_id,
                                         other_profile_id, other_table_id);

  EXPECT_EQ(other_table.project_id(), other_project_id);
  EXPECT_EQ(other_table.instance_id(), other_instance_id);
  EXPECT_EQ(other_table.table_id(), other_table_id);
  EXPECT_EQ(other_table.table_name(),
            TableName(other_project_id, other_instance_id, other_table_id));
  EXPECT_EQ(other_table.app_profile_id(), other_profile_id);
}

TEST_F(TableTest, TableConstructor) {
  std::string const other_table_id = "my-table";
  std::string const other_table_name = TableName(client_, other_table_id);
  Table table(client_, other_table_id);
  EXPECT_EQ(other_table_name, table.table_name());
}

TEST_F(TableTest, CopyConstructor) {
  Table source(client_, "my-table");
  std::string const& expected = source.table_name();
  // NOLINTNEXTLINE(performance-unnecessary-copy-initialization)
  Table copy(source);
  EXPECT_EQ(expected, copy.table_name());
}

TEST_F(TableTest, MoveConstructor) {
  Table source(client_, "my-table");
  std::string expected = source.table_name();
  Table copy(std::move(source));
  EXPECT_EQ(expected, copy.table_name());
}

TEST_F(TableTest, CopyAssignment) {
  Table source(client_, "my-table");
  std::string const& expected = source.table_name();
  Table dest(client_, "another-table");
  dest = source;
  EXPECT_EQ(expected, dest.table_name());
}

TEST_F(TableTest, MoveAssignment) {
  Table source(client_, "my-table");
  std::string expected = source.table_name();
  Table dest(client_, "another-table");
  dest = std::move(source);
  EXPECT_EQ(expected, dest.table_name());
}

TEST_F(TableTest, ChangeOnePolicy) {
  Table table(client_, "some-table", AlwaysRetryMutationPolicy());
  EXPECT_EQ("", table.app_profile_id());
  EXPECT_THAT(table.table_name(), HasSubstr("some-table"));
}

TEST_F(TableTest, ChangePolicies) {
  Table table(client_, "some-table", AlwaysRetryMutationPolicy(),
              LimitedErrorCountRetryPolicy(42));
  EXPECT_EQ("", table.app_profile_id());
  EXPECT_THAT(table.table_name(), HasSubstr("some-table"));
}

TEST_F(TableTest, ConstructorWithAppProfileAndPolicies) {
  Table table(client_, "test-profile-id", "some-table",
              AlwaysRetryMutationPolicy(), LimitedErrorCountRetryPolicy(42));
  EXPECT_EQ("test-profile-id", table.app_profile_id());
  EXPECT_THAT(table.table_name(), HasSubstr("some-table"));
}

std::string const kProjectId = "the-project";
std::string const kInstanceId = "the-instance";
std::string const kTableId = "the-table";

class ValidContextMdAsyncTest : public ::testing::Test {
 public:
  ValidContextMdAsyncTest()
      : cq_impl_(new FakeCompletionQueueImpl),
        cq_(cq_impl_),
        client_(new ::google::cloud::bigtable::testing::MockDataClient(
            Options{}.set<GrpcCompletionQueueOption>(cq_))) {
    EXPECT_CALL(*client_, project_id())
        .WillRepeatedly(::testing::ReturnRef(kProjectId));
    EXPECT_CALL(*client_, instance_id())
        .WillRepeatedly(::testing::ReturnRef(kInstanceId));
    table_ = std::make_unique<Table>(client_, kTableId);
  }

 protected:
  template <typename ResultType>
  void FinishTest(
      ::google::cloud::future<google::cloud::StatusOr<ResultType>> res_future) {
    EXPECT_EQ(1U, cq_impl_->size());
    cq_impl_->SimulateCompletion(true);
    EXPECT_EQ(0U, cq_impl_->size());
    auto res = res_future.get();
    EXPECT_FALSE(res);
    EXPECT_EQ(::google::cloud::StatusCode::kPermissionDenied,
              res.status().code());
  }

  void FinishTest(::google::cloud::future<google::cloud::Status> res_future) {
    EXPECT_EQ(1U, cq_impl_->size());
    cq_impl_->SimulateCompletion(true);
    EXPECT_EQ(0U, cq_impl_->size());
    auto res = res_future.get();
    EXPECT_EQ(::google::cloud::StatusCode::kPermissionDenied, res.code());
  }

  std::shared_ptr<FakeCompletionQueueImpl> cq_impl_;
  CompletionQueue cq_;
  std::shared_ptr<testing::MockDataClient> client_;
  std::unique_ptr<Table> table_;
};

TEST_F(ValidContextMdAsyncTest, AsyncApply) {
  testing::MockAsyncFailingRpcFactory<btproto::MutateRowRequest,
                                      btproto::MutateRowResponse>
      rpc_factory;
  EXPECT_CALL(*client_, AsyncMutateRow)
      .WillOnce(rpc_factory.Create(
          R"""(
              table_name: "projects/the-project/instances/the-instance/tables/the-table"
              row_key: "row_key"
              mutations: { delete_from_row { } }
          )""",
          "google.bigtable.v2.Bigtable.MutateRow"));
  FinishTest(table_->AsyncApply(SingleRowMutation("row_key", DeleteFromRow())));
}

TEST_F(ValidContextMdAsyncTest, AsyncCheckAndMutateRow) {
  testing::MockAsyncFailingRpcFactory<btproto::CheckAndMutateRowRequest,
                                      btproto::CheckAndMutateRowResponse>
      rpc_factory;
  EXPECT_CALL(*client_, AsyncCheckAndMutateRow)
      .WillOnce(rpc_factory.Create(
          R"""(
              table_name: "projects/the-project/instances/the-instance/tables/the-table"
              row_key: "row_key"
              true_mutations: { delete_from_row { } }
              predicate_filter: { pass_all_filter: true }
          )""",
          "google.bigtable.v2.Bigtable.CheckAndMutateRow"));
  FinishTest(table_->AsyncCheckAndMutateRow("row_key", Filter::PassAllFilter(),
                                            {DeleteFromRow()}, {}));
}

TEST_F(ValidContextMdAsyncTest, AsyncReadModifyWriteRow) {
  testing::MockAsyncFailingRpcFactory<btproto::ReadModifyWriteRowRequest,
                                      btproto::ReadModifyWriteRowResponse>
      rpc_factory;
  EXPECT_CALL(*client_, AsyncReadModifyWriteRow)
      .WillOnce(rpc_factory.Create(
          R"""(
              table_name: "projects/the-project/instances/the-instance/tables/the-table"
              row_key: "row_key"
              rules: {
                  family_name: "fam"
                  column_qualifier: "counter"
                  increment_amount: 1
              }
              rules: {
                  family_name: "fam"
                  column_qualifier: "list"
                  append_value: ";element"
              }
          )""",
          "google.bigtable.v2.Bigtable.ReadModifyWriteRow"));
  FinishTest(table_->AsyncReadModifyWriteRow(
      "row_key", ReadModifyWriteRule::IncrementAmount("fam", "counter", 1),
      ReadModifyWriteRule::AppendValue("fam", "list", ";element")));
}

TEST_F(TableTest, ApplyWithOptions) {
  Table table(client_, kTableId);
  auto status = table.Apply(SingleRowMutation("row"), NonEmptyOptions());
  EXPECT_THAT(status, StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncApplyWithOptions) {
  Table table(client_, kTableId);
  auto status = table.AsyncApply(SingleRowMutation("row"), NonEmptyOptions());
  EXPECT_THAT(status.get(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, BulkApplyWithOptions) {
  Table table(client_, kTableId);
  auto bulk = BulkMutation(SingleRowMutation("r1"), SingleRowMutation("r2"));
  auto failed = table.BulkApply(bulk, NonEmptyOptions());
  ASSERT_EQ(2, failed.size());
  EXPECT_EQ(0, failed[0].original_index());
  EXPECT_THAT(failed[0].status(), StatusIs(StatusCode::kInvalidArgument));
  EXPECT_EQ(1, failed[1].original_index());
  EXPECT_THAT(failed[1].status(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncBulkApplyWithOptions) {
  Table table(client_, kTableId);
  auto bulk = BulkMutation(SingleRowMutation("r1"), SingleRowMutation("r2"));
  auto failed = table.AsyncBulkApply(bulk, NonEmptyOptions()).get();
  ASSERT_EQ(2, failed.size());
  EXPECT_EQ(0, failed[0].original_index());
  EXPECT_THAT(failed[0].status(), StatusIs(StatusCode::kInvalidArgument));
  EXPECT_EQ(1, failed[1].original_index());
  EXPECT_THAT(failed[1].status(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, ReadRowsWithOptions) {
  Table table(client_, kTableId);
  auto reader =
      table.ReadRows(RowSet(), Filter::PassAllFilter(), NonEmptyOptions());
  auto it = reader.begin();
  EXPECT_NE(it, reader.end());
  EXPECT_THAT(*it, StatusIs(StatusCode::kInvalidArgument));
  EXPECT_EQ(++it, reader.end());
}

TEST_F(TableTest, ReadRowsWithLimitWithOptions) {
  Table table(client_, kTableId);
  auto reader =
      table.ReadRows(RowSet(), 1, Filter::PassAllFilter(), NonEmptyOptions());
  auto it = reader.begin();
  EXPECT_NE(it, reader.end());
  EXPECT_THAT(*it, StatusIs(StatusCode::kInvalidArgument));
  EXPECT_EQ(++it, reader.end());
}

TEST_F(TableTest, ReadRowWithOptions) {
  Table table(client_, kTableId);
  auto row = table.ReadRow("row", Filter::PassAllFilter(), NonEmptyOptions());
  EXPECT_THAT(row, StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncReadRowWithOptions) {
  Table table(client_, kTableId);
  auto row =
      table.AsyncReadRow("row", Filter::PassAllFilter(), NonEmptyOptions());
  EXPECT_THAT(row.get(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, CheckAndMutateRowWithOptions) {
  Table table(client_, kTableId);
  auto branch = table.CheckAndMutateRow("row", Filter::PassAllFilter(), {}, {},
                                        NonEmptyOptions());
  EXPECT_THAT(branch, StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncCheckAndMutateRowWithOptions) {
  Table table(client_, kTableId);
  auto branch = table.AsyncCheckAndMutateRow("row", Filter::PassAllFilter(), {},
                                             {}, NonEmptyOptions());
  EXPECT_THAT(branch.get(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, SampleRowsWithOptions) {
  Table table(client_, kTableId);
  auto rows = table.SampleRows(NonEmptyOptions());
  EXPECT_THAT(rows, StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncSampleRowsWithOptions) {
  Table table(client_, kTableId);
  auto rows = table.AsyncSampleRows(NonEmptyOptions());
  EXPECT_THAT(rows.get(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, ReadModifyWriteRowWithOptions) {
  Table table(client_, kTableId);
  auto r = ReadModifyWriteRule::AppendValue("cf", "cq", "v");
  auto row = table.ReadModifyWriteRow("row", r, r, NonEmptyOptions(), r);
  EXPECT_THAT(row, StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncReadModifyWriteRowWithOptions) {
  Table table(client_, kTableId);
  auto r = ReadModifyWriteRule::AppendValue("cf", "cq", "v");
  auto row = table.AsyncReadModifyWriteRow("row", r, r, NonEmptyOptions(), r);
  EXPECT_THAT(row.get(), StatusIs(StatusCode::kInvalidArgument));
}

TEST_F(TableTest, AsyncReadRowsWithOptions) {
  MockFunction<future<bool>(bigtable::Row const&)> on_row;
  EXPECT_CALL(on_row, Call).Times(0);

  MockFunction<void(Status)> on_finish;
  EXPECT_CALL(on_finish, Call).WillOnce([](Status const& status) {
    EXPECT_THAT(status, StatusIs(StatusCode::kInvalidArgument));
  });

  Table table(client_, kTableId);
  table.AsyncReadRows(on_row.AsStdFunction(), on_finish.AsStdFunction(),
                      RowSet(), Filter::PassAllFilter(), NonEmptyOptions());
}

TEST_F(TableTest, AsyncReadRowsWithLimitWithOptions) {
  MockFunction<future<bool>(bigtable::Row const&)> on_row;
  EXPECT_CALL(on_row, Call).Times(0);

  MockFunction<void(Status)> on_finish;
  EXPECT_CALL(on_finish, Call).WillOnce([](Status const& status) {
    EXPECT_THAT(status, StatusIs(StatusCode::kInvalidArgument));
  });

  Table table(client_, kTableId);
  table.AsyncReadRows(on_row.AsStdFunction(), on_finish.AsStdFunction(),
                      RowSet(), 1, Filter::PassAllFilter(), NonEmptyOptions());
}

}  // namespace
GOOGLE_CLOUD_CPP_INLINE_NAMESPACE_END
}  // namespace bigtable
}  // namespace cloud
}  // namespace google
