/*
 * ARCload (c) 2005 Stanislaw Skowronek
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <malloc.h>
#include <time.h>

#include "elf32.h"
#include "ecoff.h"
#include "endian.h"
#include "list.h"


/* memory representation of relocatable */
#define ST_NONE	0
#define ST_TEXT	1
#define ST_DATA	2
#define ST_BSS	3

LIST_STRUCT(Section)
	char		*name;
	int		type;
	unsigned	size;
	unsigned	addr;
	void		*data;
	RAW_LIST_OF(struct Reloc, relocs)
	int		nrelocs;
	int		delta;
	int		index;
	int		gccfix;
};
LIST_FIND_BY_FIELD(section_by_index, struct Section, int, index)

LIST_STRUCT(Reloc)
	int		type;
	unsigned	offset;
	struct Symbol	*sym;
};

LIST_STRUCT(Symbol)
	char		*name;
	struct Section	*section;
	unsigned	offset;
	int		undef;
	int		used;
	int		nameoff;
	int		delta;
	int		index;
};
LIST_FIND_BY_FIELD(symbol_by_index, struct Symbol, int, index)

LIST_OF(struct Section, sections)
LIST_OF(struct Symbol, symbols)
Elf32_RegInfo reginfo;


/* relocating sections */
void fatal(char *f, ...);

void perform_relocation(void)
{
	struct Section	*section;
	struct Symbol	*symbol;
	struct Reloc	*reloc;

	LIST_FOR(symbol, symbols)
		if(symbol->section) {
			symbol->delta = symbol->section->delta;
			symbol->offset += symbol->section->delta;
		} else
			symbol->delta = 0;

	LIST_FOR(section, sections) {
		LIST_FOR(reloc, section->relocs)
			reloc->offset += section->delta;
		section->addr += section->delta;
		section->delta = 0;
	}
}


void linear_relocation(unsigned int start)
{
	struct Section *section;

	LIST_FOR(section, sections) {
		section->delta = (start - section->addr);
		start += section->size;
	}
	perform_relocation();
}


/* writing ECOFF */
static char INFOSTR[] = "\nwreckoff (c) 2005 Stanislaw Skowronek\n";

void synthesize_ecoff(FILE *f)
{
	struct filehdr	fhdr;
	struct aouthdr	ahdr;
	struct scnhdr	shdr;
	struct reloc	rhdr;
	struct sgihdrr	ghdr;
	struct sgiextr	mhdr;
	struct Section	*section;
	struct Symbol	*symbol;
	struct Reloc	*reloc;
	int		i, nsyms, nscns, vptr, rptr, sptr, align;
	int		strsize, sym_sc = scNil, sym_st = 0;
	char		*strtab;

	/* reindex sections and symbols */
	nscns = 0;
	LIST_FOR(section, sections) {
		if(section->gccfix)
			section->size += 0x10;
		section->index = (nscns++);
	}

	nsyms = 0;
	LIST_FOR(symbol, symbols)
		if(symbol->used)
			symbol->index = (nsyms++);
		else
			symbol->index =- 1;

	/* calculate sizes */
	vptr = ((sizeof(struct filehdr) + sizeof(struct aouthdr)) + (nscns * sizeof(struct scnhdr)));
	align = ((vptr & 15) ? (16 - (vptr & 15)) : 0);
	vptr += align;
	rptr = vptr;
	sptr = vptr;
	LIST_FOR(section, sections) {
		if(section->data) {
			rptr += section->size;
			sptr += section->size;
		}
		sptr += (section->nrelocs * sizeof(struct reloc));
	}

	/* write file header */
	fhdr.f_magic	= MIPS2EBMAGIC;
	fhdr.f_nscns	= nscns;
	fhdr.f_timdat	= time(0);
	fhdr.f_symptr	= sptr;
	fhdr.f_nsyms	= 96;
	fhdr.f_opthdr	= 56;
	fhdr.f_flags	= (F_EXEC | F_MIPS_NO_REORG);
	swap_filehdr(&fhdr);
	fwrite(&fhdr, sizeof(struct filehdr), 1, f);

	/* write a.out header */
	ahdr.magic	= OMAGIC;
	ahdr.vstamp	= AOUT_VERSION;
	ahdr.tsize	= 0;
	ahdr.dsize	= 0;
	ahdr.bsize	= 0;
	ahdr.text_start	= 0x7fffffff;
	ahdr.data_start	= 0x7fffffff;
	ahdr.bss_start	= 0x7fffffff;

	LIST_FOR(section, sections) {
		if(section->type == ST_TEXT) {
			if(ahdr.text_start > section->addr)
				ahdr.text_start = section->addr;
			ahdr.tsize += section->size;
		}
		if(section->type == ST_DATA) {
			if(ahdr.data_start > section->addr)
				ahdr.data_start = section->addr;
			ahdr.dsize += section->size;
		}
		if(section->type == ST_BSS) {
			if(ahdr.bss_start > section->addr)
				ahdr.bss_start = section->addr;
			ahdr.bsize += section->size;
		}
	}

	if(ahdr.text_start == 0x7fffffff)
		ahdr.text_start = 0;
	if(ahdr.data_start == 0x7fffffff)
		ahdr.data_start = 0;
	if(ahdr.bss_start == 0x7fffffff)
		ahdr.bss_start = 0;

	ahdr.entry = ahdr.text_start;
	ahdr.gprmask = reginfo.ri_gprmask;

	for(i = 0; i < 4; i++)
		ahdr.cprmask[i] = reginfo.ri_cprmask[i];

	ahdr.gp_value = reginfo.ri_gp_value;
	swap_aouthdr(&ahdr);
	fwrite(&ahdr, sizeof(struct aouthdr), 1, f);

	LIST_FOR(section, sections) {
		strncpy(shdr.s_name, section->name, 8);
		shdr.s_paddr	= section->addr;
		shdr.s_vaddr	= section->addr;
		shdr.s_size	= section->size;

		if(section->data) {
			shdr.s_scnptr = vptr;
			vptr += section->size;
		} else
			shdr.s_scnptr = 0;

		if(section->nrelocs) {
			shdr.s_relptr = rptr;
			rptr += (section->nrelocs * sizeof(struct reloc));
		} else
			shdr.s_relptr = 0;

		shdr.s_lnnoptr	= 0;
		shdr.s_nreloc	= section->nrelocs;
		shdr.s_nlnno	= 0;
		shdr.s_flags	= ((section->data) ? STYP_REG : STYP_NOLOAD);

		if(section->type == ST_TEXT)
			shdr.s_flags = STYP_TEXT;
		if(section->type == ST_DATA)
			shdr.s_flags = STYP_DATA;
		if(section->type == ST_BSS)
			shdr.s_flags = STYP_BSS;

		swap_scnhdr(&shdr);
		fwrite(&shdr, sizeof(struct scnhdr), 1, f);
	}

	i = 0;
	while(align--)
		fwrite(&i, 1, 1, f);

	/* save section contents */
	LIST_FOR(section, sections)
		if(section->data) {
			if(section->gccfix) {
				fwrite(section->data, (section->size - 0x10), 1, f);
				align = 0;
				for(i = 0; i < 4; i++)
					fwrite(&align, 4, 1, f);
			} else
				fwrite(section->data, section->size, 1, f);
		}

	/* save section relocs */
	LIST_FOR(section, sections)
		LIST_FOR(reloc, section->relocs) {
			switch(reloc->type) {
			case R_MIPS_32:
				i = R_REFWORD;
				break;

			case R_MIPS_26:
				i = R_JMPADDR;
				break;

			case R_MIPS_HI16:
				i = R_REFHI;
				break;

			case R_MIPS_LO16:
				i = R_REFLO;
				break;

			default:
				fatal("wreckoff: relocation type %d not representable.\n", reloc->type);
			}

			rhdr.r_vaddr	= reloc->offset;
			rhdr.r		= R_R(reloc->sym->index, i, 1);
			swap_reloc(&rhdr);
			fwrite(&rhdr, sizeof(struct reloc), 1, f);
		}

	/* create strtab */
	strsize = (strlen(INFOSTR) + 1);
	LIST_FOR(symbol, symbols)
		if(symbol->used) {
			symbol->nameoff = strsize;
			strsize += (strlen(symbol->name) + 1);
		}

	strsize = ((strsize + 15) & (~15));
	strtab = malloc(strsize);
	strcpy(strtab, INFOSTR);
	LIST_FOR(symbol, symbols)
		if(symbol->used)
			strcpy((strtab + symbol->nameoff), symbol->name);

	/* save symbolic header */
	memset(&ghdr, 0, sizeof(struct sgihdrr));
	ghdr.magic		= SGIHDRR_MAGIC;
	ghdr.vstamp		= AOUT_VERSION;
	ghdr.issExtMax		= strsize;
	ghdr.cbSsExtOffset	= (sptr + sizeof(struct sgihdrr));
	ghdr.iextMax		= nsyms;
	ghdr.cbExtOffset	= (sptr + sizeof(struct sgihdrr) + strsize);
	swap_sgihdrr(&ghdr);
	fwrite(&ghdr, sizeof(struct sgihdrr), 1, f);

	/* save strtab */
	fwrite(strtab, strsize, 1, f);

	/* save symbols */
	LIST_FOR(symbol, symbols)
		if(symbol->used) {
			if(symbol->section) {
				if(symbol->section->type == ST_TEXT) {
					sym_sc = scText;
					sym_st = 6;
				}
				if(symbol->section->type == ST_DATA) {
					sym_sc = scData;
					sym_st = 1;
				}
				if(symbol->section->type == ST_BSS) {
					sym_sc = scBss;
					sym_st = 1;
				}
			} else
				sym_sc=scAbs;

			mhdr.flags	= 0x00000000;
			mhdr.iss	= symbol->nameoff;
			mhdr.value	= symbol->offset;
			mhdr.data	= S_DATA(sym_st, sym_sc, 0xFFFFF);
			swap_sgiextr(&mhdr);
			fwrite(&mhdr, sizeof(struct sgiextr), 1, f);
		}
}


/* reading ELF32 */
char *load_strtab(FILE *f, unsigned off, int sz)
{
	char *p = malloc(sz);
	fseek(f, off, SEEK_SET);
	fread(p, sz, 1, f);
	return p;
}


unsigned char elf32_ident[6] = {0x7F, 0x45, 0x4C, 0x46, 0x01, 0x02};
void analyze_elf32(FILE *f)
{
	Elf32_Ehdr	ehdr;
	Elf32_Shdr	shdr;
	Elf32_Sym	mhdr;
	Elf32_Rela	rhdr;
	Elf32_Rel	qhdr;
	struct Section	*section;
	struct Symbol	*symbol, *lastrsym = NULL;
	struct Reloc	*reloc;
	int 		i, j, symtab;
	char 		**strtabs, **shnames;

	LIST_INIT(sections);
	LIST_INIT(symbols);

	/* load ELF header */
	fseek(f, 0, SEEK_SET);
	fread(&ehdr, sizeof(Elf32_Ehdr), 1, f);
	swap_Ehdr(&ehdr);
	for(i = 0; i < 6; i++)
		if(ehdr.e_ident[i] != elf32_ident[i])
			fatal("wreckoff: invalid ELF file.\n");

	if(ehdr.e_machine != 8)
		fatal("wreckoff: invalid ELF file.\n");

	/* load strtabs; find symtab */
	symtab = -1;
	strtabs = calloc(sizeof(char *), ehdr.e_shnum);
	for(i = 0; i < ehdr.e_shnum; i++) {
		fseek(f, (ehdr.e_shoff + (ehdr.e_shentsize * i)), SEEK_SET);
		fread(&shdr, sizeof(Elf32_Shdr), 1, f);
		swap_Shdr(&shdr);

		if(shdr.sh_type == SHT_STRTAB)
			strtabs[i] = load_strtab(f, shdr.sh_offset, shdr.sh_size);

		if(shdr.sh_type == SHT_SYMTAB) {
			if(symtab != -1)
				fatal("wreckoff: multiple symbol tables not supported.\n");
			symtab = i;
		}
	}

	/* create sections, load reginfo */
	shnames=calloc(sizeof(char *),ehdr.e_shnum);
	for(i = 0; i < ehdr.e_shnum; i++) {
		fseek(f, (ehdr.e_shoff + (ehdr.e_shentsize * i)), SEEK_SET);
		fread(&shdr, sizeof(Elf32_Shdr), 1, f);
		swap_Shdr(&shdr);
		shnames[i] = (strtabs[ehdr.e_shstrndx] + shdr.sh_name);

		if((shdr.sh_flags & SHF_ALLOC)) {
			if(shdr.sh_type == SHT_REGINFO) {
				fseek(f, shdr.sh_offset, SEEK_SET);
				fread(&reginfo, sizeof(Elf32_RegInfo), 1, f);
				swap_RegInfo(&reginfo);
				continue;
			}

			if(shdr.sh_type >= SHT_LOPROC)
				fatal("wreckoff: unknown ALLOC section 0x%08X.\n", shdr.sh_type);

			section		= calloc(sizeof(struct Section), 1);
			section->name	= shnames[i];
			section->type	= ST_DATA;

			if((shdr.sh_flags & SHF_EXECINSTR))
				section->type = ST_TEXT;

			if(shdr.sh_type == SHT_NOBITS)
				section->type = ST_BSS;

			section->size	= shdr.sh_size;
			section->addr	= shdr.sh_addr;
			section->index	= i;

			if(shdr.sh_type != SHT_NOBITS)
				section->data = load_strtab(f, shdr.sh_offset, shdr.sh_size);

			RAW_LIST_INIT(*section, relocs);
			LIST_ADD(sections, section);
		}
	}

	/* load symbols */
	if(symtab != -1) {
		fseek(f, (ehdr.e_shoff + (ehdr.e_shentsize * symtab)), SEEK_SET);
		fread(&shdr, sizeof(Elf32_Shdr), 1, f);
		swap_Shdr(&shdr);

		if(!shdr.sh_entsize)
			fatal("wreckoff: symtab section ('%s') has zero entsize.\n", shnames[symtab]);

		if(!strtabs[shdr.sh_link])
			fatal("wreckoff: symtab section ('%s') uses null string section %d.\n", shnames[symtab], shdr.sh_link);

		for( i = 0; i < (shdr.sh_size / shdr.sh_entsize); i++) {
			fseek(f, (shdr.sh_offset + (shdr.sh_entsize * i)), SEEK_SET);
			fread(&mhdr, sizeof(Elf32_Sym), 1, f);
			swap_Sym(&mhdr);
			symbol = calloc(sizeof(struct Symbol), 1);
			symbol->name = (strtabs[shdr.sh_link] + mhdr.st_name);

			if(((mhdr.st_info & STT_MASK) == STT_SECTION) && !(*(symbol->name)))
				symbol->name = shnames[mhdr.st_shndx];

			symbol->offset = mhdr.st_value;
			symbol->section = section_by_index(sections, mhdr.st_shndx);

			if(!(symbol->section) && (mhdr.st_shndx != SHN_ABS))
				symbol->undef = 1;

			symbol->index = i;

			LIST_ADD(symbols, symbol);
		}
	}

	/* load rels and decompose relas */
	for(i = 0; i < ehdr.e_shnum; i++) {
		fseek(f, (ehdr.e_shoff + (ehdr.e_shentsize * i)), SEEK_SET);
		fread(&shdr, sizeof(Elf32_Shdr), 1, f);
		swap_Shdr(&shdr);

		if((shdr.sh_type != SHT_REL) && (shdr.sh_type != SHT_RELA))
			continue;

		section = section_by_index(sections, shdr.sh_info);

		if(!section)	/* lose relocation */
			continue;

		if(shdr.sh_link != symtab)
			fatal("wreckoff: relocation section ('%s') uses unknown symtab ('%s')\n", shnames[i], shnames[shdr.sh_link]);

		if(!shdr.sh_entsize)
			fatal("wreckoff: relocation section ('%s') has zero entsize.\n", shnames[symtab]);

		for(j = 0; j < (shdr.sh_size / shdr.sh_entsize); j++) {
			fseek(f, (shdr.sh_offset + (shdr.sh_entsize * j)), SEEK_SET);

			if(shdr.sh_type == SHT_REL) {
				fread(&qhdr, sizeof(Elf32_Rel), 1, f);
				swap_Rel(&qhdr);
				rhdr.r_offset	= qhdr.r_offset;
				rhdr.r_info	= qhdr.r_info;
				rhdr.r_addend	= 0;
			} else {
				fread(&rhdr, sizeof(Elf32_Rela), 1, f);
				swap_Rela(&rhdr);
			}

			symbol = symbol_by_index(symbols, ELF32_R_SYM(rhdr.r_info));

			if(!symbol)
				fatal("wreckoff: relocation %d in '%s' uses invalid symbol %d\n", j, shnames[i], ELF32_R_SYM(rhdr.r_info));

			if(symbol->undef)
				fatal("wreckoff: relocation %d in '%s' uses undefined symbol '%s'\n", j, shnames[i], symbol->name);

			symbol->used = 1;

			/* fix GCC/PROM bugs */
			if(ELF32_R_TYPE(rhdr.r_info) == R_MIPS_LO16) {
				if(lastrsym != symbol) {
					fprintf(stderr, "wreckoff: fixing R_MIPS_LO16 bug...\n");
					reloc		= calloc(sizeof(struct Reloc), 1);
					reloc->type	= R_MIPS_HI16;
					reloc->offset	= section->size;
					section->gccfix	= 1;
					reloc->sym	= symbol;
					LIST_ADD(section->relocs, reloc);
					section->nrelocs++;
				}
			} else {
				if(lastrsym) {
					fprintf(stderr, "wreckoff: fixing R_MIPS_HI16 bug...\n");
					reloc		= calloc(sizeof(struct Reloc), 1);
					reloc->type	= R_MIPS_LO16;
					reloc->offset	= section->size + 4;
					section->gccfix	= 1;
					reloc->sym	= symbol;
					LIST_ADD(section->relocs, reloc);
					section->nrelocs++;
				}
			}

			if(ELF32_R_TYPE(rhdr.r_info) == R_MIPS_HI16)
				lastrsym = symbol;
			else
				lastrsym = NULL;

			/* create normal reloc */
			reloc = calloc(sizeof(struct Reloc), 1);
			switch(ELF32_R_TYPE(rhdr.r_info)) {
			case R_MIPS_32:
			case R_MIPS_26:
			case R_MIPS_HI16:
			case R_MIPS_LO16:
				reloc->type = ELF32_R_TYPE(rhdr.r_info);
				break;

			default:
				fatal("wreckoff: relocation %d in '%s' (for '%s') uses unknown relocation type %d.\n", 
				       j, shnames[i], symbol->name, ELF32_R_TYPE(rhdr.r_info));
			}

			reloc->offset	= rhdr.r_offset;
			reloc->sym	= symbol;
			LIST_ADD(section->relocs, reloc);
			section->nrelocs++;

			/* .rela */
			if(rhdr.r_addend)
				fatal("wreckoff: relocation %d in '%s' uses addend for .rela.\n", j, shnames[i]);
		}
	}
}


/* status display */
void write_status(void)
{
	struct Section *section;
	LIST_FOR(section, sections)
		fprintf(stderr, "\t'%s'\taddr 0x%08X\tsize 0x%08X\n", section->name, section->addr, section->size);
}


/* main driver */
int main(int argc, char **argv)
{
	FILE *f;
	if(argc < 3)
		fatal("usage:\n\twreckoff <input-elf32> <output-ecoff>\n");

	fprintf(stderr, "wreckoff: converting '%s' to '%s'...\n", argv[1], argv[2]);

	/* Read input ELF32 file */
	f = fopen(argv[1], "r");
	if(!f)
		fatal("wreckoff: input file not found.\n");

	analyze_elf32(f);
	fclose(f);
	linear_relocation(0x10000000);

	/* Write out output ECOFF file */
	f = fopen(argv[2], "w");
	synthesize_ecoff(f);
	fclose(f);
	write_status();

	return 0;
}


void fatal(char *f, ...)
{
	va_list ap;
	va_start(ap, f);
	vfprintf(stderr, f, ap);
	exit(1);
}
