package matchers_test

import (
	. "github.com/onsi/ginkgo/v2"
	. "github.com/onsi/gomega"
	. "github.com/onsi/gomega/matchers"
)

var _ = Describe("MatchRegexp", func() {
	When("actual is a string", func() {
		It("should match against the string", func() {
			Expect(" a2!bla").Should(MatchRegexp(`\d!`))
			Expect(" a2!bla").ShouldNot(MatchRegexp(`[A-Z]`))
		})
	})

	When("actual is a stringer", func() {
		It("should call the stringer and match against the returned string", func() {
			Expect(&myStringer{a: "Abc3"}).Should(MatchRegexp(`[A-Z][a-z]+\d`))
		})
	})

	When("the matcher is called with multiple arguments", func() {
		It("should pass the string and arguments to sprintf", func() {
			Expect(" a23!bla").Should(MatchRegexp(`\d%d!`, 3))
		})
	})

	When("actual is neither a string nor a stringer", func() {
		It("should error", func() {
			success, err := (&MatchRegexpMatcher{Regexp: `\d`}).Match(2)
			Expect(success).Should(BeFalse())
			Expect(err).Should(HaveOccurred())
		})
	})

	When("the passed in regexp fails to compile", func() {
		It("should error", func() {
			success, err := (&MatchRegexpMatcher{Regexp: "("}).Match("Foo")
			Expect(success).Should(BeFalse())
			Expect(err).Should(HaveOccurred())
		})
	})

	It("shows failure message", func() {
		failuresMessages := InterceptGomegaFailures(func() {
			Expect("foo").To(MatchRegexp("bar"))
		})
		Expect(failuresMessages).To(Equal([]string{"Expected\n    <string>: foo\nto match regular expression\n    <string>: bar"}))
	})

	It("shows negated failure message", func() {
		failuresMessages := InterceptGomegaFailures(func() {
			Expect("foo").ToNot(MatchRegexp("foo"))
		})
		Expect(failuresMessages).To(Equal([]string{"Expected\n    <string>: foo\nnot to match regular expression\n    <string>: foo"}))
	})
})
