/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  zip.c

  Zip file management functions
*/

#include <sys/types.h>
#include <sys/stat.h>

#include "common.h"
#include <mamory/zip.h>

int FreeZip(s_Zip *Zip)
{
  unsigned int i;

  if(!Zip)
    return 1;

  for(i=0;i<Zip->NbFileEntries;i++)
    {
      XFREE(Zip->FileEntries[i]->LocalFileHeader);
      XFREE(Zip->FileEntries[i]->FileName);
      XFREE(Zip->FileEntries[i]->ExtraField);
      XFREE(Zip->FileEntries[i]->DataDesc);
      XFREE(Zip->FileEntries[i]->Data);
      XFREE(Zip->FileEntries[i]);
    };

  XFREE(Zip->FileEntries);
  Zip->NbFileEntries=0;

  for(i=0;i<Zip->NbCentralDirEntries;i++)
    {
      XFREE(Zip->CentralDirEntries[i]->Const);
      
      XFREE(Zip->CentralDirEntries[i]->FileName);
	    
      XFREE(Zip->CentralDirEntries[i]->ExtraField);

      XFREE(Zip->CentralDirEntries[i]->Comment);
      
      XFREE(Zip->CentralDirEntries[i]);
    };
  XFREE(Zip->CentralDirEntries);
  Zip->NbCentralDirEntries=0;

  if(Zip->CentralDirEnd)
    {
      XFREE(Zip->CentralDirEnd->Const);

      XFREE(Zip->CentralDirEnd->Comment);
    };
  
  XFREE(Zip->CentralDirEnd);
  XFREE(Zip);
  return 1;
};


/* TODO: use error codes for returned value (cf common/myerrno.h )*/
int Zipfclose(s_Zip *Zip)
{
  unsigned int res;

  if (Zip==NULL)
    return 0;
  if((res=fclose(Zip->fp))!=0)
    return res;

  Zip->fp=NULL;
  FreeZip(Zip);

  return res;
};
  
/* TODO: use error codes for the returned value, which means testing for
   space left and other things */
int WriteZipLocalFileEntry(s_Zip *Zip, 
		      s_ZipLocalFileEntry *ZipLocalFileEntry, 
		      s_ZipCentralDirEntry *CentralDirEntry)
{
  s_ZipLocalFileHeader zlfh=*ZipLocalFileEntry->LocalFileHeader;
  s_ZipLocalFileDataDesc zlfdd=*ZipLocalFileEntry->DataDesc;

  FIX_ZLFH_ENDIANNESS(zlfh);
  fwrite(&zlfh,sizeof(s_ZipLocalFileHeader),1,Zip->fp);
  fwrite(ZipLocalFileEntry->FileName,sizeof(unsigned char),
	 ZipLocalFileEntry->LocalFileHeader->FileNameLen,Zip->fp);
  if(ZipLocalFileEntry->LocalFileHeader->ExtraFieldLen>0)
    fwrite(ZipLocalFileEntry->ExtraField,sizeof(unsigned char),
	   ZipLocalFileEntry->LocalFileHeader->ExtraFieldLen,Zip->fp);
  if((ZipLocalFileEntry->LocalFileHeader->GenFlag & 8))
    fwrite(ZipLocalFileEntry->Data,sizeof(char),
	   ZipLocalFileEntry->DataDesc->CompSize,Zip->fp);
  else
    fwrite(ZipLocalFileEntry->Data,sizeof(char),
	   ZipLocalFileEntry->LocalFileHeader->CompSize,Zip->fp);

  if(ZipLocalFileEntry->DataDesc!=NULL)
    {
      FIX_ZLFDD_ENDIANNESS(zlfdd);
      fwrite(&zlfdd,sizeof(s_ZipLocalFileDataDesc),1,Zip->fp);
    };

  Zip->CentralDirEntries=XREALLOC(Zip->CentralDirEntries,
				  s_ZipCentralDirEntry*,
				  Zip->NbCentralDirEntries+1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]=
    XCALLOC(s_ZipCentralDirEntry,1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const=
    XCALLOC(s_ZipCentralDirEntryConst,1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const->Sig=CentralDirEntry->Const->Sig;

  return 1;
};

int isFileNameinZip(s_Zip *Zip,unsigned char *file)
{
  unsigned int i;

  if(!Zip || !file)
    {
      if(!Zip)
	{
	  if(file)
	    LPRINTF(DEBUG,"Zip can not be NULL in isZipLocalFile(Zip,\"%s\")",
		    file);
	  else
	    LPRINTF(DEBUG,"Zip can not be NULL in isZipLocalFile(Zip,file)");
	}
      if(!file)
	LPRINTF(DEBUG,"file can not be NULL in isZipLocalFile(Zip,file)");

      return 0;
    }

  for (i=0;i<Zip->NbCentralDirEntries;i++)
    if(Zip->CentralDirEntries[i]->Const->FileNameLen==strlen(file) &&
       strncmp(Zip->CentralDirEntries[i]->FileName,file,strlen(file))==0)
      break;
  
  if(i==Zip->NbCentralDirEntries)
    return 0;
  else
    return 1;
};

unsigned char *isFileCRCinZip(s_Zip *Zip,unsigned int CRC32,unsigned int size)
{
  unsigned int i;
  unsigned char *FileName=NULL;

  if(!Zip || !CRC32)
    {
      if(!Zip)
	LPRINTF(DEBUG,"Zip can not be NULL in isFileCRCinZip(Zip,%X)",
		CRC32);

      if(!CRC32)
	LPRINTF(DEBUG,"CRC32 can not be 0 in isFileCRCinZip(Zip,CRC32)");

      return FileName;
    }

  for (i=0;i<Zip->NbCentralDirEntries;i++)
    if(Zip->CentralDirEntries[i]->Const->CRC==CRC32 &&
       Zip->CentralDirEntries[i]->Const->UncompSize==size)
      {
	FileName=XCALLOC(unsigned char, 
			 Zip->CentralDirEntries[i]->Const->FileNameLen +1);
	strncat(FileName,Zip->CentralDirEntries[i]->FileName,
		Zip->CentralDirEntries[i]->Const->FileNameLen);
	break;
      };
  
  return FileName;
};

s_ZipLocalFileEntry *GetZipLocalFileEntry(s_Zip *Zip,unsigned char *file)
{
  s_ZipLocalFileEntry *Entry;
  unsigned int i;
  
  for (i=0;i<Zip->NbCentralDirEntries;i++)
    if(Zip->CentralDirEntries[i]->Const->FileNameLen==strlen(file) &&
       strncmp(Zip->CentralDirEntries[i]->FileName,file,strlen(file))==0)
      break;
  
  if(i==Zip->NbCentralDirEntries)
    return NULL;
  
  Entry=XCALLOC(s_ZipLocalFileEntry,1);

  
  fseek(Zip->fp,
	Zip->CentralDirEntries[i]->Const->LocalHeaderOffset,
	SEEK_SET);
  
  Entry->LocalFileHeader=XCALLOC(s_ZipLocalFileHeader,1);

  fread(Entry->LocalFileHeader,sizeof(s_ZipLocalFileHeader),1,Zip->fp);
  FIX_ZLFH_ENDIANNESS(*Entry->LocalFileHeader);
    
  Entry->FileName=XCALLOC(unsigned char,
			  Entry->LocalFileHeader->FileNameLen);

  fread(Entry->FileName,Entry->LocalFileHeader->FileNameLen,
	sizeof(unsigned char),
	Zip->fp);
  if(Entry->LocalFileHeader->ExtraFieldLen!=0)
    {  
      Entry->ExtraField=XCALLOC(unsigned char,
				Entry->LocalFileHeader->ExtraFieldLen);

      fread(Entry->ExtraField,
	    Entry->LocalFileHeader->ExtraFieldLen,
	    sizeof(unsigned char),
	    Zip->fp);
    }
  else
    Entry->ExtraField=NULL;

  if(Entry->LocalFileHeader->GenFlag & 8)
    {
      unsigned int sig=0;
      unsigned int size=0;

      Entry->DataDesc=XCALLOC(s_ZipLocalFileDataDesc,1);      
      size=ftell(Zip->fp);
      while(1)
	{
	  while(0x50!=fgetc(Zip->fp)); /* 0x50=beginning of local header sig */
	  fseek(Zip->fp,-1,SEEK_CUR);
	  fread(&sig,1,4,Zip->fp);
	  sig=LE2ME_32(sig);
	  if(sig==0x08074b50)
	    break;
	  fseek(Zip->fp,-3,SEEK_CUR);
	}
      size=ftell(Zip->fp)-4-size;
      Entry->Data=XCALLOC(char,size);
      fseek(Zip->fp,-4-size,SEEK_CUR);
      fread(Entry->Data,
	    sizeof(char),
	    size,
	    Zip->fp);      

      fread(Entry->DataDesc,sizeof(s_ZipLocalFileDataDesc),1,Zip->fp);
      FIX_ZLFDD_ENDIANNESS(*Entry->DataDesc);
    }
  else
    {
      Entry->Data=XCALLOC(char,Entry->LocalFileHeader->CompSize);
      fread(Entry->Data,
	    Entry->LocalFileHeader->CompSize,
	    sizeof(char),
	    Zip->fp);
    };

  return Entry;
};

/* TODO: use error codes for the returned value, which means testing for
   space left and other things */
int CopyZipLocalFile(s_Zip *DestZip,s_Zip *SourceZip, 
		     unsigned char *file, unsigned char *newname)
{
  s_ZipLocalFileEntry *Entry;
  s_ZipLocalFileHeader zlfh;
  s_ZipLocalFileDataDesc zlfdd;
  long offset=0;
  unsigned int i;
  
  for (i=0;i<SourceZip->NbCentralDirEntries;i++)
    if(SourceZip->CentralDirEntries[i]->Const->FileNameLen==strlen(file) &&
       strncmp(SourceZip->CentralDirEntries[i]->FileName,file,strlen(file))==0)
      break;

  Entry=GetZipLocalFileEntry(SourceZip,file);

  if(Entry==NULL)
    return -1;

  if(newname!=NULL)
    {
      XFREE(Entry->FileName);
      Entry->LocalFileHeader->FileNameLen=strlen(newname);
      Entry->FileName=XCALLOC(unsigned char,strlen(newname));

      strncpy(Entry->FileName,newname,strlen(newname));
    };
  
  fseek(DestZip->fp,0,SEEK_END);
  offset=ftell(DestZip->fp); 

  zlfh=*Entry->LocalFileHeader;
  FIX_ZLFH_ENDIANNESS(zlfh);
  DestZip->Size+=fwrite(Entry->LocalFileHeader,1,sizeof(s_ZipLocalFileHeader),DestZip->fp);
  DestZip->Size+=fwrite(Entry->FileName,sizeof(unsigned char),
		    Entry->LocalFileHeader->FileNameLen,DestZip->fp);
  if(Entry->LocalFileHeader->ExtraFieldLen>0)
    DestZip->Size+=fwrite(Entry->ExtraField,sizeof(unsigned char),
		      Entry->LocalFileHeader->ExtraFieldLen,DestZip->fp);
  if(Entry->LocalFileHeader->GenFlag & 8)
    DestZip->Size+=fwrite(Entry->Data,sizeof(char),
			  Entry->DataDesc->CompSize,DestZip->fp);
  else
    DestZip->Size+=fwrite(Entry->Data,sizeof(char),
			  Entry->LocalFileHeader->CompSize,DestZip->fp);
  if(Entry->DataDesc!=NULL)
    {
      zlfdd=*Entry->DataDesc;
      FIX_ZLFDD_ENDIANNESS(zlfdd);
      DestZip->Size+=fwrite(&zlfdd,1,sizeof(s_ZipLocalFileDataDesc),DestZip->fp);
    };

  DestZip->CentralDirEntries=XREALLOC(DestZip->CentralDirEntries,
				      s_ZipCentralDirEntry*,
				      DestZip->NbCentralDirEntries+1);

  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]=
    XCALLOC(s_ZipCentralDirEntry,1);

  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const=
    XCALLOC(s_ZipCentralDirEntryConst,1);

  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->Sig=
    SourceZip->CentralDirEntries[i]->Const->Sig;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->VerMade=
    SourceZip->CentralDirEntries[i]->Const->VerMade;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->VerExt=
    SourceZip->CentralDirEntries[i]->Const->VerExt;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->GenFlag=
    SourceZip->CentralDirEntries[i]->Const->GenFlag;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CompMethod=
    SourceZip->CentralDirEntries[i]->Const->CompMethod;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->LastModFileTime=
    SourceZip->CentralDirEntries[i]->Const->LastModFileTime;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->LastModFileDate=
    SourceZip->CentralDirEntries[i]->Const->LastModFileDate;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CRC=
    SourceZip->CentralDirEntries[i]->Const->CRC;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CompSize=
    SourceZip->CentralDirEntries[i]->Const->CompSize;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->UncompSize=
    SourceZip->CentralDirEntries[i]->Const->UncompSize;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen=
    SourceZip->CentralDirEntries[i]->Const->FileNameLen;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->ExtraFieldLen=
    SourceZip->CentralDirEntries[i]->Const->ExtraFieldLen;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CommentLen=
    SourceZip->CentralDirEntries[i]->Const->CommentLen;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->StartDiskNb=
    SourceZip->CentralDirEntries[i]->Const->StartDiskNb;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->IntFileAttr=
    SourceZip->CentralDirEntries[i]->Const->IntFileAttr;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->ExtFileAttr=
    SourceZip->CentralDirEntries[i]->Const->ExtFileAttr;
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->LocalHeaderOffset=offset;

  if(newname==NULL)
    {
      DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->FileName=
	XCALLOC(unsigned char,
		DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen);

      strncpy(DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->FileName,
	      SourceZip->CentralDirEntries[i]->FileName,
	      DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen);
    }
  else
    {
      DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen=strlen(newname);
      DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->FileName=
	XCALLOC(unsigned char,
		DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen);

      strncpy(DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->FileName,
	      newname,
	      DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->FileNameLen);
    }

  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->ExtraField=
    XCALLOC(unsigned char,
	    DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->ExtraFieldLen);  

  strncpy(DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->ExtraField,
	  SourceZip->CentralDirEntries[i]->ExtraField,
	  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->ExtraFieldLen);
  
  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Comment=
    XCALLOC(unsigned char,
	    DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CommentLen);  

  strncpy(DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Comment,
	  SourceZip->CentralDirEntries[i]->Comment,
	  DestZip->CentralDirEntries[DestZip->NbCentralDirEntries]->Const->CommentLen);

  DestZip->NbCentralDirEntries++;

  XFREE(Entry->LocalFileHeader);
  XFREE(Entry->FileName);
  XFREE(Entry->ExtraField);
  XFREE(Entry->Data);
  XFREE(Entry->DataDesc);
  XFREE(Entry);

  return 0;
};

/* TODO: use error codes for the returned value, which means testing for
   space left and other things */
int WriteZipCentralDir(s_Zip *Zip,unsigned char *Comment)
{
  s_ZipCentralDirEntryConst zcdentry;
  s_ZipCentralDirEndConst zcdend;
  long offset;
  unsigned int i,centraldirsize=0;

  fseek(Zip->fp,0,SEEK_END);
  offset=ftell(Zip->fp);

  for(i=0;i<Zip->NbCentralDirEntries;i++)
    {
      zcdentry=*Zip->CentralDirEntries[i]->Const;
      FIX_ZCDENTRY_ENDIANNESS(zcdentry);
      centraldirsize+=fwrite(&zcdentry,1,
			     sizeof(s_ZipCentralDirEntryConst),
			     Zip->fp);
      centraldirsize+=fwrite(Zip->CentralDirEntries[i]->FileName,sizeof(char),
			     Zip->CentralDirEntries[i]->Const->FileNameLen,
			     Zip->fp);
      centraldirsize+=fwrite(Zip->CentralDirEntries[i]->ExtraField,sizeof(char),
			      Zip->CentralDirEntries[i]->Const->ExtraFieldLen,
			      Zip->fp);
      centraldirsize+=fwrite(Zip->CentralDirEntries[i]->Comment,sizeof(char),
			     Zip->CentralDirEntries[i]->Const->CommentLen,
			     Zip->fp);
    };

  Zip->Size+=centraldirsize;
  Zip->CentralDirEnd=XREALLOC(Zip->CentralDirEnd,
			      s_ZipCentralDirEnd,
			      1);

  Zip->CentralDirEnd->Const=XREALLOC(Zip->CentralDirEnd->Const,
				     s_ZipCentralDirEndConst,
				     1);

  Zip->CentralDirEnd->Const->Sig=0x06054b50;
  Zip->CentralDirEnd->Const->DiskNb=0;
  Zip->CentralDirEnd->Const->CentralDirStartDisk=0;
  Zip->CentralDirEnd->Const->DiskEntriesNb=Zip->NbCentralDirEntries;
  Zip->CentralDirEnd->Const->EntriesNb=Zip->NbCentralDirEntries;
  Zip->CentralDirEnd->Const->CentralDirSize=centraldirsize;
  Zip->CentralDirEnd->Const->CentralDirStartOffset=offset;
  if(Comment!=NULL)
    {
      Zip->CentralDirEnd->Const->CommentLen=strlen(Comment);
      Zip->CentralDirEnd->Comment=XREALLOC(Zip->CentralDirEnd->Comment,
					   unsigned char,
					   Zip->CentralDirEnd->Const->CommentLen);

      strncpy(Zip->CentralDirEnd->Comment,Comment,strlen(Comment));
    }  
  else
    {
      Zip->CentralDirEnd->Const->CommentLen=0;
      Zip->CentralDirEnd->Comment=NULL;
    };

  zcdend=*Zip->CentralDirEnd->Const;
  FIX_ZCDEND_ENDIANNESS(zcdend);
  Zip->Size+=fwrite(&zcdend,1,sizeof(s_ZipCentralDirEndConst),Zip->fp);
  if(Zip->CentralDirEnd->Comment!=NULL)
    Zip->Size+=fwrite(Zip->CentralDirEnd->Comment,
		      sizeof(char),
		      Zip->CentralDirEnd->Const->CommentLen,
		      Zip->fp);

  return 0;
}
    

s_Zip *Zipfopen(const char *path,const char *mode)
{
  FILE *fp;
  s_Zip *Zip=NULL;
  s_ZipLocalFileHeader *zlfh;
  s_ZipCentralDirEntryConst *zcdec;
  unsigned int size;

  struct stat stat_buf;
  unsigned char *directory=NULL;
  unsigned char *p;

  myerrno=MENONE;

  if(strcmp(mode,"r")!=0 &&
     strcmp(mode,"r+")!=0 &&
     strcmp(mode,"w")!=0 &&
     strcmp(mode,"w+")!=0)
    {
      myerrno=MEINVAL;
      LPRINTF(DEBUG,"Zipfopen(%s,%s): %s is not a valid mode",
	      path,mode,mode);
      return NULL;
    };

  if(!is_file_accessible(path,NULL))
    {
      /* If no creation mode (r or r+) requested, no way ... */
      if(strncmp(mode,"r",1)==0)
	{
	  LPRINTF(DEBUG,"Zipfopen(%s,%s): %s",path,mode,strerror(errno));
	  return NULL;
	};

      /*... otherwise, let's verify the accessibility of parent dir */
      directory=XSTRDUP(path);
      p=directory+strlen(directory);
      while(p>=directory && *p!='/')
	p--;
      if(p>directory)
	*p=0;
      else
	{
	  XFREE(directory);
	  directory=XSTRDUP(".");
	};
      p=NULL;
      if(!is_file_writable(directory))
	{
	  LPRINTF(DEBUG,"Zipfopen(%s,%s): %s '%s' ",
		  path,mode,directory,strerror(errno));
	  myerrno=MEWACCESS;
	  XFREE(directory);
	  return NULL;
	};
      XFREE(directory);      
    }

  if(!is_file_readable(path) &&
     strncmp(mode,"r",1)==0)
    {
      LPRINTF(DEBUG,"Zipfopen(%s,%s): reading access denied",
	      path,mode);
      myerrno=MERACCESS;
      return NULL;
    }
  
  if(!is_file_writable(path) &&
     strncmp(mode,"w",1)==0)
    {
      LPRINTF(DEBUG,"Zipfopen(%s,%s): writing access denied",
	      path);
      myerrno=MEWACCESS;
      return NULL;
    }
  
  if(!(fp=fopen(path,mode)))
    {
      myerrno=MEUNKNOWN;
      return NULL;
    };

  zlfh=XCALLOC(s_ZipLocalFileHeader,1);

  Zip=XCALLOC(s_Zip,1);

  Zip->fp=fp;
  size=fread(zlfh,1,sizeof(s_ZipLocalFileHeader),fp);
  FIX_ZLFH_ENDIANNESS(*zlfh);
  Zip->Size+=size;
  
  while(size!=0 && zlfh->Sig==0x04034b50)
    {
      fseek(fp,zlfh->FileNameLen+zlfh->ExtraFieldLen,SEEK_CUR); 
      Zip->Size+=zlfh->FileNameLen+zlfh->ExtraFieldLen;
      /*      Zip->Size+=size;*/
      if(zlfh->GenFlag & 8)
	{
	  s_ZipLocalFileDataDesc zlfdd;
	  unsigned int sig=0;
	  size=ftell(fp);
	  while(1)
	    {
	      while(0x50!=fgetc(fp)); /* 0x50=beginning of local header sig */
	      fseek(fp,-1,SEEK_CUR);
	      fread(&sig,1,4,fp);
	      sig=LE2ME_32(sig);
	      if(sig==0x08074b50)
		break;
	      fseek(fp,-3,SEEK_CUR);
	    }
	  fseek(fp,-4,SEEK_CUR);
	  size=ftell(fp)-size;
	  Zip->Size+=size;
	  fread(&zlfdd,1,sizeof(s_ZipLocalFileDataDesc),fp);
	  FIX_ZLFDD_ENDIANNESS(zlfdd);
	  Zip->Size+=sizeof(s_ZipLocalFileDataDesc);
	  if(size!=zlfdd.CompSize)
	    {
	      /* TODO: Handle the error */
	      LPRINTF(WARNING,"Compressed size value for entry does not fit to the size of read data");
	      return NULL;
	    };
	}
      else
	{
	  fseek(fp,zlfh->CompSize,SEEK_CUR);
	  Zip->Size+=zlfh->CompSize;
	};
      size=fread(zlfh,1,sizeof(s_ZipLocalFileHeader),fp);
      FIX_ZLFH_ENDIANNESS(*zlfh);
      Zip->Size+=size;
    }
  XFREE(zlfh);

  if(strncmp(mode,"w",1)!=0 && (size==0 || feof(fp)))
    {
      fclose(fp);
      FreeZip(Zip);
      return NULL;
    };

  fseek(fp,sizeof(s_ZipLocalFileHeader)*(-1),SEEK_CUR);
  Zip->Size-=sizeof(s_ZipLocalFileHeader);

  zcdec=XCALLOC(s_ZipCentralDirEntryConst,1);

  size=fread(zcdec,1,sizeof(s_ZipCentralDirEntryConst),fp);
  FIX_ZCDENTRY_ENDIANNESS(*zcdec);

  Zip->Size+=size;
  
  while(size!=0 && zcdec->Sig==0x02014b50)
    {
      Zip->CentralDirEntries=XREALLOC(Zip->CentralDirEntries,
				      s_ZipCentralDirEntry*,
				      Zip->NbCentralDirEntries+1);

      Zip->CentralDirEntries[Zip->NbCentralDirEntries]=
	XCALLOC(s_ZipCentralDirEntry,1);

      Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const=zcdec;

      Zip->CentralDirEntries[Zip->NbCentralDirEntries]->FileName=
	XCALLOC(unsigned char,zcdec->FileNameLen);

      size=fread(Zip->CentralDirEntries[Zip->NbCentralDirEntries]->FileName,
		 sizeof(char),
		 zcdec->FileNameLen,
		 fp);
      Zip->Size+=size;

      Zip->CentralDirEntries[Zip->NbCentralDirEntries]->ExtraField=
        XCALLOC(unsigned char,zcdec->ExtraFieldLen);

      size=fread(Zip->CentralDirEntries[Zip->NbCentralDirEntries]->ExtraField,
		 sizeof(char),
		 zcdec->ExtraFieldLen,
		 fp);
      Zip->Size+=size;

      Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Comment=
	XCALLOC(unsigned char,zcdec->CommentLen);

      size=fread(Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Comment,
		 sizeof(char),
		 zcdec->CommentLen,
		 fp);
      Zip->Size+=size;

      Zip->NbCentralDirEntries++;

      zcdec=XCALLOC(s_ZipCentralDirEntryConst,1);

      size=fread(zcdec,1,sizeof(s_ZipCentralDirEntryConst),fp); 
      FIX_ZCDENTRY_ENDIANNESS(*zcdec);
      Zip->Size+=size;
    };
  XFREE(zcdec);

  if(strncmp(mode,"w",1)!=0 && size==0)
    {
      fclose(fp);
      FreeZip(Zip);
      return NULL;
    };

  fseek(fp,size*(-1),SEEK_CUR);
  Zip->Size-=size;
 
  Zip->CentralDirEnd=XCALLOC(s_ZipCentralDirEnd,1);

  Zip->CentralDirEnd->Const=XCALLOC(s_ZipCentralDirEndConst,1);

  size=fread(Zip->CentralDirEnd->Const,1,sizeof(s_ZipCentralDirEndConst),fp);
  FIX_ZCDEND_ENDIANNESS(*Zip->CentralDirEnd->Const);

  Zip->Size+=size;
  
  if(size==sizeof(s_ZipCentralDirEndConst) &&
     Zip->CentralDirEnd->Const->Sig==0x6054b50)
    {
      if(Zip->CentralDirEnd->Const->CommentLen!=0)
	{
	  Zip->CentralDirEnd->Comment=
	    XCALLOC(unsigned char,
		    Zip->CentralDirEnd->Const->CommentLen);
	  
	  Zip->Size+=fread(Zip->CentralDirEnd->Comment,
			   sizeof(char),
			   Zip->CentralDirEnd->Const->CommentLen,
			   fp);
	}
      else
	Zip->CentralDirEnd->Comment=NULL;
    }
  else
    Zip->CentralDirEnd->Const->Sig=0;

  if (strncmp(mode,"w",1)!=0 &&
      Zip->NbFileEntries==0 &&
      Zip->NbCentralDirEntries==0 &&
      Zip->CentralDirEnd->Const->Sig==0)
    {
      Zipfclose(Zip);
      return NULL;
    };

  rewind(Zip->fp);
  return Zip;
};

s_Zip *InitVirtualZip()
{
  s_Zip* Zip;
  
  Zip=XCALLOC(s_Zip,1);

  Zip->CentralDirEnd=XCALLOC(s_ZipCentralDirEnd,1);

  return Zip;
};  
  
void AddToVirtualZip(s_Zip *Zip,
		     unsigned char *File,
		     unsigned int CompSize)
{
  Zip->CentralDirEntries=XREALLOC(Zip->CentralDirEntries,
				  s_ZipCentralDirEntry*,
				  Zip->NbCentralDirEntries+1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]=
    XCALLOC(s_ZipCentralDirEntry,1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const=
    XCALLOC(s_ZipCentralDirEntryConst,1);

  Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const->CompSize=CompSize;
  Zip->CentralDirEntries[Zip->NbCentralDirEntries]->Const->FileNameLen=strlen(File);
  Zip->Size+=sizeof(s_ZipLocalFileHeader)+strlen(File)+CompSize+
    sizeof(s_ZipCentralDirEntryConst)+ strlen(File);

  Zip->NbCentralDirEntries++;
  
};

unsigned int GetVirtualZipSize(s_Zip *Zip)
{
  if(Zip->Size!=0)
    return (Zip->Size+sizeof(s_ZipCentralDirEndConst));
  else
    return 0;
};

/*
void repair(s_Zip *Zip)
{
  s_ZipLocalFileHeader *zlfh;
  unsigned int *startoffset;
  unsigned int i=0;

  startoffset=calloc(Zip->NbCentralDirEntries+1,sizeof(unsigned int));
  SECURE_MALLOC(startoffset);

  zlfh=calloc(1,sizeof(s_ZipLocalFileHeader));
  SECURE_MALLOC(zlfh);

  startoffset[0]=0;
  fread(zlfh,sizeof(s_ZipLocalFileHeader),1,Zip->fp);
  FIX_ZLFH_ENDIANNESS(*zlfh);
  
  while(zlfh->Sig==0x04034b50)
    {
      i++;
      fseek(Zip->fp,
	    zlfh->FileNameLen+zlfh->ExtraFieldLen+zlfh->CompSize,
	    SEEK_CUR); 
      if(zlfh->GenFlag & 8)
	fseek(Zip->fp,sizeof(s_ZipLocalFileDataDesc),SEEK_CUR);	
      startoffset[i]=ftell(Zip->fp);
      fread(zlfh,sizeof(s_ZipLocalFileHeader),1,Zip->fp);
      FIX_ZLFH_ENDIANNESS(*zlfh);
    }
  XFREE(zlfh);

  for(i=0;i<Zip->NbCentralDirEntries;i++)
    Zip->CentralDirEntries[i]->Const->LocalHeaderOffset=startoffset[i];
  XFREE(startoffset);

};
*/
  
