"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;

var _configSchema = require("@osd/config-schema");

var mime = _interopRequireWildcard(require("mime"));

var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));

var _shared = require("../../../common/constants/shared");

var _integrations_manager = require("../../adaptors/integrations/integrations_manager");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsAdaptor` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsAdaptor} adaptor The adaptor instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (adaptor, response, callback) => {
  try {
    const data = await callback(adaptor);
    return response.ok({
      body: {
        data
      }
    });
  } catch (err) {
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};

exports.handleWithCallback = handleWithCallback;

const getAdaptor = (context, _request) => {
  return new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
};

function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        dataSource: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.dataSource);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);

    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return response.ok({
        headers: {
          'Content-Type': mime.getType(request.params.path)
        },
        body: result
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = getAdaptor(context, request);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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