"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.VisualizationsRouter = VisualizationsRouter;

var _configSchema = require("@osd/config-schema");

var _custom_panel_adaptor = require("../../adaptors/custom_panels/custom_panel_adaptor");

var _custom_panels = require("../../../common/constants/custom_panels");

/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
function VisualizationsRouter(router) {
  // Fetch all the savedVisualzations
  const customPanelBackend = new _custom_panel_adaptor.CustomPanelsAdaptor();
  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {}
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const visualizationList = await customPanelBackend.getAllSavedVisualizations(opensearchNotebooksClient);
      return response.ok({
        body: {
          visualizations: visualizationList
        }
      });
    } catch (error) {
      console.error('Issue in fetching saved visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // get all saved visualizations by Id

  router.get({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/{savedVisualizationId}`,
    validate: {
      params: _configSchema.schema.object({
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const savedVisualization = await customPanelBackend.getSavedVisualizationById(opensearchNotebooksClient, request.params.savedVisualizationId);
      return response.ok({
        body: {
          visualization: savedVisualization
        }
      });
    } catch (error) {
      console.error('Issue in fetching saved visualizations by ids:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add a new visualization to the panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Added',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Add multiple visualizations to the panel

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/multiple`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationIds: _configSchema.schema.arrayOf(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const allVisualizations = await customPanelBackend.addMultipleVisualizations(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationIds);
      return response.ok({
        body: {
          message: 'Visualizations Added',
          visualizations: allVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in adding visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // Replace an existing visualization

  router.post({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/replace`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        savedVisualizationId: _configSchema.schema.string(),
        oldVisualizationId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.addVisualization(opensearchNotebooksClient, request.body.panelId, request.body.savedVisualizationId, request.body.oldVisualizationId);
      return response.ok({
        body: {
          message: 'Visualization Replaced',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in replacing visualization:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  }); // changes the position of the mentioned visualizations
  // Also removes the visualizations not mentioned

  router.put({
    path: `${_custom_panels.CUSTOM_PANELS_API_PREFIX}/visualizations/edit`,
    validate: {
      body: _configSchema.schema.object({
        panelId: _configSchema.schema.string(),
        visualizationParams: _configSchema.schema.arrayOf(_configSchema.schema.object({
          i: _configSchema.schema.string(),
          x: _configSchema.schema.number(),
          y: _configSchema.schema.number(),
          w: _configSchema.schema.number(),
          h: _configSchema.schema.number()
        }))
      })
    }
  }, async (context, request, response) => {
    const opensearchNotebooksClient = context.observability_plugin.observabilityClient.asScoped(request);

    try {
      const newVisualizations = await customPanelBackend.editVisualization(opensearchNotebooksClient, request.body.panelId, request.body.visualizationParams);
      return response.ok({
        body: {
          message: 'Visualizations Edited',
          visualizations: newVisualizations
        }
      });
    } catch (error) {
      console.error('Issue in Editing visualizations:', error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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