"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MetricsService = void 0;

var _ = require(".");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; } /*
                                                                                                                                                                                                                   * Copyright OpenSearch Contributors
                                                                                                                                                                                                                   * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                   */

class MetricsService {
  constructor(logger) {
    this.logger = logger;

    _defineProperty(this, "interval", _.METRIC_INTERVAL.ONE_MINUTE);

    _defineProperty(this, "windowSize", _.DEFAULT_WINDOW_SIZE);

    _defineProperty(this, "data", {});

    _defineProperty(this, "componentCounts", {});

    _defineProperty(this, "statusCodeCounts", {});

    _defineProperty(this, "overall", {});
  }

  setup(interval = _.METRIC_INTERVAL.ONE_MINUTE, windowSize = _.DEFAULT_WINDOW_SIZE) {
    this.interval = interval;
    this.windowSize = windowSize;

    const addMetric = (componentName, action, statusCode, value) => {
      const currInterval = Math.floor(Date.now() / this.interval);
      this.trim();

      if (!this.data[currInterval]) {
        this.data[currInterval] = {};
        this.overall[currInterval] = {
          response_time_total: 0,
          count: 0
        };
        this.componentCounts[currInterval] = {};
        this.statusCodeCounts[currInterval] = {};
      }

      if (!this.data[currInterval][componentName]) {
        this.data[currInterval][componentName] = {};
        this.componentCounts[currInterval][componentName] = 0;
      }

      if (!this.statusCodeCounts[currInterval][statusCode]) {
        this.statusCodeCounts[currInterval][statusCode] = 0;
      }

      if (!this.data[currInterval][componentName][action]) {
        this.data[currInterval][componentName][action] = {};
      }

      if (!this.data[currInterval][componentName][action][statusCode]) {
        this.data[currInterval][componentName][action][statusCode] = {
          response_time_total: 0,
          count: 0
        };
      }

      const {
        response_time_total,
        count
      } = this.data[currInterval][componentName][action][statusCode];
      this.data[currInterval][componentName][action][statusCode] = {
        response_time_total: response_time_total + value,
        count: count + 1
      };
      this.componentCounts[currInterval][componentName]++;
      this.statusCodeCounts[currInterval][statusCode]++;
      this.overall[currInterval].response_time_total += value;
      this.overall[currInterval].count++;
    };

    const getStats = () => {
      const prevInterval = Math.floor(Date.now() / this.interval) - 1;
      const data = { ...this.data[prevInterval]
      } || {};
      const overall = { ...this.overall[prevInterval]
      } || {};
      let requestsPerSecond = 0,
          responseTimeAvg = 0;

      if (Object.keys(overall).length !== 0 && overall.count !== 0) {
        responseTimeAvg = overall.response_time_total / overall.count;
        requestsPerSecond = overall.count / (this.interval / 1000);
      }

      return {
        data,
        overall: {
          response_time_avg: responseTimeAvg,
          requests_per_second: requestsPerSecond
        },
        counts_by_component: { ...this.componentCounts[prevInterval]
        } || {},
        counts_by_status_code: { ...this.statusCodeCounts[prevInterval]
        } || {}
      };
    };

    return {
      addMetric,
      getStats
    };
  }

  start() {}

  stop() {
    this.resetMetrics();
  }

  resetMetrics() {
    this.data = {};
    this.overall = {};
    this.componentCounts = {};
    this.statusCodeCounts = {};
  }

  trim() {
    const oldestTimestampToKeep = Math.floor((Date.now() - this.windowSize * this.interval) / this.interval);

    for (const timestampStr in this.data) {
      const timestamp = parseInt(timestampStr);

      if (timestamp < oldestTimestampToKeep) {
        delete this.data[timestamp];
      }
    }
  }

}

exports.MetricsService = MetricsService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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