"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.defineWlmRoutes = defineWlmRoutes;
var _configSchema = require("@osd/config-schema");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function defineWlmRoutes(router, dataSourceEnabled) {
  // Get WLM stats across all nodes in the cluster
  router.get({
    path: '/api/_wlm/stats',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query;
      let stats;
      if (!dataSourceEnabled || !((_request$query = request.query) !== null && _request$query !== void 0 && _request$query.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        stats = await client('wlm.getStats');
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        stats = await client.callAPI('wlm.getStats', {});
      }
      return response.ok({
        body: stats
      });
    } catch (error) {
      context.queryInsights.logger.error(`Failed to fetch WLM stats: ${error.message}`, {
        error
      });
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch WLM stats: ${error.message}`
        }
      });
    }
  });

  // Get WLM stats for a specific node
  router.get({
    path: '/api/_wlm/{nodeId}/stats',
    validate: {
      params: _configSchema.schema.object({
        nodeId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query2;
      const {
        nodeId
      } = request.params;
      let stats;
      if (!dataSourceEnabled || !((_request$query2 = request.query) !== null && _request$query2 !== void 0 && _request$query2.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        stats = await client('wlm.getNodeStats', {
          nodeId
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        stats = await client.callAPI('wlm.getNodeStats', {
          nodeId
        });
      }
      return response.ok({
        body: stats
      });
    } catch (error) {
      console.error(`Failed to fetch stats for node ${request.params.nodeId}:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch stats for node ${request.params.nodeId}: ${error.message}`
        }
      });
    }
  });

  // List all workload groups
  router.get({
    path: '/api/_wlm/workload_group',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query3;
      let result;
      if (!dataSourceEnabled || !((_request$query3 = request.query) !== null && _request$query3 !== void 0 && _request$query3.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.getWorkloadGroups');
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.getWorkloadGroups', {});
      }
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.customError({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch workload groups: ${error.message}`
        }
      });
    }
  });

  // Get workload group by name
  router.get({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const {
      name
    } = request.params;
    try {
      var _request$query4;
      let result;
      if (!dataSourceEnabled || !((_request$query4 = request.query) !== null && _request$query4 !== void 0 && _request$query4.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.getWorkloadGroup', {
          name
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.getWorkloadGroup', {
          name
        });
      }
      return response.ok({
        body: result
      });
    } catch (error) {
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Error fetching workload group: ${error.message}`
        }
      });
    }
  });

  // Create a new workload group
  router.put({
    path: '/api/_wlm/workload_group',
    validate: {
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        resiliency_mode: _configSchema.schema.string(),
        resource_limits: _configSchema.schema.object({
          cpu: _configSchema.schema.maybe(_configSchema.schema.number()),
          memory: _configSchema.schema.maybe(_configSchema.schema.number())
        })
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query5;
      const body = request.body;
      let result;
      if (!dataSourceEnabled || !((_request$query5 = request.query) !== null && _request$query5 !== void 0 && _request$query5.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.createWorkloadGroup', {
          body
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.createWorkloadGroup', {
          body
        });
      }
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to create workload group:', e);
      return response.internalError({
        body: {
          message: `Failed to create workload group: ${e.message}`
        }
      });
    }
  });

  // Update a workload group by name
  router.put({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        resiliency_mode: _configSchema.schema.string(),
        resource_limits: _configSchema.schema.object({
          cpu: _configSchema.schema.maybe(_configSchema.schema.number()),
          memory: _configSchema.schema.maybe(_configSchema.schema.number())
        })
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query6;
      const {
        name
      } = request.params;
      const body = request.body;
      let result;
      if (!dataSourceEnabled || !((_request$query6 = request.query) !== null && _request$query6 !== void 0 && _request$query6.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.updateWorkloadGroup', {
          name,
          body
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.updateWorkloadGroup', {
          name,
          body
        });
      }
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to update workload group:', e);
      return response.internalError({
        body: {
          message: `Failed to update workload group: ${e.message}`
        }
      });
    }
  });

  // Delete a workload group by name
  router.delete({
    path: '/api/_wlm/workload_group/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query7;
      const {
        name
      } = request.params;
      let result;
      if (!dataSourceEnabled || !((_request$query7 = request.query) !== null && _request$query7 !== void 0 && _request$query7.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.deleteWorkloadGroup', {
          name
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.deleteWorkloadGroup', {
          name
        });
      }
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error(`Failed to delete workload group '${request.params.name}':`, e);
      return response.internalError({
        body: {
          message: `Failed to delete workload group '${request.params.name}': ${e.message}`
        }
      });
    }
  });

  // Get stats for a specific workload group
  router.get({
    path: '/api/_wlm/stats/{workloadGroupId}',
    validate: {
      params: _configSchema.schema.object({
        workloadGroupId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query8;
      const {
        workloadGroupId
      } = request.params;
      let result;
      if (!dataSourceEnabled || !((_request$query8 = request.query) !== null && _request$query8 !== void 0 && _request$query8.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.getWorkloadGroupStats', {
          workloadGroupId
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.getWorkloadGroupStats', {
          workloadGroupId
        });
      }
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error(`Failed to fetch WLM stats for group ${request.params.workloadGroupId}:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to fetch WLM stats for group ${request.params.workloadGroupId}: ${error.message}`
        }
      });
    }
  });

  // Create index rule
  router.put({
    path: '/api/_rules/workload_group',
    validate: {
      body: _configSchema.schema.object({
        description: _configSchema.schema.string(),
        principal: _configSchema.schema.maybe(_configSchema.schema.object({
          username: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          role: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
        })),
        index_pattern: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        workload_group: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query9;
      const body = request.body;
      let result;
      if (!dataSourceEnabled || !((_request$query9 = request.query) !== null && _request$query9 !== void 0 && _request$query9.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.createRule', {
          body
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.createRule', {
          body
        });
      }
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error(`Failed to create rule:`, error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: {
          message: `Failed to create rule: ${error.message}`
        }
      });
    }
  });

  // Get all index rules
  router.get({
    path: '/api/_rules/workload_group',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query10;
      let result;
      if (!dataSourceEnabled || !((_request$query10 = request.query) !== null && _request$query10 !== void 0 && _request$query10.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.getRules');
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.getRules', {});
      }
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error('Failed to fetch rules:', e);
      return response.internalError({
        body: {
          message: `Failed to fetch rules: ${e.message}`
        }
      });
    }
  });

  // Delete index rule by ID
  router.delete({
    path: '/api/_rules/workload_group/{ruleId}',
    validate: {
      params: _configSchema.schema.object({
        ruleId: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const {
      ruleId
    } = request.params;
    try {
      var _request$query11;
      let result;
      if (!dataSourceEnabled || !((_request$query11 = request.query) !== null && _request$query11 !== void 0 && _request$query11.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.deleteRule', {
          ruleId
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.deleteRule', {
          ruleId
        });
      }
      return response.ok({
        body: result
      });
    } catch (e) {
      console.error(`Failed to delete index rule ${ruleId}:`, e);
      return response.internalError({
        body: {
          message: `Failed to delete index rule ${ruleId}: ${e.message}`
        }
      });
    }
  });

  // Update index rule
  router.put({
    path: '/api/_rules/workload_group/{ruleId}',
    validate: {
      params: _configSchema.schema.object({
        ruleId: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        description: _configSchema.schema.string(),
        principal: _configSchema.schema.maybe(_configSchema.schema.object({
          username: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
          role: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string()))
        })),
        index_pattern: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        workload_group: _configSchema.schema.string()
      }),
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const {
      ruleId
    } = request.params;
    const body = request.body;
    try {
      var _request$query12;
      let result;
      if (!dataSourceEnabled || !((_request$query12 = request.query) !== null && _request$query12 !== void 0 && _request$query12.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        result = await client('wlm.updateRule', {
          ruleId,
          body
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        result = await client.callAPI('wlm.updateRule', {
          ruleId,
          body
        });
      }
      return response.ok({
        body: result
      });
    } catch (error) {
      console.error('Error updating rule:', error);
      return response.customError({
        body: error.message || error,
        statusCode: error.statusCode || 500
      });
    }
  });

  // Get node level cpu and memory threshold
  router.get({
    path: '/api/_wlm/thresholds',
    validate: {
      query: _configSchema.schema.object({
        dataSourceId: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    try {
      var _request$query13, _ref, _ref2, _body$transient$wlm$w, _body$transient, _body$persistent, _body$defaults, _ref3, _ref4, _body$transient$wlm$w2, _body$transient2, _body$persistent2, _body$defaults2;
      let body;
      if (!dataSourceEnabled || !((_request$query13 = request.query) !== null && _request$query13 !== void 0 && _request$query13.dataSourceId)) {
        const client = context.wlm_plugin.wlmClient.asScoped(request).callAsCurrentUser;
        body = await client('wlm.getThresholds', {
          include_defaults: true
        });
      } else {
        const client = context.dataSource.opensearch.legacy.getClient(request.query.dataSourceId);
        body = await client.callAPI('wlm.getThresholds', {
          include_defaults: true
        });
      }
      const cpuThreshold = (_ref = (_ref2 = (_body$transient$wlm$w = (_body$transient = body.transient) === null || _body$transient === void 0 || (_body$transient = _body$transient.wlm) === null || _body$transient === void 0 || (_body$transient = _body$transient.workload_group) === null || _body$transient === void 0 || (_body$transient = _body$transient.node) === null || _body$transient === void 0 ? void 0 : _body$transient.cpu_rejection_threshold) !== null && _body$transient$wlm$w !== void 0 ? _body$transient$wlm$w : (_body$persistent = body.persistent) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.wlm) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.workload_group) === null || _body$persistent === void 0 || (_body$persistent = _body$persistent.node) === null || _body$persistent === void 0 ? void 0 : _body$persistent.cpu_rejection_threshold) !== null && _ref2 !== void 0 ? _ref2 : (_body$defaults = body.defaults) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.wlm) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.workload_group) === null || _body$defaults === void 0 || (_body$defaults = _body$defaults.node) === null || _body$defaults === void 0 ? void 0 : _body$defaults.cpu_rejection_threshold) !== null && _ref !== void 0 ? _ref : '1';
      const memoryThreshold = (_ref3 = (_ref4 = (_body$transient$wlm$w2 = (_body$transient2 = body.transient) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.wlm) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.workload_group) === null || _body$transient2 === void 0 || (_body$transient2 = _body$transient2.node) === null || _body$transient2 === void 0 ? void 0 : _body$transient2.memory_rejection_threshold) !== null && _body$transient$wlm$w2 !== void 0 ? _body$transient$wlm$w2 : (_body$persistent2 = body.persistent) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.wlm) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.workload_group) === null || _body$persistent2 === void 0 || (_body$persistent2 = _body$persistent2.node) === null || _body$persistent2 === void 0 ? void 0 : _body$persistent2.memory_rejection_threshold) !== null && _ref4 !== void 0 ? _ref4 : (_body$defaults2 = body.defaults) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.wlm) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.workload_group) === null || _body$defaults2 === void 0 || (_body$defaults2 = _body$defaults2.node) === null || _body$defaults2 === void 0 ? void 0 : _body$defaults2.memory_rejection_threshold) !== null && _ref3 !== void 0 ? _ref3 : '1';
      return response.ok({
        body: {
          cpuRejectionThreshold: parseFloat(cpuThreshold),
          memoryRejectionThreshold: parseFloat(memoryThreshold)
        }
      });
    } catch (e) {
      var _ref5, _ref6, _e$meta$statusCode, _e$meta, _ref7, _ref8, _ref9, _e$meta$body$error$re, _e$meta2, _e$meta3, _e$body;
      const status = (_ref5 = (_ref6 = (_e$meta$statusCode = e === null || e === void 0 || (_e$meta = e.meta) === null || _e$meta === void 0 ? void 0 : _e$meta.statusCode) !== null && _e$meta$statusCode !== void 0 ? _e$meta$statusCode : e === null || e === void 0 ? void 0 : e.statusCode) !== null && _ref6 !== void 0 ? _ref6 : e === null || e === void 0 ? void 0 : e.status) !== null && _ref5 !== void 0 ? _ref5 : 500;
      const message = (_ref7 = (_ref8 = (_ref9 = (_e$meta$body$error$re = e === null || e === void 0 || (_e$meta2 = e.meta) === null || _e$meta2 === void 0 || (_e$meta2 = _e$meta2.body) === null || _e$meta2 === void 0 || (_e$meta2 = _e$meta2.error) === null || _e$meta2 === void 0 ? void 0 : _e$meta2.reason) !== null && _e$meta$body$error$re !== void 0 ? _e$meta$body$error$re : e === null || e === void 0 || (_e$meta3 = e.meta) === null || _e$meta3 === void 0 || (_e$meta3 = _e$meta3.body) === null || _e$meta3 === void 0 ? void 0 : _e$meta3.message) !== null && _ref9 !== void 0 ? _ref9 : e === null || e === void 0 || (_e$body = e.body) === null || _e$body === void 0 ? void 0 : _e$body.message) !== null && _ref8 !== void 0 ? _ref8 : e === null || e === void 0 ? void 0 : e.message) !== null && _ref7 !== void 0 ? _ref7 : 'Unexpected error';
      return response.customError({
        statusCode: status,
        body: {
          message
        }
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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