#!/bin/sh
# The next line restarts using wish \
exec /usr/local/bin/wish8.6 "$0" ${1+"$@"}

package require msgcat
::msgcat::mcload [file join [file dirname [info script]] msgs]
catch {namespace import ::msgcat::mc}

#
# Make the Image format available.
#

set rundir [file dirname [info script]]
set auto_path "[list $rundir] $auto_path"
package require img::raw
package require img::dted
package require img::gif
package require img::ps
set img_version [package require Img]

wm title . [mc {Tests for available image formats}]
eval destroy [winfo children .]
eval image delete [image names]

bind . <Key-Escape> "destroy ."

frame .f
frame .img
pack .f .img -side top -expand true -fill x

button .f.v -text [mc Viewer] -command start_viewer
pack .f.v -side left -expand true -fill y

button .f.b -text [mc "More tests"] -command run_more_tests
pack .f.b -side right -expand true -fill y

message .f.m -aspect 1000 -text [format [mc "This page shows the available\
image formats of the Img extension (version %s, using Tcl/Tk %s)"] $img_version $tcl_patchLevel]
pack .f.m -side left -expand true -fill x

proc run_more_tests {} {
    global rundir
    if {[winfo exists .t]} {
	raise .t
    } else {
	toplevel .t
	wm title .t [mc {More tests}]
	frame .t.f
	pack .t.f -side top -fill both
	text .t.t -yscrollcommand [list .t.s set]
	scrollbar .t.s -command [list .t.t yview]
	pack .t.t -side left -expand true -fill both
	pack .t.s -side left -expand false -fill y
    }
    .t.t delete 1.0 end
    .t.t insert end "-------- [mc {starting tests}] --------\n"
    set olddir [pwd]
    cd [file join $rundir ".."]
    proc puts args {
        # Skip optional "-nonewline" and "channelId"
	.t.t insert end "[lindex $args end]\n"
	.t.t see end
	update
    }
    proc exit args {
    }
    set result [catch {uplevel #0 source all.tcl}]
    cd $olddir
    if $result return
    .t.t insert end "-------- [mc {end of tests}] --------\n"
    .t.t see end
}

proc start_viewer {} {
    global tkvPriv rundir
    set argv {{} .viewer}
    source [file join $rundir tkv.tcl]
    foreach i [lsort [image names]] {
	register_image .viewer $i
    }
}

set curRow 1
set curCol 0
grid columnconfigure .img 1 -weight 1

proc update_animated_gif {w option num} {
    global imgPriv
    if {[info exists imgPriv($w,data)]} {
	set im [image create photo "${w}_[expr $num+1]"]
	$im copy $imgPriv($w,$num)
	incr num
	if {[catch {$im configure -data $imgPriv($w,data) -format [list gif -index $num]}]} {
	    unset imgPriv($w,data)
	    if {$num>1} {
		set num 0
		set im $imgPriv($w,$num)
	    } else {
		# this is not an animated GIF; just stop
		unset imgPriv($w,0)
		return
	    }
	} else {
	    $im configure -data {} -file {};#free storage
	    set imgPriv($w,$num) $im
	}
    } else {
	incr num
	if {![info exists imgPriv($w,$num)]} {
	    set num 0
	}
	set im $imgPriv($w,$num)
    }
    if {[catch "$w $option $im"]} {
	catch {unset imgPriv($w,data)}
	foreach im [array names imgPriv $w,*] {
	    image delete $imgPriv($im)
	    unset imgPriv($im)
	}
    } else {
	update idletasks
	after 20 [list update_animated_gif $w $option $num]
    }
}

proc show_animated_gif {w data} {
    global imgPriv
    catch {destroy $w}

    label $w
    if {[catch {image create photo ${w}_0 -data [binary decode base64 $data] -format [list gif -index 0]} image]} {
	$w configure -text [mc "error displaying animated gif"]:\n$image" \
		-relief ridge
    } else {
	$image configure -data {} -file {} -format {};# free storage
	$w configure -image $image
	set imgPriv($w,0) $image
	set imgPriv($w,data) $data
	update idletasks
	after 20 [list update_animated_gif $w {configure -image} 0]
    }
    grid $w -row 0 -column 0 -columnspan 4 -sticky ew
    toolhelpAddBinding $w "Animated GIF"
    return $w
}

proc toolhelpInit { w { bgColor yellow } { fgColor black } } {
    global topWidget

    # Create Toolbar help window with a simple label in it.
    if { [winfo exists $w] } {
	destroy $w
    }
    toplevel $w
    set topWidget $w
    label $w.l -text "This is toolhelp" -bg $bgColor -fg $fgColor -relief ridge
    pack $w.l
    wm overrideredirect $w true
    catch  {
        ::tk::unsupported::MacWindowStyle style $w help none
    }
    wm geometry $w [format "+%d+%d" -100 -100]
}

proc toolhelpShow { x y str } {
    global topWidget

    $topWidget.l configure -text $str
    raise $topWidget
    wm geometry $topWidget [format "+%d+%d" $x [expr $y +10]]
}

proc toolhelpHide {} {
    global topWidget

    wm geometry $topWidget [format "+%d+%d" -100 -100]
}

proc toolhelpAddBinding { w str } {
    global topWidget

    if { ![info exists topWidget]} {
	toolhelpInit .poToolhelp
    }
    bind $w <Enter>  "toolhelpShow %X %Y [list $str]"
    bind $w <Leave>  "toolhelpHide"
    bind $w <Button> "toolhelpHide"
}

proc show_image {format decode type data} {
    if {![winfo exists .img.$format]} {
	label .img.l_$format -text "[string toupper ${format}]: "
	frame .img.$format
	grid .img.l_$format -row $::curRow -column $::curCol -sticky w
	grid .img.$format   -row $::curRow -column [expr $::curCol+1] -sticky ew
        incr ::curCol 2
        if { $::curCol > 2 } {
            set ::curCol 0
            incr ::curRow
        }
    }
    set w .img.$format.[string tolower $type]
    catch {destroy $w}
    if { $decode eq "base64" } {
        set data [binary decode base64 $data]
    }
    if {[catch {image create photo [list $format $type] -data $data} image]} {
	message $w -aspect 250 -bd 2 -relief ridge \
	    -text "[format [mc {error displaying %s image}] $type]: $image"
    } else {
	$image configure -data {}
	label $w -image $image
        set width  [image width $image]
        set height [image height $image]
	toolhelpAddBinding $w "[string toupper $format] : $type ($width x $height pixels)"
    }
    pack $w -side left
    update
}

show_animated_gif .img.mpph {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=}

show_image bmp base64 1-bit {
Qk3OAQAAAAAAAD4AAAAoAAAAIgAAADIAAAABAAEAAAAAAJABAABtCwAAbQsAAAIAAAACAAAA
MzPMAMwzMwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAZmTKAAAAAABIRIwAAAAAAEZAigAAAAAAQECIAAAAAAAAAAgAAAAAAAAEAA
AAAAAAABQAAAAAAAAAPAAAAAAAAAB+AAAAAAAAAC4AAAAAAAAALgAAAAAAAAAsAAAAAAAAAC
wAAAAAAAAALgAAAAAAAABuAAAAAAAAAG4AAAAAAAAAbgAAAAAAAABuAAAAAAAAAG4AAAAAAA
AAfgAAAAAAAAA+AAAAAAAAADcAAAAAAAAAPwAAAAAAAAA/AAAAAAAAAD+AAAAAAAAAP4AAAA
AAAAA/gAAAAAAAAD4AAAAAAAAAHgAAAAAAAAAfAAAAAAAAAA+AAAAAAAAAB8AAAAAAAAADgA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA}

show_image bmp base64 4-bit {
Qk1aBAAAAAAAAHIAAAAoAAAAIgAAADIAAAABAAQAAAAAAOgDAABtCwAAbQsAAA8AAAAPAAAA
////AAD//wDMzP8AmZn/AGaZ/wAzM8wAmZnMAGaZzAAzZswAzMyZAJmZmQDMZjMAZjNmADMz
ZgBmMzMABWMgAAAAAAAAAAAAAAAAAAAAAAAFVVVVVVVVVVVVVVVVVVVVUAAAAAVVVVVVVVVV
VVVVVVVVVVVQAAAABVVVVVVVVVVVVVVVVVVVVVAAAAAFVVVVVVVVVVVVVVVVVVVVUAAAAAVV
VVVVVVVVVVVVVVVVVVWAAAAABVVVVVVVVVVVVVVVVVVVVVAAAAAFVVBVU3VSVVNVB1UFBVVV
UAAAAAVVUFU1VVNVVVUFVQBVVVVgAAAABVVQVVUFU1VVVQVVBVVVVTAAAAAIVVBVVVVTVXVV
BVUFVVVVMAAAAAVVVVVVVVFVFVVVVQVVVVUgAAAABVVVVVVVVVUFVVVVVVVVVSAAAAAHVVVV
VVVVW5tVVVVVVVVVAAAAAAZVVVVVVVW8u2VVVVVVVVUAAAAAA1VVVVVVW7W7tVVVVVVVVQAA
AAADVVVVVVWCtbuzVVVVVVVVAAAAAANVVVVVVVW1vrJVVVVVVVUAAAAAAlVVVVVVWLW7UFVV
VVVVVQAAAAACVVVVVVVatbtVVVVVVVVVAAAAAABVVVVVVVq3u8NVVVVVVVUAAAAAAFVVVVVV
XrS74VVVVVVVVQAAAAAAVVVVVVWLtbuyVVVVVVVVAAAAAABVVVVVVUu1u7BVVVVVVVUAAAAA
AFVVVVVVO7W7sFVVVVVVVQAAAAAAVVVVVVUbtbu6VVVVVVVVAAAAAABVVVVVVSu9u1VVVVVV
VVUAAAAAAFVVVVVVJb67tVVVVVVVVQAAAAAAVVVVVVVVu6u7VVVVVVVVAAAAAABVVVVVVVW7
u7slVVVVVVUAAAAAAFVVVVVVVbu7u1VVVVVVVQAAAAACVVVVVVVVu7u7wlRVVVVVAAAAAAJV
VVVVVVW7u7uwV1VVVVUAAAAAA1VVVVVVVbu7u7BVhVVVVSAAAAADVVVVVYVVu7vFAFUVVVVV
MAAAAAZVVVVVhVUbu7VVVRVXVVVgAAAAB1VVVVVVVVu7u1VVFVFVVVAAAAAFVVVVVVhVVLu7
xVVVVVVVgAAAAAVVVVVVUVVVK7u1VVVVVVVQAAAACFVVVVhRVVVQq7VVVVVVVVAAAAAFVVVV
VFV1VVUgBVVVVVVVUAAAAAVVVVVUVRVVVVVVVVVVVVVQAAAABVVVVViFUVVVVVVVVVVVVVAA
AAAFVVVVVRVUFVVVVVVVVVVVUAAAAAVVVVVVVVVVVVVVVVVVVVVQAAAABVVVVVVVVVVVVVVV
VVVVVVIAAAAFVVVVVVVVVVVVVVVVVVVVUQAAAAVVVVVVVVVVVVVVVTdVVVVWAAAAAAAhNnWF
VVVVVVcAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA==}

show_image bmp base64 4-bit-RLE {
Qk1sAwAAAAAAAHYAAAAoAAAAIgAAADIAAAABAAQAAgAAAPYCAAAAAAAAAAAAABAAAAAQAAAA
////AMzM/wAA//8AmZn/AJmZzABmmcwAMzPMADNmzABmmf8AmZmZAMxmMwBmM2YAZjMzADMz
ZgDMzJkAAAAAAAAGBkMQABwAAAACBh5mAmAAAAIGHmYCYAAAAgYeZgJgAAACBh5mAmAAAAIG
HmYCcAAAAgYeZgJgAAAAIgZmYGZjVmFmY2YFZgYGZmZgAAAAACIGZmBmNmZjZmZmBmYAZmZm
QAAAAAAiBmZgZmYGY2ZmZgZmBmZmZjAAAAAAIgdmYGZmZmNmVmYGZgZmZmYwAAAAAgYKZgAW
YmYmZmZmBmZmZhAAAAACBg5mAgYOZgIQAAACBQxmAARq6g5mAgAAAAIEDGYABquqRgAMZgIA
AAACAwpmAAhqpqqmDGYCAAAAAgMKZgAIcaaqowxmAgAAAAIDDGYABqasoQAMZgIAAAACAQpm
AAhnpqpgDGYCAAAAAgEKZgAGaaaqAA5mAgAAAAIACmYACGmlqrMMZgIAAAACAApmAAhsqKrC
DGYCAAAAAgAKZgAIeqaqoQxmAgAAAAIACmYACIqmqqAMZgIAAAACAApmAAg6pqqgDGYCAAAA
AgAKZgAIKqaqqQxmAgAAAAIACmYABhqtqgAOZgIAAAACAApmAAgWrKqmDGYCAAAAAgAMZgAG
qpqqAAxmAgAAAAIADGYACKqqqhYKZgIAAAACAAxmAAaqqqoADGYCAAAAAgEMZgAKqqqqsWgA
CGYCAAAAAgEMZgAKqqqqoGUACGYCAAAAAgMMZgAUqqqqoGZ2ZmZmEAAAAgMIZgAYdmaqqrYA
ZiZmZmYwAAACBAhmABh2ZiqqpmZmJmVmZkAAAAIFDGYAFGqqqmZmJmJmZmAAAAIGCGYADGdm
aKqqtgpmAnAAAAIGCGYADGJmZhqqpgpmAmAAAAAWB2ZmZmdiZmZgmqYACmYCYAAAABYGZmZm
aGZWZmYQBgAKZgJgAAAADgZmZmZoZiYAEmYCYAAAAA4GZmZmZ3ZiABJmAmAAAAIGCGYABiZo
JgAQZgJgAAACBh5mAmAAAAIGHmYCYQAAAgYeZgJiAAACBhZmAAo1ZmZmZAAAAAAMAAASNFZ2
CGYCZQwAAAAiAAAB}

show_image bmp base64 8-bit {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==}

show_image bmp base64 8-bit-RLE {
Qk22CAAAAAAAADYEAAAoAAAAIgAAADIAAAABAAgAAQAAAIAEAAAAAAAAAAAAAAABAAAAAQAA
////AO7u7gDMzP8Amcz/AJmZ/wCZmcwAZpnMAGZmzAAzZswAMzPMAAAzzAAAAMwAAMz/AACZ
/wAA//8AAGbMAACZzADd3d0AzJmZAJkzAACZZgAAMzOZAGaZ/wCZZjMAZjNmAMyZZgCZmZkA
ZjMzADMzZgDMmTMAu7u7AMz//wBmZv8Ad3d3AIiIiADMzJkAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAG
AAcFBAIBHAAAAAEACgoGCRAKAQAAAAEAIAoBAAAAAQAfCgEJAQAAAAEAHwoBCQEAAAABAB8K
AQgBAAAAAQAECgAWCwoKCgsKCgoLCgoKBwoKCwoKCgsKCwUKAQcBAAAAABwACQoKCwAHCgoE
BgsKAgcKCwQKCgEGCgkACQAJBAoBBwEAAAAAGgAJCgoLAAsKBAkLCgoECQoLBwoLAAsKCwAA
BgoBBQEAAAAAGgAJCgoKAAoKCQcACwoECQoJCgoKAAoKCwAHBgoBBAEAAAAABgAICgoLAAcK
AA4ECQoGCwoLAAsKCwAKCQUKAQQBAAAAAAYABwoKCgsHCgANAwkKAwsKCgsKCgoACwAGCgEC
AQAAAAEAAQcOCgERARUOCgECAQAAAAEAAQYMCgAECRcjEw4KAQEBAAAAAQABBQsKAAYJFBgZ
EwUNCgEAAQAAAAEAAQQKCgAICxkTFRcTFwkMCgEAAQAAAAEAAQQKCgAICAITCRQTEwQMCgEA
AQAAAAEAAQQMCgAGFAcUGxMCDAoBAAEAAAABAAECCwoABwgTBxQTIAEADAoBAAEAAAABAAEC
CwoAByIUBxMTBwsADAoBAAEAAAABAAERCgoACAshEwYTExgEDAoBAAEAAAABAAEBCgoACAkb
ExYTExsDDAoBAAEAAAABAAEACgoACAgTFCATExMCDAoBAAEAAAABAAEACgoACBYTFAcUExMf
CwoAAwkAAAAAAAEAAQAKCgAJBBMTCRQTEwELAAoKAAMJAAAAAAABAAEACgoACQMTEwkXExMe
CwAKCgADCQAAAAAAAQABAAoKAAgCFxMcHRMVBwsKAAMJAAAAAAABAAEACgoABwIJExsZExMA
DQoBAAEAAAABAAEADAoABhMTGhMTFAwKAQABAAAAAQABAAsKAAkJExMXExMTAgsACgoBAAEA
AAABAAERCwoBBwYTAQcBCQoKAQABAAAAAQABAgsKAQcGEwAEGAILDQgKAQABAAAAAQABAgsK
AQcHEwAEAAsQCwcKAQEBAAAAAQABBAsKAQkBFAYTAAQACwoPBwoBAgEAAAABAAEECAoADw8L
CgsZExMTGAkBAAsLDgAHCgEEAQAAAAEAAQUICgAFDwoKCgMABBMADwkKCwoLDgsKEAsKCgoF
AAAAAAEAAQYNCgEZBBMECgAEDgoKDAQKAQcBAAAAAQABBwkKAA4PCgoKFhcTExMYCgoKCwcK
AQgBAAAAAQABBwgKAAwLDgsKCgsCFBMTExUKCgEJAQAAAAEAAQgHCgADDwoMAAQKAAYLARIT
EwkLCgEAAAABAAEJBwoABA0LChAECgAFCwIAEQsACwoBAAAAAQABCQcKAAUNCwoOCwAFCgEL
AQsMCgEAAAABAAEJBwoABg8PCgoMCxIKAQAAAAEACAoABwsMCwoNDgsAEQoBAAAAAQANCgEL
EgoBAAAAAQAgCgECAAABACAKAQMAAAEAFgoACwkEBgcHCQkJCgoFAAAAAA0AAAABAgMEBQYH
CAkJAAgKAQYMAAAAIgAAAQ==}

show_image bmp base64 32-bit {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}

show_image gif base64 gif87a {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}

show_image gif base64 gif89a {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}

show_image ico base64 "windows icon 32" {
AAABAAMAICAQAAAAAADoAgAANgAAABAQEAAAAAAAKAEAAB4DAAAgIAIAAAAA
ADABAABGBAAAKAAAACAAAABAAAAAAQAEAAAAAAAAAgAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAIAAAIAAAACAgACAAAAAgACAAICAAADAwMAAgICAAAAA/wAA
/wAAAP//AP8AAAD/AP8A//8AAP///wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAACIiIiIiIiIiIiIiIiAAAAAiIiIiIiIiIiIiIiIgAA
AAIiIgIiIiIiIiIiIiIAAAAO7u4O7u7uEREREe7uAAAADu7uDu7u7hERERHu
7gAAAA7u7pDu7u4RERER7u4AAAAO7u6Qme7uEREREe7uAAAADu7ukJnu7hER
ERHu7gAAAA7u7pkJnu7hEREe7u4AAAAO7u7pAJnu7hER7u7uAAAADu7u6ZAJ
7u7hHu7u7gAAAA7u7umZCZ7u7u7u7u4AAAAO7u7umZCe7u7u7u7uAAAADu67
7umQme7u7u7u7gAAAA7uu+7pkAnu7u7u7u4AAAAO7u7u6ZkJnu7u7u7uAAAA
Du7u7u6ZAJnu7u7u7gAAAAAAAAAACZCZAAAAAAAAAAAAAAAAAAmQmZAAAAAA
AAAAAAAAAAAJmQmZAAAAAAAAAAAAAAAAAJmQmZAAAAAAAAAAAAAAAACZmQmZ
AAAAAAAAAAAAAAAAAJmQCWAAAAAAAAAAAAAAAABpmQmQAAAAAAAAAAAAAAAA
AJmQkAAAAAAAAAAAAAAAAAAAmQAAAAAAAAAAAAAAAAAAAAAAAAAAAP//////
///////////////AAAAHwAAAB8AAAAfAAAAHwAAAB8AAAAfAAAAHwAAAB8AA
AAfAAAAHwAAAB8AAAAfAAAAHwAAAB8AAAAfAAAAHwAAAB8AAAAfAAAAH+fgf
P/54DP//nAP//+QD///5Af///gH////B////8f//////KAAAABAAAAAgAAAA
AQAEAAAAAACAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIAAAIAAAACAgACA
AAAAgACAAICAAADAwMAAgICAAAAA/wAA/wAAAP//AP8AAAD/AP8A//8AAP//
/wAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIiIiIiIiAADuDu4REe4AAO6e
7hER7gAA7pnuERHuAADukJ7hHu4AAO6Qnu7u7gAA6+kJ7u7uAADu6Qnu7u4A
AAAAkJAAAAAAAACZCQAAAAAAAAkJAAAAAAAAAJAAAAAAAAAAAAAAAP//AAD/
/wAAgAEAAIABAACAAQAAgAEAAIABAACAAQAAgAEAAIABAACAAQAAgAEAAOQn
AAD4HwAA/z8AAP//AAAoAAAAIAAAAEAAAAABAAEAAAAAAIAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAP///wAAAAAAAAAAAAAAAAAAAAAABEREQBERERAERERA
EREREARERUAffxDwHw9E8B83EPAfV0TwH2eR8B+rw/Af1efwH9b/8B/q//Ac
63/wHO1/8B/1v/Af9r/wAALAAAADYAAAA2AAAAGwAAAAyAAAACwAAAAUAAAA
CAAAAAAAAAAAAP/////////////////////AAAAHwAAAB8AAAAfAAAAHwAAA
B8AAAAfAAAAHwAAAB8AAAAfAAAAHwAAAB8AAAAfAAAAHwAAAB8AAAAfAAAAH
wAAAB8AAAAfAAAAH+fgPP/54DP//nAP//+YD///5Af///sH////h////8///
////}

show_image ico base64 "windows icon 16" {
AAABAAEAEBAQAAAAAAAoAQAAFgAAACgAAAAQAAAAIAAAAAEABAAAAAAAwAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAA////AAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAEREAEQABERER
EBEBARAREREREQEBEBEREREQEQEQEREREREBARAREREQEQEBEBEREREAEQAB
ERERERERERERERERAREAEQEREREBEBEBAREREQEQEREBERERARAREQEREREB
EBEBAREREQERABEBERERAREREQEREQAAARERAREAAP//AAD//wAA//8AAP//
AAD//wAA//8AAP//AAD//wAA//8AAP//AAD//wAA//8AAP//AAD//wAA//8A
AP//}

show_image jpeg base64 color {
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJCQwLDBgNDRgyIRwh
MjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/wAAR
CAAmADkDASIAAhEBAxEB/8QAHwAAAQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAA
AgEDAwIEAwUFBAQAAAF9AQIDAAQRBRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkK
FhcYGRolJicoKSo0NTY3ODk6Q0RFRkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWG
h4iJipKTlJWWl5iZmqKjpKWmp6ipqrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl
5ufo6erx8vP09fb3+Pn6/8QAHwEAAwEBAQEBAQEBAQAAAAAAAAECAwQFBgcICQoL/8QAtREA
AgECBAQDBAcFBAQAAQJ3AAECAxEEBSExBhJBUQdhcRMiMoEIFEKRobHBCSMzUvAVYnLRChYk
NOEl8RcYGRomJygpKjU2Nzg5OkNERUZHSElKU1RVVldYWVpjZGVmZ2hpanN0dXZ3eHl6goOE
hYaHiImKkpOUlZaXmJmaoqOkpaanqKmqsrO0tba3uLm6wsPExcbHyMnK0tPU1dbX2Nna4uPk
5ebn6Onq8vP09fb3+Pn6/9oADAMBAAIRAxEAPwDzLyJSuV5FIsUm4DAzW1Ywpb3sbPhoQ4LA
jPGea6jWdH0W6EdxYLJA78gICUb169KIq8JTutDWnSnUmqcFds4y2e5jkzGrOVGTt7Ct2aae
ZFzGSR6cmni1t4RiFWCseRknJHf+f51MsZG4JHwP4hXmVaqctEfVYDLqsKbTla+43S7NZLkr
dHamMEgg9R6isDVtDu7KdjFGZoOodOcD3rowWIVRkbDuUHI3DFbWnOs1n9pIIMZwwIzyBxwe
v/16ujWs9iK+VQqfFN3W3Yo6Pp/n6LbXFx9rWeCNifLbjac9T6YxXN+a/wDzy/SvedC13S5/
DtnZ3ckZmMQSVVTK8cYJ6HI96tfZvC3/AD42/wD35NdfM5dT5Pdtp3PBtEji1CVTtyR1U9vc
11WpXOnRW9rYwWxB6O+/n2OOgrm40EZyg2n2rTm8M6xp+lrrk1q76bIu4hcMwB74zkD3/TFV
KcXRcFHXudWGm6deM+axUmCQytGjrIi8Kyg81NFKrxqogVsMWk5OH78j8/YfnQ1hKlrbXU8F
xCk43ws67d4/L3H5j1rLurqVLhhno2RXlKErn2f1qEIKV7xL77QN21VxwMdqdLdvBpbpDveS
5cJGiHBzyMj8/wBB61i3OoySJjhR3C0+71FYYfIjncSgbSUOMj0J/mBW1OlJSuzz8fmsY0nG
kvQ6qO4ubaLyt8fnR/Idh3AEdfyNQeVrH/QQuf8Avv8A+tWd4fu0j0dePnEjIM+vU/zqz/aq
f8/J/wC/h/xotNSaifEO6bsVo1yy8DmvetkF3HbaW0INuYFMiYwpXA4wKKK7Y7HWtzSvNPs7
ywezurdJLYrjyyOgx29D6EcivGviF4EtfDlnFqVpeSvFLMsPkSKCVJVjncO3y9MfjRRRUirX
NKdWcfdi9DhbW3YRS3a7cW+GOepycDA+vWsaS3AcqScjvRRUU9zOcnLVlqx8+Elll43qoH1r
K2L70UVaerMluf/Z}

show_image jpeg base64 grayscale {
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/wAALCAAmADkBAREA/8QAHwAA
AQUBAQEBAQEAAAAAAAAAAAECAwQFBgcICQoL/8QAtRAAAgEDAwIEAwUFBAQAAAF9AQIDAAQR
BRIhMUEGE1FhByJxFDKBkaEII0KxwRVS0fAkM2JyggkKFhcYGRolJicoKSo0NTY3ODk6Q0RF
RkdISUpTVFVWV1hZWmNkZWZnaGlqc3R1dnd4eXqDhIWGh4iJipKTlJWWl5iZmqKjpKWmp6ip
qrKztLW2t7i5usLDxMXGx8jJytLT1NXW19jZ2uHi4+Tl5ufo6erx8vP09fb3+Pn6/9oACAEB
AAA/APMvIlK5XkUixSbgMDNX7Z7mOTMas5UZO3sK3Zpp5kXMZJHpyan0uzWS5K3R2pjBIIPU
eorA1bQ7uynYxRmaDqHTnA966/R9P8/Rba4uPtazwRsT5bcbTnqfTGK5vzX/AOeX6VFYwpb3
sbPhoQ4LAjPGea6jWdH0W6EdxYLJA78gICUb169KzBa28IxCrBWPIyTkjv8Az/OpljI3BI+B
/EKcCxCqMjYdyg5G4Yra051ms/tJBBjOGBGeQOOD1/8Ar16PoWu6XP4ds7O7kjMxiCSqqZXj
jBPQ5HvVr7N4W/58bf8A78mvBtEji1CVTtyR1U9vc11WpXOnRW9rYwWxB6O+/n2OOgrDmCQy
tGjrIi8Kyg81NFKrxqogVsMWk5OH78j8/YfnUT7QN21VxwMdqdLdvBpbpDveS5cJGiHBzyMj
8/0HrWnHcXNtF5W+Pzo/kOw7gCOv5GoPK1j/AKCFz/33/wDWrIjQRnKDafatObwzrGn6WuuT
Wrvpsi7iFwzAHvjOQPf9MVE1hKlrbXU8FxCk43ws67d4/L3H5j1rLurqVLhhno2RVa51GSRM
cKO4Wn3eorDD5Ec7iUDaShxkehP8wK0vD92kejrx84kZBn16n+dWf7VT/n5P/fw/41WjXLLw
Oa962QXcdtpbQg25gUyJjClcDjArSvNPs7ywezurdJLYrjyyOgx29D6EcivGviF4EtfDlnFq
VpeSvFLMsPkSKCVJVjncO3y9MfjXC2tuwilu124t8Mc9Tk4GB9etY0luA5Uk5HerVj58JLLL
xvVQPrWVsX3r/9k=}

show_image jpeg base64 "progressive color" {
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/2wBDAQkJCQwLDBgNDRgyIRwh
MjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjIyMjL/wgAR
CAAmADkDASIAAhEBAxEB/8QAGgAAAgMBAQAAAAAAAAAAAAAAAAQDBQYBAv/EABkBAAIDAQAA
AAAAAAAAAAAAAAECAAMFBP/aAAwDAQACEAMQAAABzA7aRqV/vvl1eIWLrCCt3stuTgrWtaNs
PsV5dljqXX57XwtLMSLehdazjQjUSQBZVQK//8QAHhAAAgIDAQEBAQAAAAAAAAAAAQIAAwQR
EiEQBRP/2gAIAQEAAQUC0ZoxegSSZUm2toZDTXuncQcvdTS05UTXyv1KL6jj84spAsNjVhT4
QfIW1UGZRq6DyHGurq4PLMemsJjWaGO2qf6wTxg9aOn6GCuOir4VibHz/8QAHxEAAgEEAgMA
AAAAAAAAAAAAAQIAAwQQERIxEyFB/9oACAEDAQE/AVUseIlC3YL3HtQ3ZxTPFwdzygDfyV7o
BdLlWI9CE7x//8QAGxEAAwEBAAMAAAAAAAAAAAAAAAECESEQEjH/2gAIAQIBAT8BXxsqukWb
o2vTMMZMvTu8EUifH//EACgQAAEDAwIDCQAAAAAAAAAAAAIAAREQITESUQMiMhMjM0FScYGR
of/aAAgBAQAGPwKllhcys0shItUs3ksJtkxBLKyxTVsgAnaYuugfqggwqMrHvS2SwotLLxCp
27j3boSdiacVhnuvldX7QeFHLF1oIZHZNxBN4d4h05emmaf/xAAgEAEAAwACAgIDAAAAAAAA
AAABABEhMUFRcWGBEJHx/9oACAEBAAE/IcshZFgLXiHkBRkjX2ZMMXcupaDQ7fZRlXxwzq2m
HKvtNw8afMR+4nAyQDPwUhn1PcOblWCA4SAgq7fyjXNBFYupQJoHCzZ/Qg6ZDC0rzWJF3bSr
hl7gHj1CvvTIIebEpAaSnHFp1UQioh8wNbjHv6jUa7TSpzzsEon/2gAMAwEAAgADAAAAEIb7
MuvRaP43Yv/EABoRAQADAQEBAAAAAAAAAAAAAAEAETEhUXH/2gAIAQMBAT8QGjax8NL2bzZn
k1Ubj8VKe1w5H5Gxahs6xyN0w2f/xAAaEQEBAQEBAQEAAAAAAAAAAAABABEhMUFR/9oACAEC
AQE/EBu05G8Fg+WvqUj1+xtC02FE8QZt6h63/8QAJBABAAICAgIBBAMAAAAAAAAAAQARITFB
UWGRcaGx0fCB4fH/2gAIAQEAAT8QvtkSkKLl8RFvQllJTrLEKYaUR2dkTm2mejzMlLHRa9vV
VP3kopAoS8XmEZJwK929QBQEyWtpz9/cQpiOEFAWXobKVLxEVQS8hjDv+46IjecMUumzzP8A
cRqyTa48sXGOt/w1ogxTIDmGiosjXPJ78HuVGgwVxK9tt03ks9/Q7nZ8b0E36Z+z/EFWbeJY
AwKAPNXYefpUHWepPryezuBl6sSs4HJFYsll1Z0v3CYYxDfe37zz/b8yzAZldCmppUMURDFp
Y0Vx09JklKrxFSl046V/MBjlFe22ij53AGls5lpwah8z5p//2Q==}

show_image jpeg base64 "progressive grayscale" {
/9j/4AAQSkZJRgABAQAAAQABAAD/2wBDAAgGBgcGBQgHBwcJCQgKDBQNDAsLDBkSEw8UHRof
Hh0aHBwgJC4nICIsIxwcKDcpLDAxNDQ0Hyc5PTgyPC4zNDL/wgALCAAmADkBAREA/8QAGgAA
AgMBAQAAAAAAAAAAAAAAAgQDBQYBAP/aAAgBAQAAAAHMenfNC3rRtFT67pJcFaomPWQTaBWP
rMsW9ZxtElKr/8QAHhAAAgIDAQEBAQAAAAAAAAAAAQIAAwQREiEQBRP/2gAIAQEAAQUC0Zox
egSSZUm2toZDTXuncQcvdTS05UTXyv1KL6jj84spAsNjVhT4QfIW1UGZRq6DyHGurq4PLMem
sJjWaGO2qf6wTxg9aOn6GCuOir4VibHz/8QAKBAAAQMDAgMJAAAAAAAAAAAAAgABERAhMRJR
AyIyEyMzQVJxgZGh/9oACAEBAAY/AqWWFzKzSyEi1SzeSwm2TEEsrLFNWyACdpi66B+qCDCo
yse9LZLCi0svEKnbuPduhJ2JpxWGe6+V1ftB4UcsXWghkdk3EE3h3iHTl6aZp//EACAQAQAD
AAICAgMAAAAAAAAAAAEAESExQVFxYYEQkfH/2gAIAQEAAT8hyyFkWAteIeQFGSNfZkwxdy6l
oNDt9lGVfHDOraYcq+03Dxp8xH7icDJAM/BSGfU9w5uVYIDhICCrt/KNc0EVi6lAmgcLNn9C
DpkMLSvNYkXdtKuGXuAePUK+9Mgh5sSkBpKccWnVRCKiHzA1uMe/qNRrtNKnPOwSif/aAAgB
AQAAABCrPPK25v/EACQQAQACAgICAQQDAAAAAAAAAAEAESExQVFhkXGhsdHwgeHx/9oACAEB
AAE/EL7ZEpCi5fERb0JZSU6yxCmGlEdnZE5tpno8zJSx0Wvb1VT95KKQKEvF5hGScCvdvUAU
BMlrac/f3EKYjhBQFl6GylS8RFUEvIYw7/uOiI3nDFLps8z/AHEask2uPLFxjrf8NaIMUyA5
hoqLI1zye/B7lRoMFcSvbbdN5LPf0O52fG9BN+mfs/xBVm3iWAMCgDzV2Hn6VB1nqT68ns7g
ZerErOByRWLJZdWdL9wmGMQ33t+88/2/MswGZXQpqaVDFEQxaWNFcdPSZJSq8RUpdOOlfzAY
5RXttoo+dwBpbOZacGofM+af/9k=}

show_image pcx base64 "true color" {\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}

show_image pcx base64 "indexed" {\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}

show_image png base64 color {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}

show_image png base64 grayscale {
iVBORw0KGgoAAAANSUhEUgAAAC4AAAAjCAAAAAG5S1ejAAAEJElEQVR4nKWTXUwcZRiFH2YG
ZpiBLsx2wW6ZpU1YC+3axWClf7ZKY5NVozHRxIYaL0xq1YuGeOFVRW40TXthk4aLxsT/xJjG
+tdiGkUl1nQV6rYokuy2IFugy8/irp3dHXYGvFiwBdQ08Vx9+d73Oznfe84LAIjQNv+ELdBy
aVqTRBSh/8p6aGMZWpDomBwwETg7020sL0NjG0XsTTT9GmlSdreji1xZ555bvzmyVvcNAI/s
pRX/8lfi4uHoREIEnrnEKwNjcxUibdt+39PX6gwNTEkkB0l+3VPtdgclQI8fs0zd8gvoMkZX
MBywY3B0Ox0FKoGcTvtKhQuQa5tBuvXmkHOCx4Y1xS7o6Qg29vFi6DvMe79paRjP5sIgAL/1
jxwKJC4ewS72+kp7WfX4UQRgFUlBjA1/2R7t25DQdpbZiQJPU5V1xVMuqW+9ykc+X37WKJLC
AmD8gkJX0HIOS3S3f+B3h48jAVNI0Viztbub4wCdC+rUf/3J7eEBOgr6l0Dbr4TUpffBl0Jw
cARr5qYvL2wPM1d8bv65a3kzPTKKBHU7AuFT53eAS3FaSsdyynQcRGiNxXaVff9wfa+Of/rB
T66OpgkmRNiYkkssj6CuDusJ6s4YDTUHTjsCuEiNauQu7MNqXJ8LekGoRwAVXNGkV+1/33Wn
mVdERYx5EWDCttbJqpQ5NqCRSgV0UbE0BBjzP/XpBDn90Te3ll3q8aXXqGIVAvSInau8rrSV
m5iaY9DaU6TUdyKAKVUhWh/qSU6Owmv9mv9wIQ+OxynqbpZ6IQ6cAgrzURXtc/ZoDnQtGWFH
6z/OW/t/dhkARbfR2DGer5avbnyZZWleCv/+C13AkYmP3bkxqy73X+y6v9F9NgLNW8Sf7DJs
yzI+X8ne3DK99r0YoDNSEUHbdb95RiozLWR5TM2ABP79fyRlc8gfiKZ7wuFQ5dOR01CU8vpp
kZMp05St+UFrga4IQN3npMg7W2/8UCm5k9uT2TcgID7/rqkF/LnDANTvkNYNvlMIZn6L73re
YfWMEdEynm8bSkIXs3Lppki21G3fsGpr3bU1yvS21InFHO8sjTMfryzNXIxbmyPXhPrm0Bkl
fXdGKr/DfVl2ZEEQqJhl9O91SF+fFMtWR+8KeKdsogdSxe3ynCdnR0dn3IqmuFyKkqwxFtmN
alutu2/qx3FRsZuGXFm7a8OsL+r5ikSFx53Vb5RJomBWO30L7WaDNTN8fthTrLrmZHnGVz58
4Z4ST81nU3/Khl6SnbWy5ZVGxu5btOmQfa7GUmRLzZhmU0oyu8B4SOwEEJ/VsFJVCq/ftCm/
xnDZOVC+gCd/dnqB+NsH1QzgnLzFxoX2CoGELiu9kxBKpo1JAOv4ygwvhsDcNC6lTgF7h25u
g7mivaD9QKlyOZxcUVyJvwDv65E2xA9RYAAAAABJRU5ErkJggg==}

show_image png base64 "transparent color" {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}

show_image png base64 "transparent grayscale" {
iVBORw0KGgoAAAANSUhEUgAAAC4AAAAjCAQAAAE2KcD0AAAFPElEQVR4nMXWXYicZxUH8N+z
M3HezCy76axJzJrtR8jWVVc29IMtTTG6hUqqqL1QWlrxQmirXpTihRdSY4pUSws1IIEWBesH
CAZFapNaSISgpUurJra2i7u2IWs+dpNMnbWzndfM9Hixs5tNmrYJCv7vHvi///ec85z/OQ/L
kODeeMZ1fq+LsTjolIqixAejx3jqj5EFknfEWFBke5zwkga62O01+9LAO38INsW9QeKmmHG1
vzrgapkttiWqkWA0hsziSr/1CX+T25fAp+Km4PZg8F2CfCseipGga+HwxYBvxur4hWQoigtp
T7grHk3rbXHAVKJIzQRq9sZ+a/XpixHFBZGqaQ/LNVTlBnVRVcKAPWnEeBrWMrXw7+uD7WfF
3EVTFdvSxaaiFJfFaGAxlLfinmj7fuIzcUhFpmUZfXvUND2a+Gr0uT+x1+cxFhsc1zDvQOrE
CC97wWH3xHDM+JMHg5YV+uNSKz2PHrfEQ2eK24OaLgVTDnnKtpj0Rx8wo+IG3VpmLFNnSNW4
futUZb7tW9b6Xcq1rFe2QdU/ztAHvIgMe9KIXDvdl4rYl7aln6VBfcbtSEupnkTRpKk0Grkt
9mHHUsV3nl378kV31f8SH1/qkbe91UVU4hanbI0ylhXybIzE12JrwN0OI/caOqPgDL4SK+xI
rI5rPK2d7opZdU3zSsYXC7kxNhs2blf6g82gV6adxmKlo5oyp0yf6Zmtpu1xrc/Gi9a6O2DG
WJyy0SsOpvE0nRiJjvq/UHJUZovDun0ynkyVOKak5kQaiH7c7IFF9V7UHVFB07NuQ26TKzSN
RH+nJkOL9HIn3kk1/cpe8NPodaWG0zIFmYIp/Yv0WS25y5WUFc172EsqqKsbVlWQyVUW6UcN
utWvzVow9Kf9wHW6HUz7XWrOOmUFaxbp+xXs1KNfrzm5ptl00puYSLkbJZkhO1OH3khFa1CQ
+7mqGh5LR8AD6QUVg+5Ly3qmbbW2ZF8ajaLnwXTnpnctu/lOz5RlKp5I3KiinWDP2w+k7XH7
hVuk8v9008DSz9/VTBeG7XHMaWuVvOJDvv7fiA/GHZ5dKvODMeuX+jQdlduoucS7yMVQjUGb
9Nndmd6jca2C57R0oyWXG/BEuoDIR2PMKe/3E1MdehWHrXIgUYmP+piGJxV1a8hRUnJUOebP
NONg3OGfakoaXjVo2KQ5+42ncVvjEl9wIH6VFhKt6zeIsSipqWtoKMmFCfk5dVg6luM2bXWc
1nad1z3jEkV9aq5X84bvJRiOgi/7sYaKYYOaHcssYCg2K7rchMfTslk6nwqu6EjTVHeVQ3Iz
qnbJ9PhuVIM65v1bjrbjTtoUo3F9bIrRGI1eL9tgzuPL9ymsxGkwbVbJvBnjpqxSs9srqr7h
kaiZt9cN+mVaeqwXCtrKCgoKOG7ewlPgnC0w57gTCrq916SPGNbvpBYm3ZnqVtim5E2rNbVM
OuI1fTIVmV69Mpma9RYedcvEj2DAgKvcqmmvY15X1LJBr4qW1XF/+rtTblRTM2HWIbMY0WMd
Mpmyklnz+s4Vf05TpumgH9ons1q3XiUzNmq6wjAeSU3vcY33qaWJdDS96g1Vl0lKGuoyawyp
ndstcE+0PG29XKYkVzavoeFqdUWNjicH4maFZW+DQnxJBbm6NTJ853wmOm2dAb1aHQtnfpPg
c/Fn7c4oZjr9KO5eMgrt9Jjz4yzxVbowo6ok87wTCbZGzZyBzgnytOOCBu859m/4sGOK6ktL
4qZ49byboHEBU2kZ5c5YKfMX42oX/849L/4D4jD0vGNZ6YMAAAAASUVORK5CYII=}

show_image ppm base64 "true color" {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}

show_image ppm base64 "indexed" {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}

show_image sgi base64 "true color compressed" {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}

show_image sgi base64 "true color uncompressed" {
AdoAAQADADIALwADAAAAAAAAAP8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABlD
PkYcAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAATXWBbZIqEd1UnGRUFAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAADBYY2BYZ7q/taOUhU48MikCAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAD01YYGROqcfS3d3TxquekmQiAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAWPjwQAAAENFZbYlZIkZ2ltr7Q3NTNvUcAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAFdmW0AJAAASSF5mYVZPW2J1gIuWoqy6TAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAeXGZbTB0AAAQvVGZmYUYiFyI2SVdjjYE0
AAAWP0IkAAAAAAAAAAAAAAAAAAAAAAAAAEJQX2BXNAkAABpDYGBcUSUCAQM2
hJybZhQABz9YYVQnAgAAAAAAAAAAAAAAAAAAAAAASE1ZZF9MGAAAAC9YY2Fa
ShAAClCVrYxDAAAVTFVcYlwmAAAAAAAAAAAAAAAAAAAAABdoN1lkX1AwBQAA
Dk5YX2dlSDMkYqajfhoAADpXR1piYlQkAAAAAAAAAAAAAAAAAAAAJoKXTGFm
XkoOAAADNk92sLacjXRxcmZOEQ0MT1c1T1xlY1MlAgAAAAAAAAAAAAAAAAAy
kp46XmBdUzIAAAARVKfv9NjHwriqoJaJfnRaZ2BGU0thXlgiAAAAAAAAAAAA
AAAACVScn1hXYGRZPQ4AAAQ8nebw2dXd39/Vy8e9s6ackYV3YGxlWE0cAAAA
AAAAAAAAAAASdKibZ0paYVdGIQAAACRmnqGfqaixv8XM2N/l4tbNxLiupJeL
g2QcAAAAAAAAAAAAABSGqpBUQV9oa2A6BwAAACU+UF5pb3R/i5Kbp7K9yc3T
29jT29TMt08FAAAAAAAAAAAAPJatjVI4V5O/qGsiAAAAAAIPO2iBTUFNWWVs
cn2Jk56qtcDM0trLaQwAAAAAAAAAAABdoKJ7Ox9lyPXlmEIHAAAAAAFCj6p5
PAoONEMxQ1RaZWRwfYeSnpY8AgAAAAAAAAAAAHeoo25UXWu67N2PQxIHAAAA
GWakqHkWAABOiq6RTAkDGCcxPEZJMgAAAAAAAAAAAAAAgq+ZcKB2eKu6p2hm
a1NDKi1CWmh0XAIAGWSfrXopAAAAAAAABQEAAAAAAAAAAAAAAACIrYapzomI
q6GAU4K3tHxgh35wZGFiTT03PWaZXwgBAQAAAAAAAAAAAAAAAAAAAAAAADRT
YHFZVparnnV0vcmiXKjm3bioqqOVi42EenBjXUs2GhgWEAsCAAAAAAAAAAAA
AAAAAAQOGjZ1oKySZ0dzfnZovfny39vZ18rAwbqvqqKelI+Fem9gUDgrDAAA
AAAAAAAAAAAAAAAAKISnoYFXDRYbPm6g3c66yM7V0dTc4tzb1dLIwrq3raCd
mIQ0AAAAAAAAAAAAAA0VExVaj7CWdz0AAARWgrClh3yAjI6Vn6imsrrCz9HT
3+La2tzS0JYVAAAAAAAAAAAcTl9QWm2dro1oPSAXG2eRtZt7Ty01RFhfZXF8
e4KNl5qdqLO8wsbLjRQAAAAAAAAAADtcX0pIgaukiFZxcGlKeaGwkWo8AAIJ
FTxUYWFSOURWYmRnc3yEkI1DBQAAAAAAAAAAO2JkWVmNsqR5RLPEs3CIs6qC
V2xNIQAAHUxYZFowAgMEDD5qna6baxoAAAAAAAAAAAA2XWJXaZ2tl2tVtsp6
ZpqznnVGVFgyBgAJNVVgYEYSAAAVU32nqYs9AAAAAAAAAAAAACdbYU96qayN
W1lxfGJzqKyTX0pZYkoRAAAfS19mXSkAAyxnkaqdbQ8AAAAAAAAAAAAAA1Fl
W4ivpYBCEAcZVI2wp4ZFOV5mXDsCAAI1WWNlVBsRWXynrIs7AAAAAAAAAAAA
AAAAS2Zrka6bbS0AABtnkradbiccTmZeThIAABpHWmRZQj5tkrSZZA0AAAAA
AAAAAAAAAABLXXGcsJBfGQADOHajq4xgCQY7Y11YMgAAAypVY11EUYGhrYc4
AwAAAAAAAAAAAAAAADNcfbCkgkwCAAZWirGjfUYAAC1YZGNMGgAAFEVYYExm
lbGdXgoAAAAAAAAAAAAAAAAADluKrJluLQAAHHCarpJvHAAAAEhdXlMxBgAA
Ik5KS32nqIckAAAAAAAAAAAAAAAAAAAOYJWsjmQUAAA+fqmojFQOAAAANlhm
XUoTAAANQlBkkaqdaQAAAAAAAAAAAAAAAAAAAA9trKWCVQgABFeSqpp0MQIA
AAATTGFlWzcDAAAlUXqnrYUdAAAAAAAAAAAAAAAAAAAABkOTkWsrAwAtb6Gp
kGAgCwQAAAUyWGNYRREAAApYibSgWgkAAAAAAAAAAAAAAAAAAAAADCs2JQoA
ADmEqKOGU0VCPjExLzpRUEpSMwAABDh9ooQnAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAUVZGqn3hZio2EgoF3eXtxb3NdIQAABhMaFQYAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAABtom6yMZMvIycTCwLq6urexsJw3AAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAADmKloZeU0s7Q09PS1NTU29Tbwz8AAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAEW97fYuFjo6UlpmXoaGko66JKgAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAADE3NT0+PgQA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAZQz4+HAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAElJeWVeEhG1TJhgUBQAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAuTV1dUGO6v7CbinlMOi4jAgAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAA5KWF9TR6nHzM3MwrmqloJaIgAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAFjw8EAAABDROWV9WSIONlqWzwtDUzLlF
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABVWVlACQAAEkRbWFlPTU1da3aB
j6Ksq0YAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAHlFZWUwdAAAELFJYWllG
HxYiNklXY4Z8NAAAEjs6JAAAAAAAAAAAAAAAAAAAAAAAAAA5SVdWUzIIAAAa
QlVeW00fAgEDNHiakFsUAAc9VV9UJgIAAAAAAAAAAAAAAAAAAAAAAEhFWV1V
QhQAAAAuTFpdVUcOAApIi6KEPgAAFUhLVV5YJgAAAAAAAAAAAAAAAAAAAAAX
YTVTXV5QMAUAAA5JVV1gV0QzJGKcnXIVAAA6V0BUXl5UJAAAAAAAAAAAAAAA
AAAAACV3ikpWWFA8CwAAAzZOaqqukIJyb3BlQw4NCk9XM0NOV2BOHgEAAAAA
AAAAAAAAAAAAK4iROlNeWU4qAAAAEUub5/HOu7GnpJ2WgH5tWWNcQE9LVF5O
HwAAAAAAAAAAAAAAAAdLlplUT15dWTwHAAAEOJPm583KzM7OzMu+s6mckY1+
aVZdXlRMHAAAAAAAAAAAAAAAD2qfmF5IWF9XRh4AAAAkZJ6Zjpulrrq2vcjO
1NHFvreppqGUiYFaHAAAAAAAAAAAAAAUf6SPTTpPXWheMQUAAAAlPk1UXml0
eoCFkJynrbzDzdbSzcrGvKdHAwAAAAAAAAAAADqJoYRINFCFsaVmGwAAAAAC
CzFofUs6RVdcZ3J3hoyPmaSvu8bKv2ULAAAAAAAAAAAAWZWedzgdWrft2Yk0
BQAAAAABP46hczwKDjFBMEBHTVdhbHl/go2HNgIAAAAAAAAAAABpnpJpVF1q
tejOfjwSBwAAABdck5dsFQAATYijh0wJAxgfMDxGQDIAAAAAAAAAAAAAAHie
iGeXc3aarphgXWJKOigtOVZhbFoCABlil594KQAAAAAAAAUBAAAAAAAAAAAA
AAAAe5x+nL2DfqOafEp/sqd4X3pvbGRhWUA2NzxklV8IAQAAAAAAAAAAAAAA
AAAAAAAAAAA0U2BmUVOSqJRuarnFn1am49S0opySkYuBdXhwY1lGKBoYFhAL
AgAAAAAAAAAAAAAAAAAEDhozaJekil9Ea3VqaL357dDO0MbEwLWrqqSYjYiB
fHJuYFA4KwwAAAAAAAAAAAAAAAAAACFzpZ9/TQ0WGztuoNnIt8LAxM3IzNXX
1M7Bwr6xpqCXlox3LgAAAAAAAAAAAAAMEg8VWoSrknI9AAAEVoKjmYR4c3+I
kZeboq6wtcDJ0tvSytHLxcuNFAAAAAAAAAAAHEtRQVdmlZ+IaD0gFxtnjaaU
ckQtNT1PWmVna3WChoaSnaKkrrnEyIQSAAAAAAAAAAA7W1dCRXCenX1WcXBj
SHmYn45iOwACCRU3UF5VSDg9TV1cZ291eIWNQwUAAAAAAAAAADBfV0xXhqSV
cESnv6Rhg6KafEloQBsAABdIWF5TMAIDBAw2Z42fi2AaAAAAAAAAAAAAL1te
V2KSoo5jTqfCdl6PopNuRlRTMgYACTVVXVdEDQAAFU97naCBMQAAAAAAAAAA
AAAnWl5PapyfgVFPa3Rdcpqhj1xEWV5KEQAAF0hSWVImAAMsXY+mlWwLAAAA
AAAAAAAAAANOW1qCoJRzQA8HGFOCn5d5RTJZWVs5AQABNFNeWEcXDVF5maKC
LwAAAAAAAAAAAAAAAEhYY4ijjmUnAAAZZJGljGonHEZXWUYSAAAaRFdeWT01
ZoqlkVwMAAAAAAAAAAAAAAAASFNmmJ+DWRkAAzJvmqKBXwkGO1ZdWDIAAAMn
TF5ZPE50n6eDNQIAAAAAAAAAAAAAAAAzVW2imXhKAgAGVICslXZGAAAjTV5W
QRkAABREWFpFXI2jjlYKAAAAAAAAAAAAAAAAAA5bg6qSbi0AABdnkaeKZxsA
AABDWl1RKwYAACJNRkR2nKB+JAAAAAAAAAAAAAAAAAAADmCOo4pgFAAAOXic
oH1MDAAAADFQWltFEwAADUFMXYumlWgAAAAAAAAAAAAAAAAAAAAPaJyhe0cH
AANLh6iZbSoCAAAAE0teV00zAwAAJUx0mZ1+HQAAAAAAAAAAAAAAAAAAAAZD
iINrKwMAImeZqYtYIAsEAAAFMlNWWEURAAAJUoelj1MJAAAAAAAAAAAAAAAA
AAAAAAwmMCUKAAAvdaSfgVNFQTcxKi81SUtFSDMAAAMtd5x9JAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAFFODqZZsVYOCeHt7dndwZmdrUBkAAAUSGhUGAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAbZZCkjGTCwLy6t6+zsbCpqaCXNwAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA5dpKGSi8jGw8rGyMzLy8zMzLU/
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABFve3t+gICIhY6Olpmb
l6GihioAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAxNzU7Nz4EAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAHU9KSiIAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAABZiZltkk5R+XyocFwYAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAN1tmZVhsydDGtKONVkI2KwIA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAPU2ZvZE622N3r7tzK
u62fZykAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABY+RBEAAAQ1WGdwX0qU
nrLEy9Xt5d7KTwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAV2ZgQQ4AABhP
XmlqWVdbaXeHmaezvbtMAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACVdampY
KAAABjJdaWthSScXJDxPaHSeizQAABZHRSsAAAAAAAAAAAAAAAAAAAAAAAAA
RVpoZ1o9CQAAGkZlZ2hdKAIBAzaLrKJpFAAIRWZwXygCAAAAAAAAAAAAAAAA
AAAAAABUUGZqZE8bAAAAN1hjbmRVEgAKVKa5lUwAABhZXGZvZScAAAAAAAAA
AAAAAAAAAAAAF2w4XGZtWjIFAAAOUWNuZ2VQMyRuta6DGgAAPWRNXGhrXisA
AAAAAAAAAAAAAAAAAAAmg5pUYWlhSg4AAAM4WXu9xKGRg4KDcFARDQxbYjVQ
XGdvViYCAAAAAAAAAAAAAAAAADKdqEFkbmRTMgAAABRZrO/539DTybuxp4+E
emJ0Yk5TS2JtXCIAAAAAAAAAAAAAAAAJWKupYVlsZFlFDgAABEKk8PXm5u7w
8Obcz8TAt62ej35gbG1iVxwAAAAAAAAAAAAAABJ7uappSmZmZVYqAAAALHSv
qaGvuMLQ1t3m7Ovw5t3VyLyypZqSZRwAAAAAAAAAAAAAF5C1oFhEYG55bzoH
AAAAJUpWXmt6gYuXoqS0uMLP2eDs6ePg2c7EUQUAAAAAAAAAAAA/pLWVVjxf
lsi3ciIAAAAAAg87c5JSQVFhbXiDjpahqbK8x9Ta29V2DgAAAAAAAAAAAGeq
sIhDH2rQ9+mbQgcAAAAAAUKgtIM9Cw82TTVGVlxqcn2KlJ+qmD8DAAAAAAAA
AAAAerend1tnecLx4pdDEgcAAAAZarSueRYAAFSZvJlWCQMdLDE8Rko7AAAA
AAAAAAAAAACKuJxxrIGHs8OvbmZrU0MqMEZnbnheAgAZcqyziTMAAAAAAAAF
AQAAAAAAAAAAAAAAAI28lLbai4y6so1VicO4gmuWhn1xcGdOQztAb6dtCwEB
AAAAAAAAAAAAAAAAAAAAAAAANmBsc11gp7ylf3rO16pft/bpyLazsKWckoV/
dWllSzYaGBYSDAIAAAAAAAAAAAAAAAAABA4bP3mvuptvT3yAeXXO//bu5ubf
29HMxMC1saeckYWAeGxcQDEMAAAAAAAAAAAAAAAAAAAohLawkF0QFhxMf7Hm
18vW2dnh5e3z7enj3NnPwsG+saedhzQAAAAAAAAAAAAAEBUTFWCgwaOBPwAA
BV6TtLOViYWQmaKusLO/ytPb2+Pw8+vm3t/dmBUAAAAAAAAAACFbYlJgcau4
mW9DIBsdc569pYNPNj9GWGd2cnyGkZaao665xMjO1dyVFAAAAAAAAAAAR2xo
U0yBtq+OZIF8cUt+qb2fcz8AAgkVP2FiZlk9RFZjbXiDfomWnkoFAAAAAAAA
AAA8ZWZaW5y6poBKwtCzcZS6uI1ZeE8hAAAdVWhvZDQCAwQNP26euahsGgAA
AAAAAAAAADZlZltrprWfcVW+04Fuo7qleVJiYTUGAAk1XmpoThIAABhWi7G1
kj0AAAAAAAAAAAAAJ1tmW3uwspJiYHt+boOtt6FpVWpsUxQAAB9UX2pfMAAD
Mm6gu6t8DwAAAAAAAAAAAAADXWthkra0hksRCBlWkrmvjVE7YGdeRQIAAjZb
b2dXHRFfiqy8kz0AAAAAAAAAAAAAAABTaW2Zu6h3LQAAG3GitqF9LyVXaGpX
EgAAHFBkZ2NIQHCbtqJtDwAAAAAAAAAAAAAAAEtfc6q2lmgZAAM4gLSzlmgJ
CUFnaV4yAAAELVVmZ0RZhbC4kDsDAAAAAAAAAAAAAAAAP2OHtrGJTgIACGSa
wamLSQAALVtmZVAhAAAUR2VqVGyevatlEAAAAAAAAAAAAAAAAAAOXpi8o380
AAAdeKi6m3IcAAAAVGZrYjUGAAApVVBPh7G1jysAAAAAAAAAAAAAAAAAABBt
pr2bahoAAEeJs7OOVw4AAAA3W2tkThMAAA1IVmicu6pvAAAAAAAAAAAAAAAA
AAAAEnmwtYxVCAAEWZi6qn0xAgAAABdSYmVdOQMAAChbhKy8lSAAAAAAAAAA
AAAAAAAAAAAIT56ieS8DAC1wqrqcYiMLBAAABTJgZ2lWEgAACl2YtqBhCgAA
AAAAAAAAAAAAAAAAAAAMKz0pCgAAOYa5tJJgTUk+NjQxP1FUVlk0AAAEOIit
jisAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABVdn7upfWqZkpKKjYWJf3V4c14h
AAAGFR0YBwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAH3SrvZ113NnR08vLy8jB
xLy6rEEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAOZ7Wyo5zg2+Hb
29vi3OPm5eDTSQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAATdIyL
j5GQnJWfn6eqrqu0s5kzAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAMTc+PkNBBAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAA=}

show_image sgi base64 "true color compressed with alpha" {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}

show_image sgi base64 "true color uncompressed with alpha" {
AdoAAQADADIALwAEAAAAAAAAAP8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAGZc
VmFVAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AABPX2BbZI2Ie1hid2dVAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAADhYY2BYZ7q/taOUhYKEb1pVAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAALU1YYGROqcfS3d3TxquekoxVAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAABRQkA/AAA0O1ZbYlZIkZ2ltr7Q3NTNvXsAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAGBmW0QiAAA2SF5mYVZPW2J1gIuWoqy6hAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAABEX2ZbTC0AACJCVGZmYUYzREU2SVdjjYFb
AABVYGRVAAAAAAAAAAAAAAAAAAAAAAAAAGNQX2BXNhkAADVFYGBcUUIzIiJI
hJybaSgANkNYYVhXVQAAAAAAAAAAAAAAAAAAAAAAbU1ZZF9MPQAAADlYY2Fa
SkAAM2CVrYxSAAAwTFVcYlxVAAAAAAAAAAAAAAAAAAAAAFV3N1lkX1A+MwAA
KU5YX2dlZFEtYqajfk0AAEBXR1piYldVAAAAAAAAAAAAAAAAAAAAWYKXTGFm
Xko8AAA1O092sLacjXRxcmZOZXBmVlc1T1xlY1ZVVQAAAAAAAAAAAAAAAAB3
kp46XmBdUzcAAAA4VKfv9NjHwriqoJaJfnRaZ2BGU0thXlhLAAAAAAAAAAAA
AAAAVW2cn1hXYGRZPyIAACJSnebw2dXd39/Vy8e9s6ackYV3YGxlWE9EAAAA
AAAAAAAAAABjdKibZ0paYVdGMgAAAEdonqGfqaixv8XM2N/l4tbNxLiupJeL
g2lLAAAAAAAAAAAAAGuGqpBUQV9oa2BLMwAAAEVRUF5pb3R/i5Kbp7K9yc3T
29jT29TMt31EAAAAAAAAAAAAYpatjVI4V5O/qGtQAAAAABEjRGiBTUFNWWVs
cn2Jk56qtcDM0trLaToAAAAAAAAAAABjoKJ7VCplyPXlmEsxAAAAACJWj6p5
STNLNkMxQ1RdamRwfYeSnplvMwAAAAAAAAAAAH+oo25UXWu67N2PQywiAAAA
MWakqHkmAABTiq6RVSIiIjdGVVBgZgAAAAAAAAAAAAAAiq+ZcKB2eKu6p2hm
a2F/T1VVWmh0XEwARGefrXpDAAAAAAAAEREAAAAAAAAAAAAAAACRrYapzomI
q6GAU4K3tHxgh35wZGFiY047PWaZazY5JAAAAAAAAAAAAAAAAAAAAAAAAEth
YHFZVparnnV0vcmiXKjm3bioqqOVi42EenBjXUs7ZlxTPikiAAAAAAAAAAAA
AAAAABEcNE51oKySZ0dzfnZovfny39vZ18rAwbqvqqKelI+Fem9gUF5XMwAA
AAAAAAAAAAAAAAAAZoSnoYFdM1VZPm6g3c66yM7V0dTc4tzb1dLIwrq3raCd
mIRyAAAAAAAAAAAAAD1fVT5bj7CWd0QAAEdWgrClh3yAjI6Vn6imsrrCz9HT
3+La2tzS0JhmAAAAAAAAAABVUF9QWm2dro1oQEQwNWeRtZt7T1VGRFhfZXF8
e4KNl5qdqLO8wsbLjV4AAAAAAAAAAFVcX0pIgaukiFZxcGlKeaGwkWo8MxEd
LzxUYWFSOURWYmRnc3yEkI1tVQAAAAAAAAAAVWJkWVmNsqR5RLPEs3CIs6qC
V2xRRAAiPUxYZFo5L0RTHz5qna6bckYAAAAAAAAAAABOXWJXaZ2tl2tVtsp6
ZpqznnVGVFhEMwAiPVVgYE1EAAA6U32nqYtXAAAAAAAAAAAAADhbYU96qayN
W1lxfGJzqKyTX0pZYko/AAAtS19mXUQAIjdnkaqdckQAAAAAAAAAAAAAM1Zl
W4ivpYBGKldAVI2wp4ZIPl5mXD82ACI4WWNlVz5AWXynrItIAAAAAAAAAAAA
AAAAUGZrka6bbUcAAFJnkradbjY2TmZeTjUAACpHWmRZQj5tkrSZZEcAAAAA
AAAAAAAAAABPXXGcsJBfKAAzSXajq4xiIiJGY11YPwAAM0NVY11EUYGhrYdZ
MwAAAAAAAAAAAAAAADxcfbCkglUiAEZWirGjfUgAAERYZGNONQAANUhYYExm
lbGdYxwAAAAAAAAAAAAAAAAAT1uKrJluNAAATHCarpJvKQAAAEhdXlNFMwAA
M05KS32nqIxVAAAAAAAAAAAAAAAAAABOYJWsjmQxAABJfqmojFRAAAAANlhm
XUo9AAAzRVBkkaqdcwAAAAAAAAAAAAAAAAAAAFJtrKWCVUMARGWSqpp0XDMA
AAA3TGFlWz4zAAAtUXqnrYVvAAAAAAAAAAAAAAAAAAAAM1STkWs3IgBKb6Gp
kGAyIREAACI9WGNYRTUAAERYibSgcVUAAAAAAAAAAAAAAAAAAAAARGaBWUQA
AF+EqKOGU0VCRlVVUUpRUEpSPwAARFd9ooRjAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAABEW5Gqn3hZio2EgoF3eXtxb3NgRQAAVWuTdVUAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAElom6yMZMvIycTCwLq6urexsJxfAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAVXyloZeU0s7Q09PS1NTU29Tbw24AAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAiIKAgZCKk5OZm56XoaGko66OSwAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAERMSVRVVVUA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAABmXFVVVQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAASVReWVeHiHFWYHNeVQAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAA2TV1dUGO6v7Cbinl/gWZORAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAACxKWF9TR6nHzM3MwrmqloJ+VQAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAT0BAPwAANDtOWV9WSIONlqWzwtDUzLl4
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABdWVlEIgAANkRbWFlPTU1da3aB
j6Ksq3oAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAARFRZWUwtAAAiPVJYWllG
L0BFNklXY4Z8WwAARFpXVQAAAAAAAAAAAAAAAAAAAAAAAABWSVdWUzMWAAA1
RFVeW003MyIiRXiakF4oADZBVV9XVVUAAAAAAAAAAAAAAAAAAAAAAG1FWV1V
QjMAAAA4TFpdVUc6ADNWi6KETAAAMEhLVV5YVQAAAAAAAAAAAAAAAAAAAABV
bzVTXV5QPjMAAChJVV1gV15RLWKcnXI8AABAV0BUXl5XVQAAAAAAAAAAAAAA
AAAAAFd3ikpWWFA8LgAANTtOaqqukIJyb3BlQ1dwW1ZXM0NOV2BSRkQAAAAA
AAAAAAAAAAAAZoiROlNeWU4uAAAAOEub5/HOu7GnpJ2WgH5tWWNcQE9LVF5O
RAAAAAAAAAAAAAAAAERilplUT15dWT4RAAAiTZPm583KzM7OzMu+s6mckY1+
aVZdXlRORAAAAAAAAAAAAAAAVGqfmF5IWF9XRi4AAABHZp6Zjpulrrq2vcjO
1NHFvreppqGUiYFeSQAAAAAAAAAAAABrf6SPTTpPXWhePyIAAABFUU1UXml0
eoCFkJynrbzDzdbSzcrGvKdwMwAAAAAAAAAAAF+JoYRINFCFsaVmPwAAAAAR
GjlofUs6RVdcZ3J3hoyPmaSvu8bKv2UzAAAAAAAAAAAAX5Wed08pWrft2Yk6
IQAAAAAiUo6hc0kzSzNBMEBHUFthbHl/go2JZDMAAAAAAAAAAABwnpJpVF1q
tejOfjwsIgAAACxck5dsJQAAUoijh1UiIiIsQ1VQVGYAAAAAAAAAAAAAAICe
iGeXc3aarphgXWJXbkxVSlZhbFpMAERll594QwAAAAAAABERAAAAAAAAAAAA
AAAAg5x+nL2DfqOafEp/sqd4X3pvbGRhWVJFOzxklWs2ORYAAAAAAAAAAAAA
AAAAAAAAAABLYWBmUVOSqJRuarnFn1am49S0opySkYuBdXhwY1lGLGZcUz4p
IgAAAAAAAAAAAAAAAAARHDRJaJekil9Ea3VqaL357dDO0MbEwLWrqqSYjYiB
fHJuYFBeVzMAAAAAAAAAAAAAAAAAAFVzpZ9/UzNVWDtuoNnIt8LAxM3IzNXX
1M7Bwr6xpqCXlox3ZQAAAAAAAAAAAAA6TkQ9WoSrknJDAABHVoKjmYR4c3+I
kZeboq6wtcDJ0tvSytHLxcuOYgAAAAAAAAAAVU1RQVdmlZ+IaEBEMDVnjaaU
ckRVRj1PWmVna3WChoaSnaKkrrnEyIRVAAAAAAAAAABVW1dCRXCenX1WcXBj
SHmYn45iOzMRHS83UF5VSDg9TV1cZ291eIWNbVUAAAAAAAAAAEVfV0xXhqSV
cESnv6Rhg6KafEloQzYAIjBIWF5TOCk+UR42Z42fi2ZGAAAAAAAAAAAARFte
V2KSoo5jTqfCdl6PopNuRlRTRDMAIj1VXVdKMwAAOk97naCBRgAAAAAAAAAA
AAA4Wl5PapyfgVFPa3Rdcpqhj1xEWV5KPwAAIUhSWVI/ACI3XY+mlXEzAAAA
AAAAAAAAADNTW1qCoJRzRClXP1OCn5d5SDZZWVs8MwARN1NeWEo0MlF5maKC
OQAAAAAAAAAAAAAAAE1YY4ijjmU9AABLZJGljGo2NkZXWUY0AAAqRFdeWT01
ZoqlkVxAAAAAAAAAAAAAAAAATFNmmJ+DWSgAM0JvmqKBYCIiRlZdWD8AADM+
TF5ZPE50n6eDVCIAAAAAAAAAAAAAAAA8VW2imXhTIgBEVICslXZIAAA1TV5W
QzMAADVHWFpFXI2jjlocAAAAAAAAAAAAAAAAAE9bg6qSbjQAAD1nkaeKZycA
AABDWl1RPTMAADNNRkR2nKCDVQAAAAAAAAAAAAAAAAAATmCOo4pgMQAAQ3ic
oH1MOgAAADFQWltFPQAAM0RMXYumlXIAAAAAAAAAAAAAAAAAAABSaJyhe0c2
ADNXh6iZbU8zAAAAN0teV005MwAALUx0mZ1+bwAAAAAAAAAAAAAAAAAAADNU
iINrNyIAOWeZqYtYMiERAAAiPVNWWEU1AAA+Uoelj2lVAAAAAAAAAAAAAAAA
AAAAAERbc1lEAABOdaSfgVNFQT5VSFFESUtFSD4AADNGd5x9XAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAARFmDqZZsVYOCeHt7dndwZmdrUzYAAERkk3VVAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAABJZZCkjGTCwLy6t6+zsbCpqaCXXwAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAFV2pKGSi8jGw8rGyMzLy8zMzLVu
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIiCgH+ChYSNiZOTlpmb
l6GhiksAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AABETElRTFVVAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAd21mZmYAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAFpkZltklpmCYmyEb2YAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAQFtmZVhsydDGtKONkJJ3X1UA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAtU2ZvZE622N3r7tzK
u62fkGYAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAFFCSEUAADQ8WGdwX0qU
nrLEy9Xt5d7KiQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAYGZgRjMAAEdP
XmlqWVdbaXeHmaezvbuEAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAFVgampY
PgAAM0VdaWthSTxERzxPaHSei1sAAFVraGYAAAAAAAAAAAAAAAAAAAAAAAAA
Z1poZ1pAGQAANUhlZ2hdSEQiIkiLrKJsKAA4SmZwY1tmAAAAAAAAAAAAAAAA
AAAAAAB+UGZqZE9EAAAAQ1hjbmRVSwAzZaa5lV0AADZZXGZvZVkAAAAAAAAA
AAAAAAAAAAAAVXw4XGZtWkEzAAApUWNuZ2VvUS1uta6DTQAAQ2RNXGhrYmYA
AAAAAAAAAAAAAAAAAABZg5pUYWlhSjwAADU8WXu9xKGRg4KDcFBmcGZjYjVQ
XGdvWldVAAAAAAAAAAAAAAAAAHedqEFkbmRTNwAAAENZrO/539DTybuxp4+E
emJ0Yk5TS2JtXEsAAAAAAAAAAAAAAABVc6upYVlsZFlHIgAAIlqk8PXm5u7w
8Obcz8TAt62ej35gbG1iWUQAAAAAAAAAAAAAAGV7uappSmZmZVY/AAAAWHav
qaGvuMLQ1t3m7Ovw5t3VyLyypZqSaksAAAAAAAAAAAAAfJC1oFhEYG55b0sz
AAAARWBWXmt6gYuXoqS0uMLP2eDs6ePg2c7EgUQAAAAAAAAAAABnpLWVVjxf
lsi3clAAAAAAESNEc5JSQVFhbXiDjpahqbK8x9Ta29V2RAAAAAAAAAAAAG6q
sIhfK2rR9+mbSzEAAAAAIlagtINJO1A5TTVGVl9ucn2KlJ+qmnVEAAAAAAAA
AAAAgrend1tne8nx4pdDLCIAAAAxarSueSYAAFmZvJlhIiIoPUZVUGF3AAAA
AAAAAAAAAACTuJxxrIGHs8OvbmZrYX9PWltnbnheTABEdqyziVQAAAAAAAAR
EQAAAAAAAAAAAAAAAJa8lLbai4y6so1VicO4gmuWhn1xcGdjVkBAb6d8Rjkk
AAAAAAAAAAAAAAAAAAAAAAAATnBsc11gp7ylf3rO16pft/bpyLazsKWckoV/
dWllSztmXFNEMCIAAAAAAAAAAAAAAAAAERw3WnmvuptvT3yAeXXO//bu5ubf
29HMxMC1saeckYWAeGxcbWIzAAAAAAAAAAAAAAAAAABmhLawkGQ+VVxMf7Hm
18vW2dnh5e3z7enj3NnPwsG+saedh3IAAAAAAAAAAAAAS19VPmGgwaOBRgAA
S16TtLOViYWQmaKusLO/ytPb2+Pw8+vm3t/dmmYAAAAAAAAAAGZeYlJgcau4
mW9GRDo5c569pYNPZlJGWGd2cnyGkZaao665xMjO1dyVXgAAAAAAAAAAZmxo
U0yBtq+OZIF8cUt+qb2fcz8zER0vP2FiZlk9RFZjbXiDfomWnnlVAAAAAAAA
AABWZWZaW5y6poBKwtCzcZS6uI1ZeFNEACI9VWhvZD0vRFMgP26euahzRwAA
AAAAAAAAAE5lZltrprWfcVW+04Fuo7qleVJiYUozACI9XmpoVUQAAEJWi7G1
klcAAAAAAAAAAAAAOFtmW3uwspJiYHt+boOtt6FpVWpsU0oAAC1UX2pfUAAi
P26gu6uCRAAAAAAAAAAAAAAzYmthkra0hlAsZkBWkrmvjVVAYGdeSkQAIjlb
b2daQkFfiqy8k0oAAAAAAAAAAAAAAABYaW2Zu6h3RwAAUnGitqF9QEdXaGpX
NgAALFBkZ2NIQHCbtqJtUQAAAAAAAAAAAAAAAE9fc6q2lmgoADNJgLSzlmoi
M01naV4/AABESFVmZ0RZhbC4kF4zAAAAAAAAAAAAAAAAS2OHtrGJVyIAU2Sa
wamLSwAARFtmZVJEAAA1SmVqVGyevatqKwAAAAAAAAAAAAAAAABPXpi8o388
AABOeKi6m3IpAAAAVGZrYkozAAA+VVBPh7G1lGYAAAAAAAAAAAAAAAAAAFlt
pr2baj8AAFSJs7OOV0AAAAA3W2tkTj0AADNMVmicu6p7AAAAAAAAAAAAAAAA
AAAAY3mwtYxVQwBEZ5i6qn1cMwAAAEBSYmVdPzUAAC9bhKy8lXkAAAAAAAAA
AAAAAAAAAABEYp6ieTwiAEpwqrqcYjYhEQAAIj1gZ2lWOAAARF2YtqB6ZgAA
AAAAAAAAAAAAAAAAAABEZpFjRAAAX4a5tJJgTUlGXllVUVFUVlk/AABEV4it
jm0AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAEVkn7upfWqZkpKKjYWJf3V4c2FH
AABVdaSGZgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAVXSrvZ113NnR08vLy8jB
xLy6rHAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABVhLWyo5zg2+Hb
29vi3OPm5eDTfwAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACZiJGQ
k5aVoZqkpKeqrqu0s51cAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAARExVVV1aVQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD65ublUAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA/+v////v39/dkNTUP
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANj/////////
/////5lzc3MHAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAUv//
/////////////////7ZmAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABG7+8/
AAAT4f///////////////////5MAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AOj//+1EAABV////////////////////kwAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAABw9////6IAAB23//////+oVn////////+TAABCqalrAAAAAAAAAAAA
AAAAAAAAAAAAAKr/////9lsAAH74/////48JBhnA////+H4AI/D///ZxBgAA
AAAAAAAAAAAAAAAAAAAAqv//////ZAAAANH//////z4AMNT////RAABx////
//9xAAAAAAAAAAAAAAAAAAAAAETf///////GGAAAV///////t6DO/////1cA
AOj///////ZrAAAAAAAAAAAAAAAAAAAAbP////////88AAAO6///////////
////Kx0d6v////////RvBQAAAAAAAAAAAAAAAABs/////////+gAAABN////
//////////////////////////9zAAAAAAAAAAAAAAAAHMT/////////9mUA
AB+6//////////////////////////////lqAAAAAAAAAAAAAAAu////////
////qgAAAID7//////////////////////////////NhAAAAAAAAAAAAAC7/
///////////FIwAAAIjD/////////////////////////////6ERAAAAAAAA
AAAAm/////////////9sAAAAAB5v3P///////////////////////////zYA
AAAAAAAAAADw////s7n//////+MlAAAAAAbE////0zEx8//////39P//////
//uJCgAAAAAAAAAAAPD//////////////2s0AAAAg/////+RAADw////4kIW
tbW1teDCfgAAAAAAAAAAAAAA8P///////////////9uIiIjF/////wgAX/j/
//+bAAAAAAAARRYAAAAAAAAAAAAAAADw////////////////////////////
x8Xu////4icEBAAAAAAAAAAAAAAAAAAAAAAAALLZ////////////////////
///////////////////oQkJCQkIMAAAAAAAAAAAAAAAAADZ/f7L/////////
/////////////////////////////////5eAOgAAAAAAAAAAAAAAAAAAZP//
///uQkJN//////////////////////////////////90AAAAAAAAAAAAADY6
Olf9/////+gAAA////////////////////////////////////01AAAAAAAA
AABT9///////////83d3gP///////4jC/////////////////////////zYA
AAAAAAAAALL/////////////////////////ACRKcP//////////////////
//+cEAAAAAAAAAAAsv/////////////////////////zfQADe//////aDAwM
Zf//////8F0AAAAAAAAAAACy//////////////////////////+5HgBB3f//
/+lDAABd//////+yAAAAAAAAAAAAALL///////////////////////////9G
AACy/////5sAFcv/////9TkAAAAAAAAAAAAADvH////////wYRVj///////0
6/////AHAA7x////929D///////QAAAAAAAAAAAAAAAA8P///////6IAAFX/
/////7mD/////1YAAKL//////////////y4AAAAAAAAAAAAAAADw////////
ogAPw//////6Qi/X////yQAAEKD///////////+gEAAAAAAAAAAAAAAAANf/
/////+QSABf///////cAAKj////5ewAAXvT/////////9F4AAAAAAAAAAAAA
AAAALv//////4AAAX///////sAAAAP////+2HQAAqv////////ZrAAAAAAAA
AAAAAAAAAAAu//////9qAADY//////82AAAA//////9NAABC8f//////6AAA
AAAAAAAAAAAAAAAAAC7//////x8AD9z/////iAgAAABa/////+UNAADV////
//9DAAAAAAAAAAAAAAAAAAAAH83////IFACb//////+jVj8AACnT/////1EA
ACf/////yxoAAAAAAAAAAAAAAAAAAAAALWtraycAAJv/////////4pSUlMj/
////0QAAEKX///9kAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABM7f//////////
///////////3eQAAEi0tLRIAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAF3/////
//////////////////+TAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
Ksj//////////////////////5MAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAINn39/f39/f39/f////////3jgAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAALm5ubm5uQsAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAA}

show_image sun base64 "true color compressed" {
WaZqlQAAACIAAAAhAAAAGAAADSYAAAACAAAAAAAAAACAKQDJp6eACP/Jp6eA
VgCADv/Jp6eAUADJp6eAEf/Jp6eATQDJp6eABf/cxMSncHCACP/cxMSAQQDJ
p6eAA//MzKdwcIAI/9zExKdwcIAI/9zExIBBAIAF/+/m5qdwcIAI/9zExKdw
cIAI/9zExIA1AMmnp4AC/9zExKdwcIAF/+/m5qdwcIAI/9zExKdwcIAI/9zE
xIAyAP/MzIAF/9zExKdwcIAG/8zMp3BwgAj/3MTEp3BwgAj/gCwAyaen7+bm
gAIA/8zMgAX/3MTEp3BwgAX/7+bmp3BwgAn/zMyncHCAAv+AAgD/zMzcxMSA
IwDJp6fv5uaABf+AAgD/zMyABf/cxMSncHCABf/v5uancHCACf/MzIACAP/M
zIAF/8mnp4AdAMmnp//MzIAF/6dwcNzExIACAP/MzIAF/9zExKdwcIAF/+/m
5qdwcIAF/4ACAP/MzIAF/9zExKdwcNzExIAXAMmnp+/m5oAF/6dwcNzExIAF
/4ACAP/MzIAF/9zExKdwcIAF/8mnp4AFAP/MzIAF/9zExKdwcIAF/9zExIAR
AMmnp//MzIAF/6dwcNzExIAI/6dwcIACAP/MzIAO/8mnp4ACAP/MzIAF/9zE
xKdwcIAF/+/m5qdwcIAOAMmnp+/m5oAF/6dwcNzExIAI/6dwcMmnp8mnp4AC
AP/MzIAL/4ACAP/MzIAF/9zExKdwcIAF/+/m5qdwcMmnp8mnp8mnp4AFAMmn
p//MzIAF/6dwcNzExIAI/6dwcO/m5oAI/4ACANzExIAC/+/m5oACAP/MzIAF
/9zExKdwcIAF/+/m5qdwcIAL/9zExIACAP/MzIAF/6dwcNzExIAI/6dwcO/m
5oAO/4AIAP/MzIAF/9zExKdwcIAF/+/m5qdwcIAR/7KIiIAI/9zExIAI/6dw
cP/MzIAF/6dwcIAI/4AIAIAF/+/m5qdwcIAF/+/m5qdwcIAI/9zExIAI/7KI
iIAR/6dwcO/m5oAF/6dwcNzExIAF//fv74AIANzExIAL/+/m5qdwcIAI/9zE
xKdwcIAI/6dwcMmnp4AL/6dwcO/m5oAF/6dwcNzExIAI/4ACAPfv74AF/4AC
AIAJ/8zMp3BwgAj/3MTEp3BwgAj/yaengAUAp3BwyaenyYACp3Bw/8zMgAX/
p3Bw3MTEgAj/gAIAgA7/gAIAyaenyYACp3BwgAj/3MTEp3BwgAj/yaengA4A
p3Bw7+bmgAX/p3Bw3MTEgAj/gAIAyaengBH/gAUAgAj/3MTEp3BwgAj/yaen
gBEAsoiIgAX/p3Bw3MTEgAj/gAIAp3BwyaengAX/3MTEp3BwgAj/gAIAgAX/
3MTEp3BwgAj/yaengBcAyYACp3Bw3MTEgAj/gAIAgAP/zMyncHCACP/cxMSn
cHCACP+AAgDcxMSncHCACP/Jp6eAHQCyiIiACP+AAgCncHCABf/v5uancHCA
CP/cxMSncHCABf/v5uaAAgCACP/Jp6eAIwDJp6eAAv+AAgCAAv/cxMSncHCA
Bf/v5uancHCACP/cxMSncHCACP+AAgCAAv/Jp6eALADv5uaABf/cxMSncHCA
Bv/MzKdwcIAI/9zExKdwcIAI/4AyAMmnp+/m5oAF/9zExKdwcIAF/+/m5qdw
cIAI/9zExKdwcIAC/8mnp4A1AKdwcP/MzIAF/9zExKdwcIAF/+/m5qdwcIAI
/9zExIA+ALKIiP/MzIAF/9zExKdwcIAG/8zMp3BwgAX/yaengEEAsoiI/8zM
gAX/3MTEp3BwgAX/yaengAIAsoiIgEcAsoiI/8zMgA7/yaengFAAsoiI/8zM
gAv/soiIgFMAsoiI7+bmgAL/9+/vyaengCwA}

show_image sun base64 "true color uncompressed" {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}

show_image sun base64 "indexed compressed" {
WaZqlQAAACIAAAAhAAAACAAABGIAAAACAAAAAQAAABsAp//EcMzmiO8Ap//E
cMzmiO8Ayf/cp//vsveADQABgAICAYAcAIAEAgGAGgABgAUCAYAZAAECAgME
gAICA4AVAAECBQSAAgIDBIACAgOAFQACAgYEgAICAwSAAgIDgBEAAQIDBAIC
BgSAAgIDBIACAgOAEAAFAgIDBAICBQSAAgIDBIACAoAOAAEGAAUCAgMEAgIG
BIACAgUEAgAFA4ALAAEGAgIABQICAwQCAgYEgAICBQAFAgIBgAkAAQUCAgQD
AAUCAgMEAgIGBAICAAUCAgMEA4AHAAEGAgIEAwICAAUCAgMEAgIBAAAFAgID
BAICA4AFAAEFAgIEA4ACAgQABYAEAgEABQICAwQCAgYEgAQAAQYCAgQDgAIC
BAEBAAWAAwIABQICAwQCAgYEgAIBAAABBQICBAOAAgIEBoACAgADAgYABQIC
AwQCAgYEgAMCAwAFAgIEA4ACAgQGgAQCgAIABQICAwQCAgYEgAUCB4ACAgOA
AgIEBQICBIACAoACAAICBgQCAgYEgAICA4ACAgeABQIEBgICBAMCAgiAAgAD
gAMCBgSAAgIDBIACAgQBgAMCBAYCAgQDgAICAAgCAgCAAgIFBIACAgMEgAIC
AQAABAEBBAUCAgQDgAICAIAEAgABAQSAAgIDBIACAgGABAAEBgICBAOAAgIA
AYAFAgAAgAICAwSAAgIBgAUABwICBAOAAgIABAECAgMEgAICAAICAwSAAgIB
gAcAAQQDgAICAAIFBIACAgMEgAICAAMEgAICAYAJAAeAAgIABAICBgSAAgID
BAICBgCAAgIBgAsAAQIAAgMEAgIGBIACAgMEgAICAAIBgA4ABgICAwQCAgUE
gAICAwSAAgKAEAABBgICAwQCAgYEgAICAwQCAYARAAQFAgIDBAICBgSAAgID
gBQABwUCAgMEAgIFBAICAYAVAAcFAgIDBAICAQAHgBcABwWABAIBgBoABwWA
AwIHgBsABwYCCAGADgA=}

show_image sun base64 "indexed uncompressed" {
WaZqlQAAACIAAAAhAAAACAAABGIAAAABAAAAAQAAABsAp//EcMzmiO8Ap//E
cMzmiO8Ayf/cp//vsvcAAAAAAAAAAAAAAAAAAAECAgIBAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAACAgICAgEAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAABAgICAgICAQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAQICAwQC
AgIDAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAECBQQCAgIDBAICAgMAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAgIGBAICAgMEAgICAwAAAAAAAAAAAAAAAAAAAAAA
AAECAwQCAgYEAgICAwQCAgIDAAAAAAAAAAAAAAAAAAAAAAAFAgIDBAICBQQC
AgIDBAICAgAAAAAAAAAAAAAAAAAAAAEGAAUCAgMEAgIGBAICAgUEAgAFAwAA
AAAAAAAAAAAAAAEGAgIABQICAwQCAgYEAgICBQAFAgIBAAAAAAAAAAAAAAEF
AgIEAwAFAgIDBAICBgQCAgAFAgIDBAMAAAAAAAAAAAEGAgIEAwICAAUCAgME
AgIBAAAFAgIDBAICAwAAAAAAAAEFAgIEAwICAgQABQICAgICAQAFAgIDBAIC
BgQAAAAAAAEGAgIEAwICAgQBAQAFAgICAgAFAgIDBAICBgQBAQEAAAEFAgIE
AwICAgQGAgICAAMCBgAFAgIDBAICBgQCAgICAwAFAgIEAwICAgQGAgICAgIA
AAAFAgIDBAICBgQCAgICAgIHAgICAwICAgQFAgIEAgICAAAAAgIGBAICBgQC
AgIDAgICBwICAgICAgQGAgIEAwICCAAAAAMCAgICBgQCAgIDBAICAgQBAgIC
AgQGAgIEAwICAgAIAgIAAgICBQQCAgIDBAICAgEAAAQBAQQFAgIEAwICAgAC
AgICAgABAQQCAgIDBAICAgEAAAAAAAQGAgIEAwICAgABAgICAgICAAACAgID
BAICAgEAAAAAAAAHAgIEAwICAgAEAQICAwQCAgIAAgIDBAICAgEAAAAAAAAA
AAEEAwICAgACBQQCAgIDBAICAgADBAICAgEAAAAAAAAAAAAABwICAgAEAgIG
BAICAgMEAgIGAAICAgEAAAAAAAAAAAAAAAABAgACAwQCAgYEAgICAwQCAgIA
AgEAAAAAAAAAAAAAAAAAAAAGAgIDBAICBQQCAgIDBAICAgAAAAAAAAAAAAAA
AAAAAAAAAQYCAgMEAgIGBAICAgMEAgEAAAAAAAAAAAAAAAAAAAAAAAAEBQIC
AwQCAgYEAgICAwAAAAAAAAAAAAAAAAAAAAAAAAAAAAcFAgIDBAICBQQCAgEA
AAAAAAAAAAAAAAAAAAAAAAAAAAAABwUCAgMEAgIBAAcAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAHBQICAgICAQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAcFAgICAgcAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABwYCCAEAAAAA
AAAAAAAAAAAAAAA=}


show_image tga base64 "true color compressed" {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}

show_image tga base64 "true color uncompressed" {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}

show_image tga base64 "true color compressed with alpha" {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}

show_image tga base64 "true color uncompressed with alpha" {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}

show_image tiff base64 uncompressed {
TU0AKgAAHggAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAADb///bttu2tv+Sttu2ttvb2/8AJEkAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD////b//+229u2
29vb2/+2ttttttu2ttu22//b//8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAJAC229skbZLb//+2tv8AJEkkbZJtkrbb///b//9tktvb//8A
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACSttu2
29u2ttvb2/8AAAC229skbZIAACSSttu229uSttsAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACQAAAAAAAAAAAAAJCQAJCSSttsAAACS
ttvb//8AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAASUkAAABJbZIAACQkAACStrYASQDb2/8AAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD/////29v//9v/29v///8AAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AABtJACSbSS2bUnbkm22km3btm22kkn/25L//9ttJAAkAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAACSSSSSSSTbkm2SSSSSSSSSJACSSSS2
bUm2bUm2bUn/27b//9v/////////////27b/tm3btm3//9v/27b//////////////9skAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAkAABJAAAAAACSSSRJAABtJABtJACSbSSSSSTbtm22km3btm3btm3btm22km3/
km22bUn/25L/27b/25Lbtm3btpLbkm3/27bbtpLbtm22km3/25L/27b//7b/27b//////9v/
/////9v/////tpL/km1tJAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAABtAABJAABJAABJAACSJCRJ
AABtJABtAACSSSRtJACSbSSSSSTbkm222222km22bUm2kknbtpLb25K2km22kknbtpLbtm3b
tpLbtm22km3btpK2km3btpLb25L//9v/25L/25Lb29v//7b/27b//7b/27b/25Lb29v/27bb
tpL//9v//////7b///////////////+2bUkAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAABJAABJAABJAABJAABJJACSSSSSJABtSQCSSSSSSSSSSSSSJCSSSSS2
bUmSSSSSJABtJABtAACSSSS2kkm2bUnbkkm2kkn/25K2kknbtm22km3bkm22km3/tm3btm3b
tm3btpLbkm22km22kkm2kknbtm3bkpL/25L/27b//7b/27b//7b/////tm3btpL/27b/tm3b
tpLbtm3//7b/25L//9v/////////////////////2/8AAAAAAAAAAAAAAACSSSQkAABtJABJ
AABtJABJAABJAABJAACSSSRtSQBtJABtSSS2kkmSJCSSJCSSSSSSSSSSSSSSSSSSSSSSSSRt
JACSSSSSSSS2bUm2km3//9u2bUm2bUm2bUm2SSS2bUm2bUm2km2SSSSSSSTbtm3btpL/25K2
km3btkm2km3btm22km3bkm3btpL/25K2tm3bkm22km3btm22bUm2bUm2bUm2kkm2bUnbtm22
tm3bkm3b25L//9v///////////8ADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAAD
AAAetgEDAAMAAAABAAEAAAEGAAMAAAABAAIAAAERAAQAAAABAAAACAESAAMAAAABAAEAAAEV
AAMAAAABAAMAAAEWAAMAAAABACgAAAEXAAQAAAABAAAeAAEaAAUAAAABAAAevAEbAAUAAAAB
AAAexAEcAAMAAAABAAEAAAEoAAMAAAABAAIAAAAAAAAACAAIAAgSwAAAAAQAABLAAAAABAAA}

show_image tiff base64 "packbits compressed" {
TU0AKgAABAqBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEA
wQCBAMEAgQDBAIEAwQCbAAvb///bttu2tv+Sttv/tv7bA/8AJEm8AKEA/v8G2///ttvbtv3b
Df+2ttttttu2ttu22//b//+/AKMACCQAttvbJG2S2////7YR/wAkSSRtkm2Sttv//9v//22S
/9v//8IAoQADkrbbtv/b/7b+2wD//gARttvbJG2SAAAkkrbbttvbkrbbwgCcAAAk9wD/JAUA
JCSSttv+AAGStv/b///CAJcA/0n+AAJJbZL/AP8k/wAIkra2AEkA29v/wgCBAMEAgQDBAIEA
wQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQCBAMEAgQDBAIEAwQD9///b//8B2///2/7/gQDQ
AB5tJACSbSS2bUnbkm22km3btm22kkn/25L//9ttJAAkgQDgACOSSSSSSSTbkm2SSSSSSSSS
JACSSSS2bUm2bUm2bUn/27b//9v3/w3btv+2bdu2bf//2//btvb/AdskigADJAAASfwAU5JJ
JEkAAG0kAG0kAJJtJJJJJNu2bbaSbdu2bdu2bdu2bbaSbf+SbbZtSf/bkv/btv/bktu2bdu2
ktuSbf/bttu2ktu2bbaSbf/bkv/btv//tv/btvz/ANv8/wDb/f8GtpL/km1tJLUAYm0AAEkA
AEkAAEkAAJIkJEkAAG0kAG0AAJJJJG0kAJJtJJJJJNuSbbbbbbaSbbZtSbaSSdu2ktvbkraS
bbaSSdu2ktu2bdu2ktu2bbaSbdu2kraSbdu2ktvbkv//2//bkv/bkv7b//8Mtv/btv//tv/b
tv/bkv7bCP/bttu2kv//2/z/ALb1/wK2bUnXAH9JAABJAABJAABJAABJJACSSSSSJABtSQCS
SSSSSSSSSSSSJCSSSSS2bUmSSSSSJABtJABtAACSSSS2kkm2bUnbkkm2kkn/25K2kknbtm22
km3bkm22km3/tm3btm3btm3btpLbkm22km22kkm2kknbtm3bkpL/25L/27b//7b/2wO2//+2
/f8Ztm3btpL/27b/tm3btpLbtm3//7b/25L//9vx/wHb//UAAZJJ/yT/AH9tJABJAABtJABJ
AABJAABJAACSSSRtSQBtJABtSSS2kkmSJCSSJCSSSSSSSSSSSSSSSSSSSSSSSSRtJACSSSSS
SSS2bUm2km3//9u2bUm2bUm2bUm2SSS2bUm2bUm2km2SSSSSSSTbtm3btpL/25K2km3btkm2
km3btm22km3bkjBt27aS/9uStrZt25JttpJt27Zttm1Jtm1Jtm1JtpJJtm1J27ZttrZt25Jt
29uS///b+P8ADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAADAAAEuAEDAAMAAAAB
gAUAAAEGAAMAAAABAAIAAAERAAQAAAABAAAACAESAAMAAAABAAEAAAEVAAMAAAABAAMAAAEW
AAMAAAABACgAAAEXAAQAAAABAAAEAgEaAAUAAAABAAAEvgEbAAUAAAABAAAExgEcAAMAAAAB
AAEAAAEoAAMAAAABAAIAAAAAAAAACAAIAAgSwAAAAAQAABLAAAAABAAA}

if 0 {
show_image tiff base64 "pixarlog compressed" {
TU0AKgAAA3Z4nO2aIWzbUBCGz7HjYyVTNDAp0CgKDRorCauiSGFjRmWRxs0rlQ2FlUWKrLKQ
sIDK1BoyjFRUjZSF7f13eYnjpZOmdeuU3Wc1se+dn17uf/ee5SuRYRiGYRiGYRiGYRiGYRiG
YRg/YxVt3EG04IpLJkqocJ8VD6S1746ELsOP8UNMxO1J9MbDPQs2EvOKcV66uOsZoo34lwxb
JraMfEsif+S06Lrj74/5XNi2iG6jdbCKruMRz3gdfKUbF/OFi/c0vHbzfNvqBDl1giENKHdt
492dI/7sdIElMRV+i0l0G62igcx3ZMBhnUFc0ebV4fZzeBfDfx2ofyGr1DycOt+3/RXnALIB
63rTqnYo9SVYBxptVS11+WDz//WZh/MQ8Z5E8/AiwjdmPnTg9qfg1sV+Hn7gabiK3gWbSDPl
rcdsGIZhGIZhGIZhGIZhnCebXc1IKxeJ2LSGkbija+8G/zAXEd4LDqkTbFuo091T312l8iYd
79JhK7ngGY+dNSfoklKP38eX4VOMt73NN77Gr4B3tP4NOeLbCWY84qfYt18x4ntPUCd133XG
zv+RSOp5RD57DlRSiSo5Y59RY1HQV2fH1KRyunbDy/D/0VRjr3HVirQqMuJJxO2UvDY+R4hm
DKUe3RXywudIHbVrdBe8lNzJJfK08y2kQr5k7RfXS0ZeZaJNIfYl+35S8dF7j1fI8dHV8fmC
P7P23OO7+Dl8/idVvZCYz7hpz6VyNHKz/6ACxj8kqLNk5MWxIrkoktKhP0Re49uX+qtWxUtn
1cjP2Nvhi8/h7vyg7FDUXLKe+7o6tEKs/bgrGUEiFfdTeqji6qN1evSaiLYH34X4/piVaOnx
Qzx1Gr72f03MX9xdEdlcVh5dhZaMLBkS1XJCuZJxo1XXKOwWsOld6nPQbiC9asT70l+5+93Q
cMFY//rSk1eBaj7IgaoxW3wGlXu710X7QetwpxnRqTxSEHlEupJZk+x3v2xnIRk1VMvk3oJp
32tz7a2Du7svVNl0tUFdmtt+NiNWOuf1vE6+m/8aUR/5bSuTqBYu06/2UTjeLU7h8+h+r6nf
ZfAMMKYbUaTH3+Lj+1Tf+n4E/6K2c+kKqS26Py0k7xDTmcz+tPZ04VUmyTLvX4p/wdR4Gsm4
uV54beCT1Z5edCVNpc/TSn0HoOsh4gAOAQAAAwAAAAEAQAAAAQEAAwAAAAEAKAAAAQIAAwAA
AAMAAAQkAQMAAwAAAAGAjQAAAQYAAwAAAAEAAgAAAREABAAAAAEAAAAIARIAAwAAAAEAAQAA
ARUAAwAAAAEAAwAAARYAAwAAAAEAKAAAARcABAAAAAEAAANuARoABQAAAAEAAAQqARsABQAA
AAEAAAQyARwAAwAAAAEAAQAAASgAAwAAAAEAAgAAAAAAAAAIAAgACBLAAAAABAAAEsAAAAAE
AAA=}
}

if 0 {
show_image tiff base64 "logluv compressed" {
TU0AKgAAAAgADgEAAAMAAAABAEAAAAEBAAMAAAABACgAAAECAAMAAAADAAAAtgEDAAMAAAAB
h3QAAAEGAAMAAAABAAIAAAERAAQAAAABAAAAAAESAAMAAAABAAEAAAEVAAMAAAABAAMAAAEW
AAMAAAABACgAAAEXAAQAAAABAAAAAAEaAAUAAAABAAAAvAEbAAUAAAABAAAAxAEcAAMAAAAB
AAEAAAEoAAMAAAABAAIAAAAAAAAAEAAQABASwAAAAAQAABLAAAAABAAA}
}

show_image tiff base64 "jpeg compressed" {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}


show_image tiff base64 "deflate compressed" {
TU0AKgAAAah4nO2Y643EMAiEXQzNUIyboRiaoRh2ADuP+7N70knZkxhZlmNH2fkMRPGO0Wq1
Wq1Wq9Vqtb5L5m5qqi7Rm5kP4qdNfSSHc3dYDtfmMD8TQS3mn3b3TjRgm6YEgsaeYzwRg4TC
9PcjRMLYzpmxcABW8+ifNviJ6DIktLKN/vv3/xRHwXJuPoV5HTwqKK1Wq9VqtVqtz+VxwMjv
8ej/39fUpCGTdLLJVJmm6NlNQIQlev+AhyVMaDBfA6GYAU41nFrzqLSFSw3GjJf6ZeEpUvwu
bxDOcFREAkpXRKph7IjRzOjAvEnOC9jzcC7rnlp13YB29ip4wvxT1Jn+qwlthMQ5QWDbwlsE
RTis4uQX+cZl+zCPQfWWGZgsEkfHhaML3KyQN901lIjwL/6a4GvL5EEK4QErneKSKqPW0qaD
+awarnpJlkTIOnI9Aic1o1LsuEHuMdIyHTgrPyVTVKsM/YfuRzY4odxz3v2KBfaf021aDQqi
E2q3SYs0QxN1cdSOHnW0i8uWyYoR79dFIK95nedr5HhObRRmctXuUC9uNR31AAAOAQAAAwAA
AAEAQAAAAQEAAwAAAAEAKAAAAQIAAwAAAAMAAAJWAQMAAwAAAAGAsgAAAQYAAwAAAAEAAgAA
AREABAAAAAEAAAAIARIAAwAAAAEAAQAAARUAAwAAAAEAAwAAARYAAwAAAAEAKAAAARcABAAA
AAEAAAGfARoABQAAAAEAAAJcARsABQAAAAEAAAJkARwAAwAAAAEAAQAAASgAAwAAAAEAAgAA
AAAAAAAIAAgACBLAAAAABAAAEsAAAAAEAAA=}

show_image xbm text bitmap {
#define flagup_width 48
#define flagup_height 48
static char flagup_bits[] = {
   0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe0, 0x7f, 0x00,
   0x00, 0x00, 0x00, 0xe0, 0x7f, 0x00, 0x00, 0x00, 0x00, 0xef, 0x6a, 0x00,
   0x00, 0x00, 0xc0, 0x7b, 0x75, 0x00, 0x00, 0x00, 0xe0, 0xe0, 0x6a, 0x00,
   0x00, 0x00, 0x30, 0x60, 0x75, 0x00, 0x00, 0x00, 0x18, 0xe0, 0x7f, 0x00,
   0x00, 0x00, 0x0c, 0xe0, 0x7f, 0x00, 0x00, 0x00, 0x06, 0xe0, 0x04, 0x00,
   0x00, 0x00, 0x03, 0xe0, 0x04, 0x00, 0x00, 0x80, 0x01, 0xe0, 0x06, 0x00,
   0x00, 0xc0, 0x1f, 0xe0, 0x07, 0x00, 0x00, 0xe0, 0x7f, 0xe0, 0x07, 0x00,
   0x00, 0x70, 0xe0, 0xe0, 0x05, 0x00, 0x00, 0x38, 0x80, 0xe1, 0x04, 0x00,
   0x00, 0x18, 0x80, 0xf1, 0x04, 0x00, 0x00, 0x0c, 0x00, 0xfb, 0x04, 0x00,
   0x00, 0x0c, 0x00, 0xff, 0x04, 0x00, 0x00, 0x86, 0x1f, 0xee, 0x04, 0x00,
   0x00, 0x06, 0x06, 0xe6, 0x04, 0x00, 0x00, 0x06, 0x00, 0xe6, 0x04, 0x00,
   0x00, 0x06, 0x00, 0xe6, 0x04, 0x00, 0x00, 0x06, 0x00, 0x66, 0x04, 0x00,
   0x7f, 0x56, 0x52, 0x06, 0xe4, 0xff, 0x00, 0x76, 0x55, 0x06, 0x04, 0x00,
   0x00, 0x56, 0x57, 0x06, 0x04, 0x00, 0x00, 0x56, 0x55, 0x06, 0x06, 0x00,
   0x00, 0x56, 0xd5, 0x06, 0x03, 0x00, 0x00, 0x06, 0x00, 0x86, 0x01, 0x00,
   0x54, 0x06, 0x00, 0xc6, 0x54, 0x55, 0xaa, 0x06, 0x00, 0x66, 0xaa, 0x2a,
   0x54, 0x06, 0x00, 0x36, 0x55, 0x55, 0xaa, 0x06, 0x00, 0xbe, 0xaa, 0x2a,
   0x54, 0xfe, 0xff, 0x6f, 0x55, 0x55, 0xaa, 0xfc, 0xff, 0xa7, 0xaa, 0x2a,
   0x54, 0x01, 0x88, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x60, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa0, 0xaa, 0x2a,
   0x54, 0x55, 0x8d, 0x50, 0x55, 0x55, 0xaa, 0xaa, 0x8a, 0xa8, 0xaa, 0x2a,
   0x54, 0x55, 0x95, 0x54, 0x55, 0x55, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0x2a,
   0x54, 0x55, 0x55, 0x55, 0x55, 0x15, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
}

set teapot {",
".. c #998074",
"#. c #84aadc",
"a. c #c47e55",
"b. c #4c6ea0",
"c. c #d48a64",
"d. c #ecccbc",
"e. c #8c6252",
"f. c #b1a094",
"g. c #714628",
"h. c #af8a76",
"i. c #2c5284",
"j. c #98745f",
"k. c #8f5124",
"l. c #844614",
"m. c #5074a6",
"n. c #aeb2b9",
"o. c #3e291c",
"p. c #44689b",
"q. c #4c2705",
"r. c #1b2b43",
"s. c #e1c8bb",
"t. c #dd782d",
"u. c #9c5824",
"v. c #547aac",
"w. c #906e5a",
"x. c #545253",
"y. c #e07e33",
"z. c #bcaeac",
"A. c #aeb6bc",
"B. c #6f737b",
"C. c #7e5942",
"D. c #b55e1a",
"E. c #7c5134",
"F. c #8c7a70",
"G. c #62524d",
"H. c #bc8a6c",
"I. c #936754",
"J. c #6e6d75",
"K. c #8cb0e2",
"L. c #d4beb4",
"M. c #2a201a",
"N. c #c48664",
"O. c #412105",
"P. c #a95a19",
"Q. c #747a7c",
"R. c #2a2e33",
"S. c #484c53",
"T. c #8690a1",
"U. c #6c524c",
"V. c #505967",
"W. c #040204",
"X. c #8cb6e4",
"Y. c #8f4912",
"Z. c #846e5c",
"0. c #244a7c",
"1. c #dcc2b6",
"2. c #33261b",
"3. c #81736c",
"4. c #3c3b46",
"5. c #735f53",
"6. c #545e69",
"7. c #bc7c58",
"8. c #4c2f1d",
"9. c #a4806b",
".# c #604c4c",
"## c #d4916b",
"a# c #5c80b2",
"b# c #e3d3cd",
"c# c #c8ad9a",
"d# c #bf6c2e",
"e# c #08121c",
"f# c #9c5e34",
"g# c #9298a1",
"h# c #8b7f7c",
"i# c #835f44",
"j# c #a46e54",
"k# c #474649",
"l# c #a65d26",
"m# c #be6420",
"n# c #7c9ed2",
"o# c #ac622c",
"p# c #726e6c",
"q# c #e1a684",
"r# c #a48676",
"s# c #54464c",
"t# c #a0a4ac",
"u# c #b37b5a",
"v# c #947a70",
"w# c #543d31",
"x# c #743e12",
"y# c #98bcee",
"z# c #d49e7c",
"A# c #64320c",
"B# c #d4cac6",
"C# c #2c3642",
"D# c #6e686b",
"E# c #5c463e",
"F# c #7f7575",
"G# c #bcb2b0",
"H# c #979dac",
"I# c #a47a61",
"J# c #8c9094",
"K# c #1c262c",
"L# c #d3c3bc",
"M# c #9cc2f4",
"N# c #af7455",
"O# c #fcad5f",
"P# c #6486b8",
"Q# c #d4742d",
"R# c #6d4e3d",
"S# c #242224",
"T# c #3b444c",
"U# c #181515",
"V# c #6e4024",
"W# c #749bcc",
"X# c #afabac",
"Y# c #7e6a64",
"Z# c #343634",
"0# c #e8ac8a",
"1# c #343a44",
"2# c #b18f7a",
"3# c #dda07f",
"4# c #6f380d",
"5# c #2f5684",
"6# c #a08d86",
"7# c #5f6068",
"8# c #625751",
"9# c #16253b",
".a c #94b6e9",
"#a c #dfcec8",
"aa c #3c6194",
"ba c #ecdad3",
"ca c #948a89",
"da c #a4c8fa",
"ea c #bcb6b8",
"fa c #604432",
"ga c #cc6d29",
"ha c #949ea4",
"ia c #413024",
"ja c #a48a7a",
"ka c #9a867d",
"la c #1c1a1c",
"ma c #342116",
"na c #bc8364",
"oa c #9c8674",
"pa c #844e25",
"qa c #3c4a5c",
"ra c #582c0d",
"sa c #c7835a",
"ta c #613c24",
"ua c #614b41",
"va c #adaeb5",
"wa c #805c4c",
"xa c #0a0b0f",
"ya c #c4b2ac",
"za c #648abc",
"Aa c #af846a",
"Ba c #c68a69",
"Ca c #483c39",
"Da c #7f400d",
"Ea c #646e74",
"Fa c #a85515",
"Ga c #042244",
"Ha c #816759",
"Ia c #95817c",
"Ja c #9c9aa1",
"Ka c #9c6b54",
"La c #7e4924",
"Ma c #0c1a2c",
"Na c #5f595f",
"Oa c #6f615e",
"Pa c #474149",
"Qa c #8c9aac",
"Ra c #cbbeb8",
"Sa c #806e68",
"Ta c #82624e",
"Ua c #949294",
"Va c #b4652c",
"Wa c #9c9ea4",
"Xa c #72574c",
"Ya c #2a303e",
"Za c #bc9175",
"0a c #9c7a64",
"1a c #261a13",
"2a c #aca6ac",
"3a c #341a04",
"4a c #593520",
"5a c #d49878",
"6a c #26282c",
"7a c #475261",
"8a c #3f3535",
"9a c #63676f",
".b c #a47255",
"#b c #bc967c",
"ab c #52433a",
"bb c #80a4d6",
"cb c #544c4c",
"db c #bc723c",
"eb c #645e54",
"fb c #a4abb4",
"gb c #c89070",
"hb c #8c7469",
"ib c #c7b8b3",
"                                                                                                                                ",
"                                                                                                                                ",
"                                                                                                                                ",
"                                                            wawae.e.wa                                                          ",
"                                                          .#j#N#7.a.db.#                                                        ",
"                                                            8.g.x#C.o.                                                          ",
"                                                  N#DaA#raY.Xa4aV#4aPas#q.W.W.6a                                                ",
"                                            i#O.l.4.4.4.4.4.s#wa7.N#wawaXawawawaA#q.C.                                          ",
"                                        3a3a4#wawawae.e.e.I.j#m#d#VaN#KaI.e.e.e.e.waDaO..#j#j#N#N#j#fa                          ",
"                                        O.A#wae.e.I.e.I.I.Kal#m#m#D.u.KaKaKaKaI.e.I.e.Da3aXaE#C#W.o.C.                          ",
"                                      w#e.k.I.I.Kaf#KaKaKaKal#m#m#D.l#j#Kaj#KaKaKaKaKaFai#u.    GaDaI.                          ",
"                                      .#waKaI.KaKaj#j#j#j#j#VaD.m#FaVaj#j#j#j#j#j#j#KaKae.U.    9#o.Ka                          ",
"                                      .#Xae.Kaj#j#N#N#N#N#N#j#N#j#N#N#N#N#N#dbdbN#N#N#I.waU.  e#M.taPa                          ",
"                                    .#U.Xawae.Kaj#N#7.dbj#a.a.a.a.a.a.a.j#7.sa##naj#I.e.waXaU.Yama.#                            ",
"                                    .#U.Xawae.I.Ka.bna3#0#3###sasaa.sac.##Ba7.u#N#Kaf#f#waXaU.4.s#Pa                            ",
"      j#N#5aW.Y.u#j#R#            8a.#XaXae.e.I.KaN#naz#0#q#gb7.7.N#N#N#u#nau#N#N#Kaf#f#e.waU.LaU.                              ",
"        4.Hahb.bI.j#KaY.          s#U.XaXawae.KaKaN#na3#0#3#gbna7.N#u#u#N#7.u#N#.bj#KaI.e.waXa.#                                ",
"              G.2#e.KaI.E#        .#U.Xawae.P.Kaj#N#H.z#q#3#gb7.7.7.7.N#N#u#u#N#.bj#KaI.e.TaXaU.0.                              ",
"                Ca.#e.j#Va  5#aa0.MaU.Xawak.d#I.j#I#N.5a3#5aBa7.dbN#N#dbN#N#u#I#j#KaKaI.e.waXaU.0.i.p.                          ",
"                  abU.I.KapaaaaaxaGaU.C.wau.VaKaj#I#na5a5agbN.u#N#7.N#N#N#N#N#.bj#KaKaI.e.waXa.#R#m.m.m.p.                      ",
"            zaW#0.0.s#waKam#0.0.e#GaU.C.k.l#I.KaKa.bnagbgbBa7.u#7.N#N#N#N#N#.bj#j#KaKaI.e.waXa.#.#b.m.m.i.i.0.                  ",
"    5#aaW#W#n#aaaa0.U..#e.Kau.ta4aLak.k.u.o#e.KaKa.bu#H.H.nau#N#N#N#N#N#N#j#N#j#KaKaI.I.e.waXa.#fab.m.i.i.i.i.i.0.              ",
"p.p.p.aan#W#0.0.0.0.p.uaU.KaKal#o#d#gal#o#o#e.I.Kaj#N#Aau#u#N#N#N#.bN#N#j#.bj#j#KaKaI.I.e.waXa.#E#m.m.v.m.i.i.m.n#bbn#          ",
"p.p.n#n#n#p.0.0.0.0.0.xa.#waKaKagaQ#t.y.d#dbi#e.I.Ka.bN#I#N#N#N#N#.bN#N#j#N#j#KaKaI.f#e.waXaU.U.w#5#m.m.i.i.i.m.bbbbbbbbaa      ",
"n#n#n#n#n#n#0.aaaap.aaaas#s#waKaj#na##O#dbl#wae.I.I.j.j#j#j#j#j#j#j#.bj#j#j#KaKaKaI.e.e.waXaE.Hai.m.m.i.i.i.i.p.p.bbb.b.p.p.aa  ",
"p.bbn#bbp.p.p.p.p.p.aaaaaaua..#be.Kaj#j#u.XawaTae.I.KaKaKaKaj#j#j#j#j#KaKaKaKaKaI.I.i#waXawaR#uav.m.m.v.v.i.b.bbbbbbbbp.p.b.p.p.",
"bbp.p.p.p.p.p.n#p.0.0.0.0.p.8#Oa2#ZaZaZak.XaXawae.e.e.I.KaKaKaj#KaKaKaKaI.w.I.I.I.e.waXa0aXa5.5#i.m.m.v.v.p.p.#.#.#.#.#.#.b.bb#.",
"bbW#p.p.p.n#bbbbp.0.0.0.0.0.0.0.x.D#3.3.w.waU.wawae.e.e.e.f#I.I.I.I.I.I.I.f#e.e.wawaXae.w.ua4.5#5#i.7ai.b.b.b.p.b.#.#.#.p.b.b.#.",
"p.bbp.bbbbp.p.p.p.0.0.6.p.p.p.0.0.0.0.9#R#wae.9.j.wawae.Tae.e.e.e.I.e.Tai#e.wawaw.Aaw.hbOa6aC#5#m.v.v.#.p.p.b.b.p.#.#.b.b.p.p.b.",
"0.0.0.p.p.p.p.aa0.0.p.p.p.p.p.p.0.0.p.qaT#s#Z.0a..h.I#I#ZaZaI#gbI#.bH..b.bH.h.j.r#HaU.PaV.V.Na1#v.m.K.K.K.#.b.#.#.#.#.K.p.b.m.p.",
"0.0.0.0.p.0.0.0.0.0.0.p.p.p.p.0.0.p.p.qaqaqa2.iaR#HaY#ja2#2#I#Za2#0a2#2#h.jaoaF.8#V.6aC#C#C#CaNab.bbbbK.#.b.b.b.#.K.K.#.#.#.X.#.",
"b.b.p.p.p.0.i.0.0.0.0.p.0.0.0.i.0.0.qaqak#6aR.6a3aiawaY#OaOaOaD#Iah#D#OaF#cbPa6a6aW.V.6.1#C#7ap#b.b.b.b.b.b.b.b.b.K.K.#.b.b.b.b.",
"bb#.#.#.#.b.b.i.0.p.p.p.0.b.b.0.0.0.qar.YaYaYaR.K#Z#E.xa1a2.S#.#x.x.S.R.laU#1#W.W.9a6.J#J#T.7av#K.K.K.K.b.b.m.m.b.m.K.b.p.m.b.b.",
"#.#.#.#.#.#.K.#.#.#.#.#.#.b.b.b.#.p.qaqaYaR.YaR.Pa6ak#S#lalaW.W.xaW.W.W.W.U#U#ebT.hag#g#haT.7ac#K.K.K.K.K.K.b.b.#.K.K.K.K.m.b.b.",
"#.#.#.b.b.b.b.#.#.bb#.#.#.b.K.bbK.#.T.T.J#R.YaS.S.x.ebebR.2.S#S#S#T#S#S#S.9a7aEaEafbhag#6.7aV.m.K.K.K.K.X.K.m.X.K.K..a.aX.K.b.m.",
"b.b.p.b.p.b.p.K.K.K.b.b.b.b.#.#.#.#.T.T.J#UaJ#J#g#g#haWaWa7#9a2at#fbA.t#9a6.9aJ.EaEafbV.6.V.Nam.b.m..a.am.b.m.m.m.K.X.K..a.a.am.",
"  p.b.p.p.b.p.K.b.b.p.b.b.b.b.K.#.K.7a7aV.9aWat#X#WahaH#ha6.t#n.n.n.n.fbfb9a9an.t#t#fbfb7aV.7#m.m.m..am.m.m.m.m.m.m..a.a.am.m.m.",
"  b.p.b.#.#.#.#.#.b.b.b.b.b.b.K.Pax.V.7aNa7#7#t#vafbB.B.B.B.B.n.A.A.A.fbfb9aA.t#t#fbfbQaH#7#D#m.X..a.a.aX.v.v.v.v.m.m..am.v.m.m.",
"    #.K.K.K.K.K.#.b.b.b.b.K.K.K.T.S.V.9aB.ebNaV.p#9a9aB.D#9a9aB.A.A.A.9aB.B.Eafbn.t#g#hat#X#m..ay#y#y#y#.ay#v.v.v..aX.y#.av.v.v.",
"    #.K.K.#.K.K.K.K.K.K.K.K.K.QaT.V.7#Ea9af.f.2at#OaD#F#Q.J.9aEaB.Ea9a6.Ea9aEaV.H#H#t#2aD#J.v.m.y#y#y#y#X..ay#y#y#y#y#y#.ay#.av.",
"      K.K.K.K.b.b.b.b.K.K.K.K.g#g#t#D#fbWag#c#X#2a2aeb7#D#Q.vaA.2at#p#9a6.V.7#7#9avaSaSaSav.m.v.v.y#.ay#X.v.v.v.v.y#y#y#y#y#y#y#",
"      K.K.b.m.b.b.b.b.K.K.K.K.Ja7#7#6.2az.#ac#z.G#eaG#p#2a2a2at#t#t#t#7#9aD#J.SaL#RaRaF#v.v.v.v.v.v.y#m.v.v.a#a#v.v.M#M#M#y#y#  ",
"        b.b.m.b.m.b.b.b.K.X.7#7#7#7#D#3.1.z.yac#yaibh#caB#L#eaz.z.z.z.z.F#h##ab##aL#ibibv.a#m.m.v.y#M#y#a#v.v.v.a#a#a#a#M#      ",
"        p.b.m.b.b.m.b.b.m.b.7#9a9aSahb..m..ayayahbhbF.h#Ias.RaRaibibL.L#kab#bab#1.L.L.y#y#a#v.v.M#y#y#M#y#v.a#a#a#v.a#a#        ",
"          b.m.b.b.X.K..aK.b.7#D#F#hbv#m.b.m.m.F.F.F.v#v#F.v#L.L.L.L#..ka6#h.s.1.L#1.y#M#y#daM#M#day#y#M#M#M#a#a#a#a#            ",
"          p.b.X.X.X.X.X.X.X.9aD#6#..m..aX.y#m.m.F.v#v#v#....1.Ia....kakar#r#r#1.s.M#M#y#a#a#a#M#M#y#y#daM#dadada                ",
"            .a.aX..aX..aX..aD#SaF..ay#y#.aX..am.m.........s.s.9.9.r#r#r#r#kakar#M#M#a#a#a#a#a#a#M#M#M#M#dadada                  ",
"            K.X..a.a.a.a.aH#vab#1.y#X.X.y#y#y#y#v.v...s.d.d.d.s.r#r#h.jajajar#a#M#v.a#a#a#a#a#a#P#dadadaM#                      "\};
}

show_image xpm text color "/* XPM */
static char * teapot\[\] = \{
\"64 48 204 2\",
\"   c #145ec4$teapot"

show_image xpm text transparent "/* XPM */
static char * teapot\[\] = \{
\"64 48 204 2\",
\"   s None c None$teapot"


show_image ps text "GhostScript" {%!PS-Adobe-3
%%Title: postscript.ps
%%BoundingBox: 66 648 146 720
%%Pages: 1
%%DocumentProcSets: Adobe_distill 0.96
%%EndComments
%%BeginProcSet: Adobe_distill 0.96
/PROLOGUE 30 40 add dict def
 % 30 procedure entries + room for 40 cached font dictionaries
 PROLOGUE begin
 /clip { } def    % causes problems. remove if "clip" is needed
 /bdef { bind def } bind def	/ldef { load def } bdef
 /T { moveto show } bdef	/A { moveto ashow } bdef
 /W { moveto widthshow } bdef	/AW { moveto awidthshow } bdef
 /f /fill ldef			/R { { rlineto } repeat } bdef
 /r /rlineto ldef		/L { { lineto } repeat } bdef
 /m /moveto ldef		/l { moveto lineto stroke } bdef
 /x { 0 rlineto } bdef		/y { 0 exch rlineto } bdef
 /c /curveto ldef		/cp /closepath ldef
 /s /stroke ldef		/w /setlinewidth ldef
 /g /setgray ldef		/j /setlinejoin ldef
 /d /setdash ldef		/F /setfont ldef
 /C /setcmykcolor where { /setcmykcolor get }{ %ifelse
   { %def
     1 sub 3 { 3 index add neg dup 0 lt { pop 0 } if 3 1 roll } repeat
     setrgbcolor
   } bind
 } ifelse def
 /selectfont where { pop }{ %ifelse
     /selectfont { exch findfont exch scalefont setfont } bdef
 } ifelse
 /MF { exch findfont exch makefont setfont } bdef
 /FF /selectfont ldef
 /DF { selectfont currentfont def } bdef
 /BEGINPAGE { pop /pagesave save def } bdef
 /ENDPAGE { pop pagesave restore showpage } def
 /REMAP { %def
   FontDirectory 2 index known { pop pop pop } { %ifelse
     findfont dup length dict begin
       { 1 index /FID ne {def}{pop pop} ifelse } forall
       exch dup length 0 gt { /Encoding exch def }{ pop } ifelse
     currentdict end definefont pop
   } ifelse
 } bdef
 /RECODE { %def
    3 -1 roll 1 index findfont /Encoding get 256 array copy exch
    0 exch { %forall
     dup type/nametype eq
       { 3 {2 index} repeat put pop 1 add }{ exch pop }ifelse
    } forall pop 3 1 roll REMAP
 } bdef
 end %PROLOGUE
%%EndProcSet: Adobe_distill 0.96
%%EndProlog
%%BeginSetup
PROLOGUE begin
%%EndSetup
%%Page: 1 1
1 BEGINPAGE
1 g
2 setlinecap
144 53 m
390 x
-19 y
-390 x
eofill
0 g
1.268 w
0 setlinecap
4 setmiterlimit
[] 0 d
109.698 714.182 m
111.525 713.672 112.472 713.234 113.624 712.431 c
-22.3128 -25.8381 r
97.4019 682.652 103.794 672.84 100.329 664.844 c
102.034 675.687 90.7436 683.82 84.1099 684.842 c
25.588 29.3407 r
f
1.0361 w
f
f
91.8805 654.468 m
96.0255 654.285 100.155 654.188 104.481 654.188 c
113.804 654.188 122.752 654.636 131.141 655.44 c
134.222 655.737 136.724 658.133 137.108 661.279 c
137.79 666.855 138.107 672.696 138.107 678.705 c
138.107 684.714 137.79 690.555 137.108 696.131 c
136.724 699.277 134.222 701.674 131.141 701.97 c
127.476 702.321 123.704 702.604 119.84 702.813 c
119.84 702.813 103.842 668.373 102.231 664.827 c
99.8549 659.595 96.601 655.765 91.8805 654.468 c
f
0.7851 w
77.8202 655.44 m
74.7171 655.948 72.2887 657.716 71.8525 661.279 c
71.1725 666.855 70.8549 672.696 70.8549 678.705 c
70.8549 684.714 71.1725 690.555 71.8525 696.131 c
72.2381 699.277 74.7417 701.674 77.8202 701.97 c
82.1335 702.383 86.5932 702.702 91.1764 702.916 c
8.16241 9.07727 r
99.2863 711.98 101.266 712.345 102.996 711.921 c
-28.4564 -32.9334 r
80.6983 681.906 98.7446 677.354 98.7446 666.708 c
98.7446 655.332 86.3243 654.054 77.8202 655.44 c
f
0.0843 w
139.67 644.154 m
140.194 644.154 140.619 644.576 140.619 645.098 c
140.619 645.619 140.194 646.042 139.67 646.042 c
139.146 646.042 138.721 645.619 138.721 645.098 c
138.721 644.576 139.146 644.154 139.67 644.154 c
1 ENDPAGE
%%Trailer
end %PROLOGUE
%%Pages: 1
%%EOF}


show_image dted base64 "level 0" {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}

show_image raw base64 "8-bit integer green channel" {
TWFnaWM9UkFXICAgICAgICAgIApXaWR0aD01MCAgICAgICAgICAgCkhlaWdo
dD01MCAgICAgICAgICAKTnVtQ2hhbj0xICAgICAgICAgIApCeXRlT3JkZXI9
SW50ZWwgICAgClNjYW5PcmRlcj1Cb3R0b21VcCAKUGl4ZWxUeXBlPWJ5dGUg
ICAgIAqLk5GWnJSdm42Fh3x9enZ5b3ZzYFlZX3CtcGlaVGNwcmyrqpyWkZeN
kZKVnZmboZ2YoZijn56boaammY6HfHp+eXhxc3NkWlhbcrpwZ1dTYXBwbbCh
oKCMmZiloZ6cpKivo6ako6msrqemsaiWj4V5fH57e3J3cmhdV1lxuG9lWFVf
b3GKpayknZKfpK6vraWvsKmeqJWYpqqhm6Wxp6Gbinx+foB6cnRxamFYWG61
c2NaVV9xcI+xrZ6ZkpilrKKcpbKsrLGomp+smZOLlaqspZeOg4F+f3RvcXBr
YVdYbrd1ZlxWX3FujKOemJihoqmdjoygrrGtpaijpauYl42Tp7SsoY6AgHx6
cmxtcGtiWVpsnXFoXVhib2uNqJ+inqWrnJiPkaS1ubGqm6SvqamfqJeYqaeo
kIJ+end1bG1uamNaW2tza2lcWmdtaYmpo7Cyr6unoqeTn6uytaSam6qnnpuq
oI6dl4qNg3p6d3NtcGxqZFpea2praF1bZm1qjpuTmaipopymqJCYoZKcoZKX
j56boKaimZWUiX59eXd4dW1ya2pmXmBqeW1oXV1mbGmKiomKk6GcoaijmZaX
j4mSjpF4i56ikJGjnJSJhYB/dndzb3Bsa2pjYW2icmtgX2ZraZyTjnx/nKKa
jaCgmpCQjZiSkYqBl5SNkKWknJKHe3V7enZzb21sa2Fhb6hzbmFiaWtqqKuj
jIGYlY+QpqWel5CIhZSTio6aj5CMnaecl5B/d3x+eXhwb2tsYGFwqnFvX2Rs
bmyXmIeOk5mJk5CnpJyblIqJlZSOnaqjpJKUmqCcl4Z1dYB9fHBxampgYG+o
cXFfZG1wbJSMjpyrqaSVk5uhoJ+Yh4GZopOip6eqnY+TpKeViH99hIB9c3Rs
amFib5dvcGJlbG16oI2bpaioq5iOmaqmmpKUk6qzmZmXlY+TmZqck42Ffn2B
g4B6dmxqYWRsb2lrYWBtboafn6OfoZiboaOloJmXlJyhq7CimaCZkY6ap6SO
iYp9dnyCenJwbmpbXmx2Z2dbWm5vhKOjm6SZlI+frKOPlJeSlaOopKGZmJ+c
maGWlYyGjIWAgX95cG1vbF1eaphsaF5fbW2EpZ2WoKCanZ2VkomUmpuopqGe
n5ebm5uXmpGKg5CNfIN/gHhvbGxsX19npG1pYGNsaoagmJ2bnJmbkouFl5SL
naCopJ+fopydj4aLjH+GnZ2KgH9/eHBvb2lfYGmibmdfZG5shaWdn5mKh5CE
g5Cro5uXoZuXlp6dmJWQiYqPi5KamIKAe3p3cnNxaF9eappvZ15lcGyjn5mX
kYuKkI2Sm6ORlpSRjJOUnJ2Uk5iSk5OZoaKQiIV+eHl1c3FnX11qfG1nXWVx
baOclJaZkpWVnKShlJegkY6Ump+bnpWSlYaNmJOXoZaGgX17fXlzcmdcXGly
a2VaZnFuoZiTlYyHmpWVop6UkpuSmJ+fop6ioJWUjYmTlZSamo+Df3x6eXJ0
aFpZZ4xvYldlcW2onpWUjYuVlpWen5ubmpaSnaKpnaKnoZiQm6ShoKWmmIt/
e3x4c3JrXVhko3NfVWJwbauknZCao6ShpKmpoaGcmZqfp6qbnKilmZWep6Gd
p6qfkYF+fnpycGxhV2Khc19WYXBtq6aal5+noZ6qraqmnqOcnZ2qqYqGnqCd
oaahl5GbrKaWhH98dW5zbGNYYpByYlhkbYmmn5+lqJuUlKiyrqSin5OIkqeg
jYiRnJ2lqaSfpJagnZODfHl0bXFqY1tkb2phW2ZrhJ6dp6umoKaWoaSknJiT
joiXnp+Oio6OjYmTnZ6dm5iOhYB6d3RucGpmX2Z+bmNdZmt/jouLmp+dnpyX
kI+UkJCNjZOPipCTlJmbjoWUlI6doZWKfnl5dnFva2phaZZzZ2Boa4SdkoaR
l5Cao5qSjY6Sk5aZnZeIjp6elJOPlJ6blJSfnJSDdXt9d3BsaWBpl3NoYGxt
gZOQmp+bkpqgnpqMh5aim5mTkpqXp62pmJGcn56dkpiglYh/gYN9eGxpY2uD
b2pkbGyUlJ2ko6SXmKSgmJSYqLCrn5WfpJijpqCWnJmbko6bn5OIhHt6gHl3
bWddZ3NnYlxqbZ2hnqGWlKGkl5GWl5+opaCfoaCdmaGinJiemZqamJiUi42G
goB+dXNsaV1njGxjX2proZyZnJqal5GOkpWepaOcn56anJuXmpmVlpufm5WM
jIiFlpGDf3x1cm9nXmiRbWNia2ufoJyWjY2Jh5udl5ybl5igoZyWjpKVjZCU
nJmVkoyQk5mcj4J7eHdycGZeZoBsYmFtbaCXlpCOkpaenpWXkI+WnZ6Xlo+Q
lpeTlJucnJSUi4+UlZyRhX17enRxZVtld2teYG5tn5aUjo6VlZ6al5mTmJ6h
m5SUi5SinZuWl5yjnpWQmZuboZ2MgHp5dHRmWmKMblpebHynnZKUnJyepaSf
nJeZnqiflZSbnKmkn5+Yk5yimZqhnZmlpJSDfXpzdGldYI5uW15riaWcmaGg
mqKsp6GhmpWfpZ2aoaCbo6OcmJKLjpyfpKSenZuhlYN8dm9waF9fcmpdYWqF
naGmpKCdn6ajm5iQipuepKejoZualJCQj46Pk5GMlZqYm5WJgHl2cHBpZGWA
bmBjaYKUjZKcmZyblJGSkZCSmJGOmpubnZqblYyOl5yYlZGWmZSam5GAent2
cWtlZ4hwY2drgpaTmJqUnJ2Yjo+ZnpyXlJeblpqem52YlZegpZ6Ul5mXlZeX
j4WAgXx0a2Znd2xkaGyFm5+fnZydnZiXnaamoJygn56enZ2XlY+Tl5+fm5mb
mpaUl5GMhX99dndrY2V5aV9la6Cdm5iWmZWSlJuioJ2gnZyXl5mUk4yLkZmV
l5aVmJyZk42MjZWKgHt1dWxjZoBrYWdqn5yYkY6NkpqWmJiXnZ6blY+Oj5iY
jpOYnJiUj5SYmpaSjpOXm42AeXd2bWFjd2peZ2yemJSPk5eem5eSk5qfmZWQ
k5eenZiUlpifoZ2Yl5menJSVmJmfloV8d3ZtYGF+altlbKSblJeZnKOhnZmZ
nqSalZiZnaShnZ2fnJ6joJyXkJaenZ+fnKCbin52c2xkYHZpXWVroJ6foZ2g
pqGdmJGboJ6goZ6jpaCckpWZmZmWkZCPkpaTk5mamZSEe3Vya2dld2thZmqZ
k5abm5uZlJKRkpyVlJqcm5uWk5OVmJyYl5qSlJqflpSWm5iWloeAfXh2aWl7
bWVqg5eYnJ2YnJmTmJ6gmJmcnpmcmpSYnp2bl5WYlJSYnp2YmZeSk5WOh4F9
d3VoZ3FnY2iGnp2Yl5qVlpugn6CfnZeYmpaXnKGf}

show_image raw base64 "32-bit float depth channel" {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}

if 0 {
show_image other base64 pdf {
JVBERi0xLjIKJcfsj6IKNCAwIG9iago8PC9MZW5ndGggNSAwIFIvRmlsdGVyIC9GbGF0ZURl
Y29kZT4+CnN0cmVhbQp4nF1TSY7cQAy7+xX1AkFbaXlG3jBAgAAzh/z/Esp2uyeDPhhUayEp
FZMsnt/9/fg6/h6y/hzCHdS1UlyodH0dIrJJNyIWlLpE1OcLrGQOjLg6MOIm6+NoMULWiopN
bevz6HRiaUQ0aKMHW1L7itQitGA2UvwdDuhoIaw8zSN3UOVqTnILNLAiXQVy3NNvCvQs+M57
Zv4Gj6LivWK7UxS0dDCknBGl2jNY6MKoKmBkljwYfQUD2d8ZqjkKgINASEyEfCq208nc0Fon
YSdZIiFy7IldQmbACe4FraKUfVYkzEAgarhgIBJyvIGi+IYrCWqm4I6UJ4mvV4PmfTW4J3TI
tZCHQzdmDqeLY7Jgpf6IAG7KKVAsLAMBxWIEqrEvuAAc8xXp2Rtgkdhp038B7PfcCyQZ5WDV
oXIuWGdCI387SO/GjezVgcZyuRaAP1b3gXVmopb1ZfXXkSM/z6Imx+4xpHAsseGHgD2abFzu
2+qEzjwjl9fJN4vb6zd+ef1Ebq+fDrfZz4zH7GFhc0O32cPSJR+zXzIes0vnMPZlNk73ejWT
gFNDYs/IOM3tkTXmGUbMsaPF9cAwq8KuyNwxFMc4gJY+d47S0Td/68nSNxkiUNpz159H8Twg
g1LkcKyGBe6TkQYL3jhwTDwreSJ7gw9eZeBY3GZpPBU/FjY7/HX8A9+h3WJlbmRzdHJlYW0K
ZW5kb2JqCjUgMCBvYmoKNTE2CmVuZG9iagozIDAgb2JqCjw8Ci9UeXBlIC9QYWdlCi9NZWRp
YUJveCBbMCAwIDYxMiA3OTJdCi9QYXJlbnQgMiAwIFIKL1Jlc291cmNlcyA8PCAvUHJvY1Nl
dCBbL1BERl0KPj4KL0NvbnRlbnRzIDQgMCBSCj4+CmVuZG9iagoyIDAgb2JqCjw8IC9UeXBl
IC9QYWdlcyAvS2lkcyBbCjMgMCBSCl0gL0NvdW50IDEKPj4KZW5kb2JqCjEgMCBvYmoKPDwg
L1R5cGUgL0NhdGFsb2cgL1BhZ2VzIDIgMCBSCj4+CmVuZG9iago2IDAgb2JqCjw8IC9DcmVh
dGlvbkRhdGUgKEQ6MTk5NzEyMDUyMjU2MzcpCi9Qcm9kdWNlciAoQWxhZGRpbiBHaG9zdHNj
cmlwdCA1LjEwKQo+PgplbmRvYmoKeHJlZgowIDcKMDAwMDAwMDAwMCA2NTUzNSBmIAowMDAw
MDAwNzk5IDAwMDAwIG4gCjAwMDAwMDA3NDAgMDAwMDAgbiAKMDAwMDAwMDYyMCAwMDAwMCBu
IAowMDAwMDAwMDE1IDAwMDAwIG4gCjAwMDAwMDA2MDEgMDAwMDAgbiAKMDAwMDAwMDg0OCAw
MDAwMCBuIAp0cmFpbGVyCjw8IC9TaXplIDcgL1Jvb3QgMSAwIFIgL0luZm8gNiAwIFIKPj4K
c3RhcnR4cmVmCjkzOQolJUVPRgo=}
}

