/* 
 * Copyright (C) 2003 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: menu.c,v 1.33 2004/12/15 19:23:19 dsalt Exp $
 *
 * main/context menu creation / handling
 */

#include "globals.h"

#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "menu.h"
#include "playlist.h"
#include "engine.h"
#include "utils.h"
#include "gtkvideo.h"
#include "vis.h"

/* global pointers to menus: */

GtkWidget *popup_menu = NULL, *menubar = NULL;

GtkMenuShell *media_menu[2] = { NULL, NULL };
GtkCheckMenuItem *fullscreen_menu_item[2] = { NULL, NULL },
		 *deinterlace_menu_item[2] = { NULL, NULL },
		 *auto_resize_menu_item[2] = { NULL, NULL },
		 *toolbar_item[3][2] = { { NULL, NULL }, { NULL, NULL },
					 { NULL, NULL } };
GSList *aspect_menu_items[2] = { NULL, NULL };

/*
 * context menu
 */

#if 0
static gint context_menu_cb (GtkWidget *widget, GdkEvent *event) {

  if (event->type == GDK_BUTTON_PRESS) {
    GdkEventButton *bevent = (GdkEventButton *) event; 

    if (bevent->button == 3) {
      gtk_menu_popup (GTK_MENU(widget), NULL, NULL, NULL, NULL,
		      bevent->button, bevent->time);
      /* Tell calling code that we have handled this event; the buck
       * stops here. */
      return TRUE;
    }
  }
  
  /* Tell calling code that we have not handled this event; pass it on. */
  return FALSE;
}
#endif

static void autoplay_cb (GtkWidget* widget, gpointer data) {

  char **mrls, *mrl;
  int    n, i, pos;

  mrls = xine_get_autoplay_mrls (xine, data, &n);

  if (!mrls) {
    playlist_logo();
    display_error (_("Autoplay input plugin failed"),
		   _("Autoplay for input plugin '%s' failed.\n"
		     "Check engine output for further details."),
		   data);
    return;
  }

  mrl = mrls[0]; i = 0; pos = -1;
  while (mrl) {

    if (pos<0)
      pos = playlist_add_mrl (mrl, -1);
    else
      playlist_add_mrl (mrl, -1);

    mrl = mrls[++i];
  }

  playlist_play (pos);
}

static void add_autoplay_entries (GtkMenuShell *submenu, int index) {

  GtkWidget    *item = NULL;
  int           i;
  const char *const *autoplay_ids = xine_get_autoplay_input_plugin_ids (xine);

  static const struct {
    char id[8];
    const char *stock;
  } stock[] = {
    { "VCD", GTK_STOCK_CDROM },
    { "VCDO", GTK_STOCK_CDROM },
    { "DVD", GTK_STOCK_CDROM },
    { "CD", GTK_STOCK_CDROM },
  };

  if (!autoplay_ids) {
    logprintf ("menu: warning: didn't get any autoplay entries\n");
    return;
  }

  i=0;
  while (autoplay_ids[i]) {
    int j = sizeof (stock) / sizeof (stock[0]);
    while (j--)
      if (!strcmp (autoplay_ids[i], stock[j].id))
      {
	item = gtk_image_menu_item_new_with_label (autoplay_ids[i]);
	gtk_image_menu_item_set_image
	  (GTK_IMAGE_MENU_ITEM(item),
	   gtk_image_new_from_stock (stock[j].stock, GTK_ICON_SIZE_MENU));
	break;
      }
    if (j < 0)
      item = gtk_menu_item_new_with_label (autoplay_ids[i]);

    g_signal_connect (GTK_OBJECT(item), "activate", 
			G_CALLBACK (autoplay_cb), 
			(gpointer) strdup (autoplay_ids[i]));

    gtk_menu_shell_insert (submenu, item, ++index);
    gtk_widget_show(item);

    i++;
  }
}

static void vis_cb (GtkWidget* widget, gpointer data) {

  /* printf ("menu: vis_cb %s\n", data); */

  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget))) {
    xine_cfg_entry_t vis_entry;

    if (xine_config_lookup_entry (xine, "misc.vis", &vis_entry))
    {
      vis_entry.str_value = data;
      xine_config_update_entry (xine, &vis_entry);
    }

    pthread_mutex_lock (&engine_lock);
    vis_set (GTK_VIDEO(gtv), data, &audio_port);
    pthread_mutex_unlock (&engine_lock);
  }
}

static void add_vis_entries (GtkMenuShell *submenu, 
			     GtkRadioMenuItem *none_item) {

  GtkWidget         *item;
  int                i;
  const char        *vis;
  GSList            *group;
  const char *const *pol;

  vis = xine_config_register_string (xine, "misc.vis", "none",
				     _("Default visualisation setting"),
				     NULL, 40, NULL, CONFIG_DATA_NONE);

  group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM (none_item));
  if (!strcasecmp (vis, "none"))
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM(none_item), TRUE);

  pol = xine_list_post_plugins_typed (xine, XINE_POST_TYPE_AUDIO_VISUALIZATION);
  if (pol) {

    i = 0;
    while (pol[i]) {

      /* item = gtk_menu_item_new_with_label (pol[i]); */
      item = gtk_radio_menu_item_new_with_label (group, pol[i]);
      group = gtk_radio_menu_item_get_group (GTK_RADIO_MENU_ITEM(item));

      g_signal_connect (GTK_OBJECT(item), "activate", 
			G_CALLBACK (vis_cb), (gpointer) strdup (pol[i]));

      gtk_menu_shell_insert (submenu, item, 1);
      gtk_widget_show(item);

      if (!strcasecmp (vis, pol[i]))
	gtk_check_menu_item_set_active ( GTK_CHECK_MENU_ITEM(item), TRUE);

      i++;
    }
  }
}

static void open_cb (void) {
  engine_exec ("open_show ();", NULL, NULL);
}

static void open_mrl_cb(void) {
  engine_exec ("open_mrl_show ();", NULL, NULL);
}

static void playlist_cb(void) {
  engine_exec ("playlist_show ();", NULL, NULL);
}

static void preferences_cb(void) {
  engine_exec ("preferences_show ();", NULL, NULL);
}

static void keybindings_cb(void) {
  engine_exec ("keybindings_show ();", NULL, NULL);
}

static void exit_cb(void) {
  engine_exec ("exit ();", NULL, NULL);
}

static void fullscreen_cb(void) {
  engine_exec ("set_fullscreen ();", NULL, NULL);
}

static void tbar_toggle_cb (void)
{
  engine_exec ("toolbar_show ();", NULL, NULL);
}

static void tbar_position_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_toolbar_position (%d);", NULL, NULL, action);
}

static void factor_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_video_size (%d);", NULL, NULL, action);
}

static void zoom_in_cb(void) {
  engine_exec ("set_zoom (get_zoom()+5);", NULL, NULL);
}
static void zoom_out_cb(void) {
  engine_exec ("set_zoom (get_zoom()-5);", NULL, NULL);
}
static void zoom_100_cb(void) {
  engine_exec ("set_zoom (100);", NULL, NULL);
}

static void aspect_cb(gpointer data, guint action, GtkWidget *widget)
{
  v_engine_exec ("set_aspect (%d);", NULL, NULL, action);
}

static void deinterlace_cb (gpointer callback_data,
			    guint callback_action,
			    GtkWidget *widget) {

  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    engine_exec ("set_deinterlace (1);", NULL, NULL);
  else
    engine_exec ("set_deinterlace (0);", NULL, NULL);
}

static void auto_resize_cb (gpointer callback_data,
			    guint callback_action,
			    GtkWidget *widget) {

  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    engine_exec ("set_auto_resize (1);", NULL, NULL);
  else
    engine_exec ("set_auto_resize (0);", NULL, NULL);
}

static void mediamarks_add_cb(void) {
  engine_exec ("mm_add_show ();", NULL, NULL);
}

static void mediamarks_manage_cb(void) {
  engine_exec ("mm_manage_show ();", NULL, NULL);
}

static void mediamarks_import_cb(void) {
  engine_exec ("import_mediamarks ();", NULL, NULL);
}

static void about_cb(void) {
  engine_exec ("about_show ();", NULL, NULL);
}

static void log_cb(void) {
  engine_exec ("log_show ();", NULL, NULL);
}

static void stream_info_cb(void) {
  engine_exec ("stream_info_show ();", NULL, NULL);
}

static void wizards_cb(void) {
  engine_exec ("run_wizards ();", NULL, NULL);
}

static void snapshot_cb(void) {
  engine_exec ("snapshot ();", NULL, NULL);
}

static void vis_none_cb (gpointer callback_data,
			 guint callback_action,
			 GtkWidget *widget) {
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget))) {
    xine_cfg_entry_t vis_entry;

    if (xine_config_lookup_entry (xine, "misc.vis", &vis_entry))
    {
      vis_entry.str_value = "none";
      xine_config_update_entry (xine, &vis_entry);
    }
    engine_exec ("set_vis (\"none\");", NULL, NULL);
  }
}

static void settings_cb(void) {
  engine_exec ("settings_show ();", NULL, NULL);
}

static void sub_cb (gpointer data, guint action, GtkWidget *widget)
{
  if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM (widget)))
    v_engine_exec ("set_sub (%d);", NULL, NULL, action);
}

static void popdown_idle_cb (GtkWidget *widget)
{
  gdk_threads_enter ();
  gtk_menu_popdown (GTK_MENU(widget));
  gdk_threads_leave ();
}

static gboolean popup_key_cb (GtkWidget *widget, GdkEventKey *event,
			      gpointer data)
{
  if (event->keyval != GDK_F10 || (event->state & GXINE_MODIFIER_MASK))
    return FALSE;
  g_idle_add ((GSourceFunc) popdown_idle_cb, widget);
  return TRUE;
}

#define STOCK(x) "<StockItem>", GTK_STOCK_##x
#define ASPECT(x) aspect_cb, XINE_VO_ASPECT_##x
static GtkItemFactoryEntry menu_items[] = {
  { N_("/_File"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/File/_Open..."),		"<control>O",	open_cb, 0,		STOCK(OPEN) },
  { N_("/File/Open _MRL..."),		"<control>M",	open_mrl_cb, 0,		NULL },
  { N_("/File/Play_list..."),		NULL,     	playlist_cb, 0,		NULL },
  { "/File/sep0",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Preferences..."),	NULL,		preferences_cb, 0,	STOCK(PREFERENCES) },
  { N_("/File/_Keybindings..."),	NULL,		keybindings_cb, 0,	NULL },
  { "/File/sep1",			NULL,		NULL, 0,		"<Separator>" },
  { "/File/sep2",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Snapshot..."),		"<control>S",	snapshot_cb, 0,		NULL },
  { "/File/sep3",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/File/_Quit"),			"<control>Q",	exit_cb, 0,		STOCK(QUIT) },
  { N_("/_View"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/_Fullscreen mode"),	"<control>F",	fullscreen_cb, 0,	"<CheckItem>" },
  { N_("/View/Fullscreen _toolbar"),	NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Fullscreen toolbar/_Visible"),	NULL,	tbar_toggle_cb, 0,	"<CheckItem>" },
  { N_("/View/Fullscreen toolbar/At _top"),	NULL,	tbar_position_cb, 1,	"<RadioItem>" },
  { N_("/View/Fullscreen toolbar/At _bottom"),	NULL,	tbar_position_cb, 0,	"/View/Fullscreen toolbar/At top" },
  { "/View/sep3",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_50%"),			NULL,		factor_cb, 50,		NULL },
  { N_("/View/_100%"),			NULL,		factor_cb, 100,		NULL },
  { N_("/View/_200%"),			NULL,		factor_cb, 200,		NULL },
  { "/View/sep4",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/Zoom _in"),		NULL,		zoom_in_cb, 0,		STOCK(ZOOM_IN) },
  { N_("/View/Zoom _out"),		NULL,		zoom_out_cb, 0,		STOCK(ZOOM_OUT) },
  { N_("/View/_Zoom 100%"),		NULL,		zoom_100_cb, 0,		STOCK(ZOOM_100) },
  { "/View/sep5",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Deinterlace"),		"<control>I",	deinterlace_cb, 0,	"<CheckItem>" },
  { N_("/View/Auto _resize"),		NULL,		auto_resize_cb, 0,	"<CheckItem>" },
  { N_("/View/_Aspect ratio"),		NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Aspect ratio/_Auto"),	NULL,		ASPECT(AUTO),		"<RadioItem>" },
  { N_("/View/Aspect ratio/_Square"),	NULL,		ASPECT(SQUARE),		"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_4:3"),	NULL,		ASPECT(4_3),		"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_16:9"),	NULL,		ASPECT(ANAMORPHIC),	"/View/Aspect ratio/Auto"},
  { N_("/View/Aspect ratio/_2:1"),	NULL,		ASPECT(DVB),		"/View/Aspect ratio/Auto"},
  { N_("/View/A\\/V _settings..."),	NULL,		settings_cb, 0,		NULL },
  { "/View/sep6",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Visualisations"),	NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Visualisations/_None"),	NULL,		vis_none_cb, 0,		"<RadioItem>"},
  { "/View/sep7",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/View/_Subtitles"),		NULL,		NULL, 0,		"<Branch>" },
  { N_("/View/Subtitles/_Auto"),	NULL,		sub_cb, -1,		"<RadioItem>"},
  { N_("/View/Subtitles/_None"),	NULL,		sub_cb, -2,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _0"),	NULL,		sub_cb, 0,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _1"),	NULL,		sub_cb, 1,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _2"),	NULL,		sub_cb, 2,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _3"),	NULL,		sub_cb, 3,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _4"),	NULL,		sub_cb, 4,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _5"),	NULL,		sub_cb, 5,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _6"),	NULL,		sub_cb, 6,		"/View/Subtitles/Auto"},
  { N_("/View/Subtitles/Channel _7"),	NULL,		sub_cb, 7,		"/View/Subtitles/Auto"},
  { N_("/_Media"),			NULL,		NULL, 0,		"<Branch>" },
  { N_("/Media/_Add media mark..."),	"<control>D",	mediamarks_add_cb, 0,	STOCK(NEW) },
  { N_("/Media/_Manage media marks..."), "<control>B",	mediamarks_manage_cb, 0, NULL },
  { N_("/Media/_Import media marks..."), NULL,		mediamarks_import_cb, 0, STOCK(OPEN) },
  { "/Media/sep8",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/_Help"),			NULL,		NULL, 0,		"<LastBranch>" },
  { N_("/Help/_About..."),		NULL,		about_cb, 0,		NULL },
  { N_("/Help/Engine _Log..."),		"<control>L",	log_cb, 0,		NULL },
  { N_("/Help/Stream _Info..."),	NULL,		stream_info_cb, 0,	NULL },
  { "/Help/sep6",			NULL,		NULL, 0,		"<Separator>" },
  { N_("/Help/_Re-run setup wizards..."), NULL,		wizards_cb, 0,		NULL },
};

GtkWidget *create_menu_tree (GtkItemFactory **factory, GtkWidget *window,
			     GtkAccelGroup **accel,
			     GtkItemFactoryEntry *menu_items,
			     gint nmenu_items)
{
  if (window)
  {
    GtkAccelGroup *accel_group =
      (accel && *accel) ? *accel : gtk_accel_group_new ();
    if (accel)
      *accel = accel_group;
    *factory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>",
				     accel_group);
#ifdef ENABLE_NLS
    gtk_item_factory_set_translate_func (GTK_ITEM_FACTORY (*factory),
					 (GtkTranslateFunc) gettext,
					 NULL, NULL);
#endif
    gtk_item_factory_create_items (*factory, nmenu_items, menu_items, NULL);
    gtk_window_add_accel_group (GTK_WINDOW (window), accel_group);
  }
  else
  {
    *factory = gtk_item_factory_new (GTK_TYPE_MENU, "<main>", NULL);
#ifdef ENABLE_NLS
    gtk_item_factory_set_translate_func (GTK_ITEM_FACTORY (*factory),
					 (GtkTranslateFunc) gettext,
					 NULL, NULL);
#endif
    gtk_item_factory_create_items (*factory, nmenu_items, menu_items, NULL);
  }

  return gtk_item_factory_get_widget (*factory, "<main>");
}


void create_menus (GtkWidget *window)
{
  GtkItemFactory *item_factory[2];
  GtkRadioMenuItem *none_item;
  int i;

  if (window)
    menubar = create_menu_tree (&item_factory[0], window, NULL,
				menu_items, num_menu_items (menu_items));
  popup_menu = create_menu_tree (&item_factory[1], NULL, NULL,
				 menu_items, num_menu_items (menu_items));

  for (i = !window; i < 2; ++i)
  {
    GtkMenuShell *menu;
    xine_cfg_entry_t entry;

    media_menu[i] = GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i], "/Media"));

    fullscreen_menu_item[i]  = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen mode"));

    deinterlace_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Deinterlace"));
    if (xine_config_lookup_entry (xine, "gui.deinterlace_by_default", &entry))
      deinterlace_menu_item[i]->active = entry.num_value;

    auto_resize_menu_item[i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Auto resize"));
    auto_resize_menu_item[i]->active = TRUE;

    toolbar_item[0][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/Visible"));
    toolbar_item[1][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/At bottom"));
    toolbar_item[2][i] = GTK_CHECK_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Fullscreen toolbar/At top"));
    if (xine_config_lookup_entry (xine, "gui.fullscreen_toolbar", &entry))
    {
      toolbar_item[0][i]->active = (entry.num_value >> 1) & 1;
      toolbar_item[1][i]->active = (entry.num_value & 1);
      toolbar_item[2][i]->active = !(entry.num_value & 1);
    }

    none_item = GTK_RADIO_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Aspect ratio/Auto"));
    aspect_menu_items[i] = gtk_radio_menu_item_get_group (none_item);

    menu = GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i],
							"/File"));
    add_autoplay_entries (menu, g_list_index (menu->children,
				  gtk_item_factory_get_widget (item_factory[i],
							       "/File/sep1")));

    none_item = GTK_RADIO_MENU_ITEM (gtk_item_factory_get_item (item_factory[i], "/View/Visualisations/None"));
    add_vis_entries (GTK_MENU_SHELL (gtk_item_factory_get_widget (item_factory[i], "/View/Visualisations")), none_item);

  }

  g_signal_connect (GTK_OBJECT(popup_menu), "key-press-event",
		    G_CALLBACK(popup_key_cb), NULL);
}
