// (C) 2001, Cornelius Schumacher

#include <qstringlist.h>

#include <ksimpleconfig.h>
#include <kstandarddirs.h>
#include <kurl.h>
#include <kdebug.h>

#include "packageimpl.h"
#include "bugimpl.h"
#include "bugdetailsimpl.h"

#include "bugcache.h"

BugCache::BugCache()
{
    m_cachePackages = new KSimpleConfig(locateLocal("appdata","packages.cache"));
    m_cacheBugs = new KSimpleConfig(locateLocal("appdata","bugs.cache"));
}

BugCache::~BugCache()
{
    m_cachePackages->sync();
    m_cacheBugs->sync();

    delete m_cachePackages;
    delete m_cacheBugs;
}

void BugCache::savePackageList( const Package::List &pkgs )
{
    Package::List::ConstIterator it;
    for (it = pkgs.begin(); it != pkgs.end(); ++it) {
        m_cachePackages->setGroup((*it).name());
        m_cachePackages->writeEntry("numberOfBugs",(*it).numberOfBugs());
        writePerson(m_cachePackages,"Maintainer",(*it).maintainer());
    }
}

Package::List BugCache::loadPackageList()
{
    Package::List pkgs;

    QStringList packages = m_cachePackages->groupList();
    QStringList::ConstIterator it;
    for( it = packages.begin(); it != packages.end(); ++it ) {
        if ((*it) == "<default>") continue;
        m_cachePackages->setGroup(*it);
        int numberOfBugs = m_cachePackages->readNumEntry("numberOfBugs");
        Person maintainer = readPerson( m_cachePackages, "Maintainer");
        pkgs.append( Package( new PackageImpl( (*it), numberOfBugs,
                                                maintainer ) ) );
    }

    return pkgs;
}

void BugCache::invalidatePackageList()
{
    // Completely wipe out packages.cache
    QStringList packages = m_cachePackages->groupList();
    QStringList::ConstIterator it;
    for( it = packages.begin(); it != packages.end(); ++it ) {
        if ((*it) == "<default>") continue;
        m_cachePackages->deleteGroup(*it, true);
    }
}

void BugCache::saveBugList( const Package &pkg, const Bug::List &bugs )
{
    QStringList bugList;

    Bug::List::ConstIterator it;
    for( it = bugs.begin(); it != bugs.end(); ++it ) {
        QString number = (*it).number();
        bugList.append( number );
        m_cacheBugs->setGroup( number );
        m_cacheBugs->writeEntry( "Title", (*it).title() );
        m_cacheBugs->writeEntry( "Severity", Bug::severityToString((*it).severity()) );
        m_cacheBugs->writeEntry( "Status", Bug::statusToString((*it).status()) );
        m_cacheBugs->writeEntry( "Age", (*it).age() );
        writePerson( m_cacheBugs, "Submitter", (*it).submitter() );
    }

    m_cachePackages->setGroup( pkg.name() );
    m_cachePackages->writeEntry( "bugList", bugList );
}

Bug::List BugCache::loadBugList( const Package &pkg, bool disconnected )
{
//    kdDebug() << "Loading bug list for " << pkg.name() << endl;

    Bug::List bugList;

    m_cachePackages->setGroup( pkg.name() );
    QStringList bugs = m_cachePackages->readListEntry( "bugList" );

//    kdDebug() << "  Bugs: " << (bugs.join(",")) << endl;

    QStringList::ConstIterator it;
    for( it = bugs.begin(); it != bugs.end(); ++it ) {
        if ( m_cacheBugs->hasGroup(*it) )
        {
            m_cacheBugs->setGroup(*it);
            QString title = m_cacheBugs->readEntry("Title");
            if ( !title.isEmpty() ) // dunno how I ended up with an all empty bug in the cache
            {
                Person submitter = readPerson( m_cacheBugs, "Submitter" );
                Bug::Status status = Bug::stringToStatus( m_cacheBugs->readEntry("Status") );
                Bug::Severity severity = Bug::stringToSeverity( m_cacheBugs->readEntry("Severity") );
                uint age = m_cacheBugs->readNumEntry( "Age" );
                bugList.append( Bug( new BugImpl( title, submitter, (*it),
                                                  age, severity, status ) ) );
            }
        } else {
            // This bug is in the package cache's buglist but not in the bugs cache
            // Probably a new bug, we need to fetch it - if we're not in disconnected mode
            kdWarning() << "Bug " << *it << " not in bug cache" << endl;
            if ( !disconnected )
                return Bug::List(); // returning an empty list will trigger a reload of the buglist
        }
    }

    return bugList;
}

void BugCache::invalidateBugList( const Package& pkg )
{
    kdDebug() << "BugCache::invalidateBugList " << pkg.name() << endl;
    // Erase bug list for this package
    m_cachePackages->setGroup( pkg.name() );
    m_cachePackages->writeEntry("bugList",QString::null);
}

void BugCache::saveBugDetails( const Bug &bug, const BugDetails &details )
{
    m_cacheBugs->setGroup( bug.number() );

    m_cacheBugs->writeEntry( "Version", details.version() );
    m_cacheBugs->writeEntry( "Source", details.source() );
    m_cacheBugs->writeEntry( "Compiler", details.compiler() );
    m_cacheBugs->writeEntry( "OS", details.os() );

    QStringList senders;
    QStringList texts;
    QStringList dates;

    BugDetailsPart::List parts = details.parts();
    BugDetailsPart::List::ConstIterator it;
    for ( it = parts.begin(); it != parts.end(); ++it ) {
        senders.append( (*it).sender.fullName() );
        texts.append( (*it).text );
        dates.append( (*it).date );
    }

    m_cacheBugs->writeEntry( "Details", texts );
    m_cacheBugs->writeEntry( "Senders", senders );
    m_cacheBugs->writeEntry( "Dates", dates );
}

BugDetails BugCache::loadBugDetails( const Bug &bug )
{
    if ( !m_cacheBugs->hasGroup( bug.number() ) ) {
        return BugDetails();
    }

    m_cacheBugs->setGroup( bug.number() );

    BugDetailsPart::List parts;

    QStringList texts = m_cacheBugs->readListEntry( "Details" );
    QStringList senders = m_cacheBugs->readListEntry( "Senders" );
    QStringList dates = m_cacheBugs->readListEntry( "Dates" );

    QStringList::ConstIterator itTexts = texts.begin();
    QStringList::ConstIterator itSenders = senders.begin();
    QStringList::ConstIterator itDates = dates.begin();
    while( itTexts != texts.end() ) {
        parts.append( BugDetailsPart( Person(*itSenders), *itDates, *itTexts ) );

        ++itTexts;
        ++itSenders;
        ++itDates;
    }

    if ( parts.count() == 0 ) {
        return BugDetails();
    }

    QString version = m_cacheBugs->readEntry( "Version" );
    QString source = m_cacheBugs->readEntry( "Source" );
    QString compiler = m_cacheBugs->readEntry( "Compiler" );
    QString os = m_cacheBugs->readEntry( "OS" );

    return BugDetails( new BugDetailsImpl( version, source, compiler, os,
                                           parts ) );
}

void BugCache::invalidateBugDetails( const Bug& bug )
{
    m_cacheBugs->deleteGroup( bug.number(), true );
}

void BugCache::writePerson( KSimpleConfig *file, const QString &key,
                            const Person &p )
{
    QStringList values;
    values.append(p.name);
    values.append(p.email);
    file->writeEntry( key, values );
}

struct Person BugCache::readPerson( KSimpleConfig *file, const QString &key )
{
    struct Person p;
    QStringList values = file->readListEntry(key);
    if ( values.count() > 0 )
        p.name = values[0];
    if ( values.count() > 1 )
        p.email = values[1];
    return p;
}
