/*
 *   Copyright (C) 1991-2000 by Jonathan Naylor HB9DRD/G4KLX
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include <gtk/gtk.h>

#include "global.h"

static void SatEditCb(GtkWidget *, gpointer);
static void LocEditCb(GtkWidget *, gpointer);
static void UserEditCb(GtkWidget *, gpointer);

static void Update_Sat(struct Sat_Struct *);
static void Delete_Sat(struct Sat_Struct *);
static void Delete_Sat_OK(void);

static void Update_Loc(struct Loc_Struct *);
static void Delete_Loc(struct Loc_Struct *);
static void Delete_Loc_OK(void);

static GtkWidget *SatName;
static GtkWidget *SatCat;
static GtkWidget *SatNo;
static GtkWidget *SatEpoch;
static GtkWidget *SatMeanAnom;
static GtkWidget *SatMeanMot;
static GtkWidget *SatIncl;
static GtkWidget *SatEccen;
static GtkWidget *SatArgPer;
static GtkWidget *SatRaan;
static GtkWidget *SatDrag;
static GtkWidget *SatOrbit;
static GtkWidget *SatAlon;
static GtkWidget *SatAlat;
static GtkWidget *SatModes;
static GtkWidget *SatFreq[3];

static GtkWidget *LocName;
static GtkWidget *LocLat;
static GtkWidget *LocLong;
static GtkWidget *LocHeight;

static GtkWidget *UserCall;
static GtkWidget *UserLat;
static GtkWidget *UserLong;
static GtkWidget *UserHeight;
static GtkWidget *UserTime;
static GtkWidget *UserTrack;
static GtkWidget *UserList;
static GtkWidget *UserDate;

static int Parse_Line(char *, char **);
static int Range_Check(double, double, double);

static struct Sat_Struct *Sat_Selection = NULL;
static struct Loc_Struct *Loc_Selection = NULL;

static void SatEditCb(GtkWidget *w, gpointer data)
{
	GtkWidget *window = (GtkWidget *)data;
	struct Sat_Struct Temp_Data, *Data;
	double Epoch;
	char *s;
	char *argv[10];
	char *MA_Day_Name;
	int  argc;
	int  i;
	
	memset(&Temp_Data, '\0', sizeof(struct Sat_Struct));

	Temp_Data.Aflag    = FALSE;
	Temp_Data.No_Modes = 0;

	s = gtk_entry_get_text(GTK_ENTRY(SatName));
	strcpy(Temp_Data.Name, s);
	if (strlen(Temp_Data.Name) == 0) {
		Warning_Box(" A Satellite name must be entered");
		return;
	}
	if (Sat_Selection == NULL) {
		if (Find_Satellite_Name(Temp_Data.Name) != NULL) {
			Warning_Box("Satellite name is already used");
			return;
		}
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatCat));
	strcpy(Temp_Data.Category, s);

	s = gtk_entry_get_text(GTK_ENTRY(SatNo));
	Temp_Data.Catalogue_No = atol(s);
	if (Temp_Data.Catalogue_No < 0L) {
		Warning_Box("Catalogue number must be greater than zero");
		return;
	}
	if (Sat_Selection == NULL) {
		if (Find_Satellite_Cat_No(Temp_Data.Catalogue_No) != NULL) {
			Warning_Box("Catalogue Number is already used");
			return;
		}
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatEpoch));
	Epoch = atof(s);

	Temp_Data.Epoch_Year = Epoch / 1000;
	if (!Range_Check((double)Temp_Data.Epoch_Year, 1988.0, 2010.0)) {
		Warning_Box("Epoch Year must be between\n1988 and 2010");
		return;
	}

	Temp_Data.Epoch_Day = Epoch - (double)(Temp_Data.Epoch_Year * 1000);
	if (!Range_Check(Temp_Data.Epoch_Day, 0.0, 367.0)) {
		Warning_Box("Epoch Day must be between\n0 and 365");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatIncl));
	Temp_Data.Inclination = atof(s);
	if (!Range_Check(Temp_Data.Inclination, 0.0, 180.0)) {
		Warning_Box("Inclination must be between\n0 and 180 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatMeanAnom));
	Temp_Data.Mean_Anomoly = atof(s);
	if (!Range_Check(Temp_Data.Mean_Anomoly, 0.0, 360.0)) {
		Warning_Box("Mean Anomoly must be between\n0 and 360 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatMeanMot));
	Temp_Data.Mean_Motion = atof(s);
	if (!Range_Check(Temp_Data.Mean_Motion, 0.1, 19.0)) {
		Warning_Box("Mean Motion must be between\n0.1 and 19 revs/day");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatEccen));
	Temp_Data.Eccentricity = atof(s);
	if (!Range_Check(Temp_Data.Eccentricity, 0.0, 0.999)) {
		Warning_Box("Eccentricity must be between\n0 and 1");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatArgPer));
	Temp_Data.Arg_Of_Perigee = atof(s);
	if (!Range_Check(Temp_Data.Arg_Of_Perigee, 0.0, 360.0)) {
		Warning_Box("Arg of Perigee must be between\n0 and 360 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatRaan));
	Temp_Data.RAAN = atof(s);
	if (!Range_Check(Temp_Data.RAAN, 0.0, 360.0)) {
		Warning_Box("RAAN must be between\n0 and 360 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatDrag));
	Temp_Data.Drag = atof(s);
	if (!Range_Check(Temp_Data.Drag, -0.5, 0.5)) {
		Warning_Box("Drag must be between\n+-0.5 revs/day/day");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatOrbit));
	Temp_Data.Epoch_Orbit = atol(s);
	if (Temp_Data.Epoch_Orbit < 0L) {
		Warning_Box("Orbit must be positive");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatAlon));
	if ((argc = Parse_Line(s, argv)) != 0) {
		Temp_Data.Alon = atof(argv[0]);
		if (!Range_Check(Temp_Data.Alon, 0.0, 360.0)) {
			Warning_Box("Alon must be between\n0 and 360 degrees");
			return;
		}
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatAlat));
	if ((argc = Parse_Line(s, argv)) != 0) {
		Temp_Data.Alat = atof(argv[0]);
		if (!Range_Check(Temp_Data.Alat, -90.0, 90.0)) {
			Warning_Box("Alat must be between\n+-90 degrees");
			return;
		}
	}

	if (Temp_Data.Alon != 0.0 && Temp_Data.Alat != 0.0)
		Temp_Data.Aflag = TRUE;

	s = gtk_editable_get_chars(GTK_EDITABLE(SatModes), 0, -1);

	argc = Parse_Line(s, argv);

	for (i = 0; i < argc; i++) {
		Temp_Data.MA_Day[Temp_Data.No_Modes] = -1;

		MA_Day_Name = strtok(argv[i], ":");

		if (strcasecmp(MA_Day_Name, "Sun") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1000;

		if (strcasecmp(MA_Day_Name, "Mon") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1001;

		if (strcasecmp(MA_Day_Name, "Tue") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1002;

		if (strcasecmp(MA_Day_Name, "Wed") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1003;

		if (strcasecmp(MA_Day_Name, "Thu") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1004;

		if (strcasecmp(MA_Day_Name, "Fri") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1005;

		if (strcasecmp(MA_Day_Name, "Sat") == 0)
			Temp_Data.MA_Day[Temp_Data.No_Modes] = 1006;

		if (Temp_Data.MA_Day[Temp_Data.No_Modes] == -1) {
			if (!Range_Check((double)atoi(MA_Day_Name), 0.0, 256.0)) {
				Warning_Box("MA must be between\n0 and 256");
				return;
			}
			Temp_Data.MA_Day[Temp_Data.No_Modes] = atoi(MA_Day_Name);
		}

		strncpy(Temp_Data.Mode[Temp_Data.No_Modes], strtok(NULL, ":"), 4);

		Temp_Data.No_Modes++;
	}

	g_free(s);

	s = gtk_entry_get_text(GTK_ENTRY(SatFreq[0]));
	Temp_Data.Frequency1 = atof(s);
	if (!Range_Check(Temp_Data.Frequency1, 0.0, 24000.0)) {
		Warning_Box("Frequency must be between\n0 and 24GHz");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatFreq[1]));
	Temp_Data.Frequency2 = atof(s);
	if (!Range_Check(Temp_Data.Frequency2, 0.0, 24000.0)) {
		Warning_Box("Frequency must be between\n0 and 24GHz");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(SatFreq[2]));
	Temp_Data.Frequency3 = atof(s);
	if (!Range_Check(Temp_Data.Frequency3, 0.0, 24000.0)) {
		Warning_Box("Frequency must be between\n0 and 24GHz");
		return;
	}

	if (Sat_Selection != NULL) {
		Sat_List = g_slist_remove(Sat_List, Sat_Selection);
		g_free(Sat_Selection);
	}

	Data  = g_malloc(sizeof(struct Sat_Struct));
	*Data = Temp_Data;

	Sat_List = g_slist_insert_sorted(Sat_List, Data, Sat_Comparison);

	if (Write_Sat_Data())
		gtk_widget_destroy(window);
}

void Add_Sat_Data(void)
{
	Update_Sat(NULL);
}

void Update_Sat_Data(void)
{
	Choose_Satellite(Update_Sat);
}

static void Update_Sat(struct Sat_Struct *Sat)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *notebook;
	GtkWidget *notelabel[3];
	GtkWidget *table[3];
	GtkWidget *label[18];
	GtkWidget *button[3];
	GtkWidget *bbox;
	char ma_buffer[101];
	char buffer[101];
	int i;
	
	window = gtk_window_new(GTK_WINDOW_DIALOG);

	gtk_window_set_title(GTK_WINDOW(window), "Edit Satellite");
	gtk_container_set_border_width(GTK_CONTAINER(window), BORDER_WIDTH);

	vbox = gtk_vbox_new(FALSE, BORDER_WIDTH);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	notelabel[0] = gtk_label_new("Identification");
	table[0]     = gtk_table_new(3, 2, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table[0], notelabel[0]);

	label[0]  = gtk_label_new("Name");
	label[1]  = gtk_label_new("Categories");
	label[2]  = gtk_label_new("Catalogue No");

	SatName   = gtk_entry_new_with_max_length(19);
	SatCat    = gtk_entry_new_with_max_length(19);
	SatNo     = gtk_entry_new_with_max_length(19);

	gtk_table_attach(GTK_TABLE(table[0]), label[0], 0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), label[1], 0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), label[2], 0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), SatName,  1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), SatCat,   1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), SatNo,    1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	notelabel[1] = gtk_label_new("Elements");
	table[1]     = gtk_table_new(9, 2, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table[1], notelabel[1]);

	label[3]  = gtk_label_new("Epoch");
	label[4]  = gtk_label_new("Inclination");
	label[5]  = gtk_label_new("Mean Anomoly");
	label[6]  = gtk_label_new("Mean Motion");
	label[7]  = gtk_label_new("Eccentricity");
	label[8]  = gtk_label_new("Arg of Perigee");
	label[9]  = gtk_label_new("RAAN");
	label[10] = gtk_label_new("Orbit No");
	label[11] = gtk_label_new("Drag");

	SatEpoch    = gtk_entry_new_with_max_length(19);
	SatIncl     = gtk_entry_new_with_max_length(19);
	SatMeanAnom = gtk_entry_new_with_max_length(19);
	SatMeanMot  = gtk_entry_new_with_max_length(19);
	SatEccen    = gtk_entry_new_with_max_length(19);
	SatArgPer   = gtk_entry_new_with_max_length(19);
	SatRaan     = gtk_entry_new_with_max_length(19);
	SatOrbit    = gtk_entry_new_with_max_length(19);
	SatDrag     = gtk_entry_new_with_max_length(19);

	gtk_table_attach(GTK_TABLE(table[1]), label[3],    0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[4],    0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[5],    0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[6],    0, 1, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[7],    0, 1, 4, 5,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[8],    0, 1, 5, 6,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[9],    0, 1, 6, 7,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[10],   0, 1, 7, 8,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[11],   0, 1, 8, 9,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatEpoch,    1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatIncl,     1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatMeanAnom, 1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatMeanMot,  1, 2, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatEccen,    1, 2, 4, 5,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatArgPer,   1, 2, 5, 6,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatRaan,     1, 2, 6, 7,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatOrbit,    1, 2, 7, 8,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), SatDrag,     1, 2, 8, 9,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	notelabel[2] = gtk_label_new("Miscellaneous");
	table[2]     = gtk_table_new(6, 2, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table[2], notelabel[2]);

	label[12] = gtk_label_new("Alon");
	label[13] = gtk_label_new("Alat");
	label[14] = gtk_label_new("Mode List");
	label[15] = gtk_label_new("Frequency 1");
	label[16] = gtk_label_new("Frequency 2");
	label[17] = gtk_label_new("Frequency 3");

	SatAlon    = gtk_entry_new_with_max_length(19);
	SatAlat    = gtk_entry_new_with_max_length(19);
	SatModes   = gtk_entry_new_with_max_length(19);
	SatFreq[0] = gtk_entry_new_with_max_length(19);
	SatFreq[1] = gtk_entry_new_with_max_length(19);
	SatFreq[2] = gtk_entry_new_with_max_length(19);

	gtk_table_attach(GTK_TABLE(table[2]), label[12],   0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), label[13],   0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), label[14],   0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), label[15],   0, 1, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), label[16],   0, 1, 4, 5,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), label[17],   0, 1, 5, 6,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatAlon,     1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatAlat,     1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatModes,    1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatFreq[0],  1, 2, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatFreq[1],  1, 2, 4, 5,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[2]), SatFreq[2],  1, 2, 5, 6,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, TRUE, TRUE, 0);

	button[0] = gtk_button_new_with_label("Save");
	button[1] = gtk_button_new_with_label("Cancel");
	button[2] = gtk_button_new_with_label("Help");

	gtk_container_add(GTK_CONTAINER(bbox), button[0]);
	gtk_container_add(GTK_CONTAINER(bbox), button[1]);
	gtk_container_add(GTK_CONTAINER(bbox), button[2]);

	gtk_signal_connect(GTK_OBJECT(button[0]), "clicked", GTK_SIGNAL_FUNC(SatEditCb), window);
	gtk_signal_connect_object(GTK_OBJECT(button[1]), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
	gtk_signal_connect(GTK_OBJECT(button[2]), "clicked", GTK_SIGNAL_FUNC(HelpCb), (gpointer)3);

	GTK_WIDGET_SET_FLAGS(button[0], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[1], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[2], GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button[0]);

	if (Sat != NULL) {
		gtk_entry_set_text(GTK_ENTRY(SatName), Sat->Name);

		gtk_entry_set_text(GTK_ENTRY(SatCat), Sat->Category);

		sprintf(buffer, "%ld", Sat->Catalogue_No);
		gtk_entry_set_text(GTK_ENTRY(SatNo), buffer);

		sprintf(buffer, "%.8f", Sat->Epoch_Day + (double)(Sat->Epoch_Year * 1000));
		gtk_entry_set_text(GTK_ENTRY(SatEpoch), buffer);

		sprintf(buffer, "%.4f", Sat->Inclination);
		gtk_entry_set_text(GTK_ENTRY(SatIncl), buffer);

		sprintf(buffer, "%.4f", Sat->Mean_Anomoly);
		gtk_entry_set_text(GTK_ENTRY(SatMeanAnom), buffer);

		sprintf(buffer, "%.8f", Sat->Mean_Motion);
		gtk_entry_set_text(GTK_ENTRY(SatMeanMot), buffer);

		sprintf(buffer, "%.7f", Sat->Eccentricity);
		gtk_entry_set_text(GTK_ENTRY(SatEccen), buffer);

		sprintf(buffer, "%.4f", Sat->Arg_Of_Perigee);
		gtk_entry_set_text(GTK_ENTRY(SatArgPer), buffer);

		sprintf(buffer, "%.4f", Sat->RAAN);
		gtk_entry_set_text(GTK_ENTRY(SatRaan), buffer);

		sprintf(buffer, "%ld", Sat->Epoch_Orbit);
		gtk_entry_set_text(GTK_ENTRY(SatOrbit), buffer);

		sprintf(buffer, "%e", Sat->Drag);
		gtk_entry_set_text(GTK_ENTRY(SatDrag), buffer);

		if (Sat->Aflag) {
			sprintf(buffer, "%.2f", Sat->Alon);
			gtk_entry_set_text(GTK_ENTRY(SatAlon), buffer);

			sprintf(buffer, "%.2f", Sat->Alat);
			gtk_entry_set_text(GTK_ENTRY(SatAlat), buffer);
		} else {
			gtk_entry_set_text(GTK_ENTRY(SatAlon), "");
			gtk_entry_set_text(GTK_ENTRY(SatAlat), "");
		}

		*buffer = '\0';

		for (i = 0; i < Sat->No_Modes; i++) {
			if (Sat->MA_Day[i] > 256) {
				switch (Sat->MA_Day[i]) {
					case 1000:
						strcat(buffer, "Sun:");
						break;
					case 1001:
						strcat(buffer, "Mon:");
						break;
					case 1002:
						strcat(buffer, "Tue:");
						break;
					case 1003:
						strcat(buffer, "Wed:");
						break;
					case 1004:
						strcat(buffer, "Thu:");
						break;
					case 1005:
						strcat(buffer, "Fri:");
						break;
					case 1006:
						strcat(buffer, "Sat:");
						break;
					default:
						strcat(buffer, "???:");
						break;
				}
			} else {
				sprintf(ma_buffer, "%03d:", Sat->MA_Day[i]);
				strcat(buffer, ma_buffer);
			}

			strcat(buffer, Sat->Mode[i]);
			strcat(buffer, " ");
		}

		gtk_entry_set_text(GTK_ENTRY(SatModes), buffer);

		if (Sat->Frequency1 != 0.0) {
			sprintf(buffer, "%.3f", Sat->Frequency1);
			gtk_entry_set_text(GTK_ENTRY(SatFreq[0]), buffer);
		} else {
			gtk_entry_set_text(GTK_ENTRY(SatFreq[0]), "");
		}

		if (Sat->Frequency2 != 0.0) {
			sprintf(buffer, "%.3f", Sat->Frequency2);
			gtk_entry_set_text(GTK_ENTRY(SatFreq[1]), buffer);
		} else {
			gtk_entry_set_text(GTK_ENTRY(SatFreq[1]), "");
		}

		if (Sat->Frequency3 != 0.0) {
			sprintf(buffer, "%.3f", Sat->Frequency3);
			gtk_entry_set_text(GTK_ENTRY(SatFreq[2]), buffer);
		} else {
			gtk_entry_set_text(GTK_ENTRY(SatFreq[2]), "");
		}
	}

	Sat_Selection = Sat;

	for (i = 0; i < 18; i++)
		gtk_widget_show(label[i]);

	gtk_widget_show(SatName);
	gtk_widget_show(SatCat);
	gtk_widget_show(SatNo);
	gtk_widget_show(SatEpoch);
	gtk_widget_show(SatIncl);
	gtk_widget_show(SatMeanAnom);
	gtk_widget_show(SatMeanMot);
	gtk_widget_show(SatEccen);
	gtk_widget_show(SatArgPer);

	gtk_widget_show(SatRaan);
	gtk_widget_show(SatOrbit);
	gtk_widget_show(SatDrag);
	gtk_widget_show(SatAlon);
	gtk_widget_show(SatAlat);
	gtk_widget_show(SatModes);

	for (i = 0; i < 3; i++)
		gtk_widget_show(SatFreq[i]);

	for (i = 0; i < 3; i++)
		gtk_widget_show(button[i]);

	gtk_widget_show(bbox);

	for (i = 0; i < 3; i++) {
		gtk_widget_show(table[i]);
		gtk_widget_show(notelabel[i]);
	}

	gtk_widget_show(notebook);
	gtk_widget_show(vbox);
	gtk_widget_show(window);

}

void Delete_Sat_Data(void)
{
	Choose_Satellite(Delete_Sat);
}

static void Delete_Sat(struct Sat_Struct *Sat)
{
	char buffer[40];

	Sat_Selection = Sat;

	sprintf(buffer, "Delete %s ?", Sat_Selection->Name);

	YN_Box(buffer, Delete_Sat_OK, (void (*)(void))NULL);
}

static void Delete_Sat_OK(void)
{
	Sat_List = g_slist_remove(Sat_List, Sat_Selection);

	g_free(Sat_Selection);

	Write_Sat_Data();
}

static void LocEditCb(GtkWidget *w, gpointer data)
{
	GtkWidget *window = (GtkWidget *)data;
	struct Loc_Struct Temp_Data, *Temp;
	char *s;
	
	memset(&Temp_Data, '\0', sizeof(struct Loc_Struct));

	s = gtk_entry_get_text(GTK_ENTRY(LocName));
	strcpy(Temp_Data.QTH, s);
	if (strlen(Temp_Data.QTH) == 0) {
		Warning_Box("A Location must be entered");
		return;
	}
	if (Loc_Selection == NULL) {
		if (Find_Location_Name(Temp_Data.QTH) != NULL) {
			Warning_Box("Location name is already used");
			return;
		}
	}

	s = gtk_entry_get_text(GTK_ENTRY(LocLat));
	Temp_Data.Latitude = atof(s);
	if (!Range_Check(Temp_Data.Latitude, -90.0, 90.0)) {
		Warning_Box("Latitude must be between\n+-90 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(LocLong));
	Temp_Data.Longitude = -atof(s);
	if (Temp_Data.Longitude < 0.0)
		Temp_Data.Longitude += 360.0;
	if (!Range_Check(Temp_Data.Longitude, 0.0, 360.0)) {
		Warning_Box("Longitude must be between\n+-360 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(LocHeight));
	Temp_Data.Height = atof(s);
	if (Temp_Data.Height < -10.0) {
		Warning_Box("Height must be greater than -10m");
		return;
	}

	if (Loc_Selection != NULL) {
		Loc_List = g_slist_remove(Loc_List, Loc_Selection);
		g_free(Loc_Selection);
	}

	Temp  = g_malloc(sizeof(struct Loc_Struct));
	*Temp = Temp_Data;

	Loc_List = g_slist_insert_sorted(Loc_List, Temp, Loc_Comparison);

	if (Write_Loc_Data())
		gtk_widget_destroy(window);
}

void Add_Loc_Data(void)
{
	Update_Loc(NULL);
}

void Update_Loc_Data(void)
{
	Choose_Location(Update_Loc);
}

void Update_Loc(struct Loc_Struct *Loc)
{
	GtkWidget *window;
	GtkWidget *hbox;
	GtkWidget *frame;
	GtkWidget *table;
	GtkWidget *label[4];
	GtkWidget *button[3];
	GtkWidget *bbox;
	char buffer[20];
	int i;
	
	window = gtk_window_new(GTK_WINDOW_DIALOG);

	gtk_window_set_title(GTK_WINDOW(window), "Edit Location");
	gtk_container_set_border_width(GTK_CONTAINER(window), BORDER_WIDTH);

	hbox = gtk_hbox_new(FALSE, BORDER_WIDTH);
	gtk_container_add(GTK_CONTAINER(window), hbox);

	frame = gtk_frame_new(NULL);
	gtk_box_pack_start(GTK_BOX(hbox), frame, TRUE, TRUE, 0);

	table = gtk_table_new(4, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);

	label[0] = gtk_label_new("Location");
	label[1] = gtk_label_new("Latitude");
	label[2] = gtk_label_new("Longitude");
	label[3] = gtk_label_new("Height");

	LocName   = gtk_entry_new_with_max_length(50);
	LocLat    = gtk_entry_new_with_max_length(50);
	LocLong   = gtk_entry_new_with_max_length(50);
	LocHeight = gtk_entry_new_with_max_length(50);

	gtk_table_attach(GTK_TABLE(table), label[0],  0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), label[1],  0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), label[2],  0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), label[3],  0, 1, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	gtk_table_attach(GTK_TABLE(table), LocName,   1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), LocLat,    1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), LocLong,   1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table), LocHeight, 1, 2, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	bbox = gtk_vbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_START);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	gtk_box_pack_end(GTK_BOX(hbox), bbox, TRUE, TRUE, 0);

	button[0] = gtk_button_new_with_label("Save");
	button[1] = gtk_button_new_with_label("Cancel");
	button[2] = gtk_button_new_with_label("Help");

	gtk_container_add(GTK_CONTAINER(bbox), button[0]);
	gtk_container_add(GTK_CONTAINER(bbox), button[1]);
	gtk_container_add(GTK_CONTAINER(bbox), button[2]);

	gtk_signal_connect(GTK_OBJECT(button[0]), "clicked", GTK_SIGNAL_FUNC(LocEditCb), window);
	gtk_signal_connect_object(GTK_OBJECT(button[1]), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
	gtk_signal_connect(GTK_OBJECT(button[2]), "clicked", GTK_SIGNAL_FUNC(HelpCb), (gpointer)5);

	GTK_WIDGET_SET_FLAGS(button[0], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[1], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[2], GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button[0]);

	if (Loc != NULL) {
		gtk_entry_set_text(GTK_ENTRY(LocName), Loc->QTH);

		sprintf(buffer, "%.2f", Loc->Latitude);
		gtk_entry_set_text(GTK_ENTRY(LocLat), buffer);

		if (Loc->Longitude > 180.0)
			sprintf(buffer, "%.2f", -(Loc->Longitude - 360.0));
		else
			sprintf(buffer, "%.2f", -Loc->Longitude);
		gtk_entry_set_text(GTK_ENTRY(LocLong), buffer);

		sprintf(buffer, "%.0f", Loc->Height);
		gtk_entry_set_text(GTK_ENTRY(LocHeight), buffer);
	}
	
	Loc_Selection = Loc;

	for (i = 0; i < 4; i++)
		gtk_widget_show(label[i]);

	gtk_widget_show(LocName);
	gtk_widget_show(LocLat);
	gtk_widget_show(LocLong);
	gtk_widget_show(LocHeight);

	for (i = 0; i < 3; i++)
		gtk_widget_show(button[i]);

	gtk_widget_show(table);
	gtk_widget_show(frame);
	gtk_widget_show(bbox);
	gtk_widget_show(hbox);
	gtk_widget_show(window);
}

void Delete_Loc_Data(void)
{
	Choose_Location(Delete_Loc);
}

void Delete_Loc(struct Loc_Struct *Loc)
{
	char buffer[40];

	Loc_Selection = Loc;

	sprintf(buffer, "Delete %s ?", Loc_Selection->QTH);

	YN_Box(buffer, Delete_Loc_OK, (void (*)(void))NULL);
}

static void Delete_Loc_OK(void)
{
	Loc_List = g_slist_remove(Loc_List, Loc_Selection);

	g_free(Loc_Selection);

	Write_Loc_Data();
}

static void UserEditCb(GtkWidget *w, gpointer data)
{
	GtkWidget *window = (GtkWidget *)data;
	char Callsign[21];
	double Latitude, Longitude, Height;
	int Time_Interval, List_Interval, Time_Offset, Date_Format;
	char *s;
	
	s = gtk_entry_get_text(GTK_ENTRY(UserCall));
	strcpy(Callsign, s);

	s = gtk_entry_get_text(GTK_ENTRY(UserLat));
	Latitude = atof(s);
	if (!Range_Check(Latitude, -90.0, 90.0)) {
		Warning_Box("Latitude must be between\n+-90 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserLong));
	Longitude = -atof(s);
	if (Longitude < 0.0)
		Longitude += 360.0;
	if (!Range_Check(Longitude, 0.0, 360.0)) {
		Warning_Box("Longitude must be between\n+-360 degrees");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserHeight));
	Height = atof(s);
	if (Height < -10.0) {
		Warning_Box("Height must be greater than -10m");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserTime));
	Time_Offset = atoi(s);
	if (!Range_Check((double)Time_Offset, -12.0, 12.0)) {
		Warning_Box("Time Offset must be between\n+-12 hours");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserTrack));
	Time_Interval = atoi(s);
	if (Time_Interval < 1) {
		Warning_Box("Track Step must be greater than zero");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserList));
	List_Interval = atoi(s);
	if (List_Interval < 1) {
		Warning_Box("List Step must be greater than zero");
		return;
	}

	s = gtk_entry_get_text(GTK_ENTRY(UserDate));
	if (strcmp(s, "DDMMYYYY") == 0) {
		Date_Format = DATE_FORMAT_DDMMYYYY;
	} else if (strcmp(s, "MMDDYYYY") == 0) {
		Date_Format = DATE_FORMAT_MMDDYYYY;
	} else {
		Warning_Box("Invalid Date Format, must be\neither DDMMYYYY or MMDDYYYY");
		return;
	}

	strcpy(User_Data.Callsign, Callsign);
	User_Data.Latitude      = Latitude;
	User_Data.Longitude     = Longitude;
	User_Data.Height        = Height;
	User_Data.Time_Interval = Time_Interval;
	User_Data.List_Interval = List_Interval;
	User_Data.Time_Offset   = Time_Offset;
	User_Data.Date_Format   = Date_Format;
	
	if (Write_User_Data())
		gtk_widget_destroy(window);
}

void Update_User_Data(void)
{
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *notebook;
	GtkWidget *notelabel[2];
	GtkWidget *table[2];
	GtkWidget *label[8];
	GtkWidget *button[3];
	GtkWidget *bbox;
	char buffer[20];
	int i;
	
	window = gtk_window_new(GTK_WINDOW_DIALOG);

	gtk_window_set_title(GTK_WINDOW(window), "Preferences");
	gtk_container_set_border_width(GTK_CONTAINER(window), BORDER_WIDTH);

	vbox = gtk_vbox_new(FALSE, BORDER_WIDTH);
	gtk_container_add(GTK_CONTAINER(window), vbox);

	notebook = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(vbox), notebook, TRUE, TRUE, 0);

	table[0]     = gtk_table_new(4, 2, FALSE);
	notelabel[0] = gtk_label_new("Location");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table[0], notelabel[0]);

	label[0] = gtk_label_new("Callsign");
	label[1] = gtk_label_new("Latitude");
	label[2] = gtk_label_new("Longitude");
	label[3] = gtk_label_new("Height");

	UserCall   = gtk_entry_new_with_max_length(10);
	UserLat    = gtk_entry_new_with_max_length(10);
	UserLong   = gtk_entry_new_with_max_length(10);
	UserHeight = gtk_entry_new_with_max_length(10);

	gtk_table_attach(GTK_TABLE(table[0]), label[0], 0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), label[1], 0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), label[2], 0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), label[3], 0, 1, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	gtk_table_attach(GTK_TABLE(table[0]), UserCall,   1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), UserLat,    1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), UserLong,   1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[0]), UserHeight, 1, 2, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	table[1]     = gtk_table_new(4, 2, FALSE);
	notelabel[1] = gtk_label_new("Time");
	gtk_notebook_append_page(GTK_NOTEBOOK(notebook), table[1], notelabel[1]);

	label[4] = gtk_label_new("Time Offset");
	label[5] = gtk_label_new("Track Step");
	label[6] = gtk_label_new("List Step");
	label[7] = gtk_label_new("Date Format");

	UserTime   = gtk_entry_new_with_max_length(10);
	UserTrack  = gtk_entry_new_with_max_length(10);
	UserList   = gtk_entry_new_with_max_length(10);
	UserDate   = gtk_entry_new_with_max_length(10);

	gtk_table_attach(GTK_TABLE(table[1]), label[4], 0, 1, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[5], 0, 1, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[6], 0, 1, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), label[7], 0, 1, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	gtk_table_attach(GTK_TABLE(table[1]), UserTime,   1, 2, 0, 1,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), UserTrack,  1, 2, 1, 2,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), UserList,   1, 2, 2, 3,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);
	gtk_table_attach(GTK_TABLE(table[1]), UserDate,   1, 2, 3, 4,
		GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND, BORDER_ITEM, BORDER_ITEM);

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	gtk_box_pack_end(GTK_BOX(vbox), bbox, TRUE, TRUE, 0);

	button[0] = gtk_button_new_with_label("Save");
	button[1] = gtk_button_new_with_label("Cancel");
	button[2] = gtk_button_new_with_label("Help");

	gtk_container_add(GTK_CONTAINER(bbox), button[0]);
	gtk_container_add(GTK_CONTAINER(bbox), button[1]);
	gtk_container_add(GTK_CONTAINER(bbox), button[2]);

	gtk_signal_connect(GTK_OBJECT(button[0]), "clicked", GTK_SIGNAL_FUNC(UserEditCb), window);
	gtk_signal_connect_object(GTK_OBJECT(button[1]), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(window));
	gtk_signal_connect(GTK_OBJECT(button[2]), "clicked", GTK_SIGNAL_FUNC(HelpCb), (gpointer)6);

	GTK_WIDGET_SET_FLAGS(button[0], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[1], GTK_CAN_DEFAULT);
	GTK_WIDGET_SET_FLAGS(button[2], GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button[0]);

	gtk_entry_set_text(GTK_ENTRY(UserCall), User_Data.Callsign);

	sprintf(buffer, "%.2f", User_Data.Latitude);
	gtk_entry_set_text(GTK_ENTRY(UserLat), buffer);

	if (User_Data.Longitude > 180.0)
		sprintf(buffer, "%.2f", -(User_Data.Longitude - 360.0));
	else if (User_Data.Longitude < -180.0)
		sprintf(buffer, "%.2f", -(User_Data.Longitude + 360.0));
	else
		sprintf(buffer, "%.2f", -User_Data.Longitude);
	gtk_entry_set_text(GTK_ENTRY(UserLong), buffer);

	sprintf(buffer, "%.0f", User_Data.Height);
	gtk_entry_set_text(GTK_ENTRY(UserHeight), buffer);

	sprintf(buffer, "%d", User_Data.Time_Offset);
	gtk_entry_set_text(GTK_ENTRY(UserTime), buffer);

	sprintf(buffer, "%d", User_Data.Time_Interval);
	gtk_entry_set_text(GTK_ENTRY(UserTrack), buffer);

	sprintf(buffer, "%d", User_Data.List_Interval);
	gtk_entry_set_text(GTK_ENTRY(UserList), buffer);

	switch (User_Data.Date_Format) {
		case DATE_FORMAT_DDMMYYYY:
			gtk_entry_set_text(GTK_ENTRY(UserDate), "DDMMYYYY");
			break;
		case DATE_FORMAT_MMDDYYYY:
			gtk_entry_set_text(GTK_ENTRY(UserDate), "MMDDYYYY");
			break;
		default:
			g_error("mtrack: unknown date format %d\n", User_Data.Date_Format);
			break;
	}

	for (i = 0; i < 8; i++)
		gtk_widget_show(label[i]);

	gtk_widget_show(UserCall);
	gtk_widget_show(UserLat);
	gtk_widget_show(UserLong);
	gtk_widget_show(UserHeight);
	gtk_widget_show(UserTime);
	gtk_widget_show(UserList);
	gtk_widget_show(UserTrack);
	gtk_widget_show(UserDate);

	for (i = 0; i < 3; i++)
		gtk_widget_show(button[i]);

	gtk_widget_show(bbox);
	gtk_widget_show(table[0]);
	gtk_widget_show(table[1]);
	gtk_widget_show(notelabel[0]);
	gtk_widget_show(notelabel[1]);
	gtk_widget_show(notebook);
	gtk_widget_show(vbox);
	gtk_widget_show(window);
}

static int Parse_Line(char *Line, char **argv)
{
	char *s;
	char *t;
	int  argc;

	argc = 0;
	t = Line;

	while ((s = strtok(t, " ")) != NULL) {
		t = NULL;
		argv[argc++] = s;
	}

	return argc;
}

static int Range_Check(double Value, double Bottom, double Top)
{
	if (Value < Bottom || Value > Top)
		return FALSE;
	
	return TRUE;
}
