// Copyright 2020 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     https://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#include "platform_v2/base/exception.h"

#include <vector>

#include "platform_v2/base/exception_test.nc.h"
#include "gmock/gmock.h"
#include "gtest/gtest.h"

namespace location::nearby {

TEST(ExceptionOr, Result_Copy_NonConst) {
  ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Expect a copy when not explicitly moving the result.
  std::vector<int> copy = exception_or_vector.result();
  EXPECT_FALSE(copy.empty());
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Modifying |exception_or_vector| should not affect the copy.
  exception_or_vector.result().clear();
  EXPECT_FALSE(copy.empty());
}

TEST(ExceptionOr, Result_Copy_Const) {
  const ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Expect a copy when not explicitly moving the result.
  std::vector<int> copy = exception_or_vector.result();
  EXPECT_FALSE(copy.empty());
  EXPECT_FALSE(exception_or_vector.result().empty());
}

TEST(ExceptionOr, Result_Reference_NonConst) {
  ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Getting a reference should not modify the source.
  std::vector<int>& reference = exception_or_vector.result();
  EXPECT_FALSE(reference.empty());
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Modifying |exception_or_vector| should reflect in the reference.
  exception_or_vector.result().clear();
  EXPECT_TRUE(reference.empty());
}

TEST(ExceptionOr, Result_Reference_Const) {
  const ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Getting a reference should not modify the source.
  const std::vector<int>& reference = exception_or_vector.result();
  EXPECT_FALSE(reference.empty());
  EXPECT_FALSE(exception_or_vector.result().empty());
}

TEST(ExceptionOr, Result_Move_NonConst) {
  ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Moving the result should clear the source.
  std::vector<int> moved = std::move(exception_or_vector).result();
  EXPECT_FALSE(moved.empty());
}

TEST(ExceptionOr, Result_Move_Const) {
  const ExceptionOr<std::vector<int>> exception_or_vector({1, 2, 3});
  EXPECT_FALSE(exception_or_vector.result().empty());

  // Moving const rvalue reference will result in a copy.
  std::vector<int> moved = std::move(exception_or_vector).result();
  EXPECT_FALSE(moved.empty());
}

TEST(ExceptionOr, ExplicitConversionWorks) {
  class A {
   public:
    A() = default;
  };
  class B {
   public:
    B() = default;
    explicit B(A) {}
  };
  ExceptionOr<A> a(A{});
  ExceptionOr<B> b(a);
  EXPECT_TRUE(a.ok());
  EXPECT_TRUE(b.ok());
}

TEST(ExceptionOr, ExplicitConversionFailsToCompile) {
  class A {
   public:
    A() = default;
  };
  class B {
   public:
    B() = default;
  };
  ExceptionOr<A> a(A{});
  EXPECT_NON_COMPILE("no matching constructor", { ExceptionOr<B> b(a); });
}

}  // namespace location::nearby
