"""faker.providers.person.fr_BE - 8-9 Jan 2023.

Last names and male and female first names for locale 'fr_BE' (French-speaking Belgium).
Source: Statbel (Directorate-general Statistics - Statistics Belgium), https://statbel.fgov.be/en/about-statbel, 2022.
"""
from collections import Counter, OrderedDict

from .. import Provider as PersonProvider


class Provider(PersonProvider):
    """Provider of `Person` data (male and female first and last names) for locale 'fr_BE' (Belgium French, Walloon)."""

    # As per https://nl.wikipedia.org/wiki/Familienaam#Belgi%C3%AB, from 1 Jun 2014 a child can get
    # the family name of either parent, or both parents' family names separated by a space.
    # As per https://nl.wikipedia.org/wiki/Geboortenaam#Situatie_in_Belgi%C3%AB, marriage does NOT affect family names.
    #
    # Below reflects these with an assumed 10% doubled name (for both genders).
    formats_male = OrderedDict(
        (("{{first_name_male}} {{last_name}}", 90), ("{{first_name_male}} {{last_name}} {{last_name}}", 10))
    )
    formats_female = OrderedDict(
        (("{{first_name_female}} {{last_name}}", 90), ("{{first_name_female}} {{last_name}} {{last_name}}", 10))
    )
    formats = OrderedDict(Counter(formats_male) + Counter(formats_female))

    # 500 most common male first names in Wallonia in 2022 as per StatBel
    # note this (simplistically) models 'fr_BE' as identical to Wallonia (and ignores Brussels)
    #
    # Data sourced from sheet 'Men' columns H-I ('Wallonia') rows 3..502 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.2%20Voornamen%20van%20vrouwen%20en%20mannen/First%20names_Population_2022.xlsx
    # and then sorted by name
    first_names_male = OrderedDict(
        (
            ("Aaron", 1922),
            ("Abdelkader", 616),
            ("Abdellah", 461),
            ("Achille", 1414),
            ("Adam", 2606),
            ("Adem", 467),
            ("Adil", 420),
            ("Adrian", 605),
            ("Adriano", 858),
            ("Adrien", 6155),
            ("Ahmed", 2136),
            ("Ahmet", 590),
            ("Alain", 19185),
            ("Alan", 943),
            ("Albert", 4465),
            ("Aldo", 446),
            ("Alessandro", 1919),
            ("Alessio", 2295),
            ("Alex", 1955),
            ("Alexander", 682),
            ("Alexandre", 10947),
            ("Alexis", 4144),
            ("Alfonso", 462),
            ("Alfred", 1239),
            ("Ali", 2331),
            ("Allan", 676),
            ("Alphonse", 769),
            ("Amaury", 1760),
            ("Amine", 731),
            ("Amir", 708),
            ("Anas", 580),
            ("Andrea", 1254),
            ("Andreas", 513),
            ("Andrew", 702),
            ("André", 14002),
            ("Andy", 1646),
            ("Angelo", 2519),
            ("Anthony", 6441),
            ("Antoine", 9416),
            ("Antonin", 856),
            ("Antonino", 1203),
            ("Antonio", 4203),
            ("Antony", 516),
            ("Armand", 1205),
            ("Arnaud", 7665),
            ("Arno", 442),
            ("Arthur", 6664),
            ("Auguste", 581),
            ("Augustin", 1679),
            ("Aurélien", 2779),
            ("Axel", 3429),
            ("Aymeric", 803),
            ("Ayoub", 727),
            ("Baptiste", 3228),
            ("Basile", 1467),
            ("Bastien", 2791),
            ("Baudouin", 798),
            ("Benjamin", 7872),
            ("Benoit", 4522),
            ("Benoît", 5283),
            ("Bernard", 10951),
            ("Bertrand", 2061),
            ("Bilal", 794),
            ("Boris", 1012),
            ("Brahim", 579),
            ("Brandon", 1183),
            ("Brian", 470),
            ("Brice", 650),
            ("Brieuc", 504),
            ("Bruno", 6673),
            ("Bryan", 2696),
            ("Calogero", 1580),
            ("Camille", 857),
            ("Carl", 792),
            ("Carlo", 672),
            ("Carlos", 687),
            ("Carmelo", 1179),
            ("Cedric", 702),
            ("Charles", 5063),
            ("Charlie", 801),
            ("Charly", 1027),
            ("Christian", 16417),
            ("Christophe", 15728),
            ("Christopher", 2673),
            ("Claude", 9224),
            ("Claudio", 1005),
            ("Claudy", 802),
            ("Clément", 4170),
            ("Colin", 795),
            ("Constantin", 523),
            ("Corentin", 3187),
            ("Cyril", 2617),
            ("Cyrille", 473),
            ("Cédric", 8739),
            ("Célestin", 499),
            ("Côme", 475),
            ("Damien", 5442),
            ("Daniel", 15507),
            ("Danny", 845),
            ("Dany", 1893),
            ("Dario", 805),
            ("David", 15198),
            ("Denis", 4726),
            ("Didier", 10307),
            ("Diego", 3121),
            ("Dimitri", 3493),
            ("Dirk", 531),
            ("Domenico", 1149),
            ("Dominique", 7222),
            ("Donovan", 1092),
            ("Dorian", 2208),
            ("Dylan", 3990),
            ("Eddy", 3176),
            ("Eden", 1395),
            ("Edgard", 528),
            ("Edmond", 793),
            ("Edouard", 2111),
            ("El", 919),
            ("Elias", 840),
            ("Elie", 603),
            ("Eliot", 621),
            ("Eliott", 1280),
            ("Elliot", 453),
            ("Emile", 3218),
            ("Emilien", 1348),
            ("Emilio", 433),
            ("Emmanuel", 4817),
            ("Enzo", 2545),
            ("Eric", 14938),
            ("Erik", 445),
            ("Ernest", 606),
            ("Erwan", 492),
            ("Erwin", 563),
            ("Esteban", 1476),
            ("Ethan", 3769),
            ("Etienne", 3779),
            ("Eugène", 620),
            ("Evan", 912),
            ("Ezio", 538),
            ("Fabian", 3892),
            ("Fabien", 1986),
            ("Fabio", 1078),
            ("Fabrice", 6668),
            ("Fabrizio", 818),
            ("Farid", 424),
            ("Fernand", 2023),
            ("Fernando", 444),
            ("Filippo", 762),
            ("Flavio", 900),
            ("Florent", 2512),
            ("Florian", 4813),
            ("Francesco", 2848),
            ("Francis", 7850),
            ("Francisco", 638),
            ("Franck", 1287),
            ("Franco", 1059),
            ("Frank", 1371),
            ("Franz", 925),
            ("François", 10479),
            ("François-Xavier", 450),
            ("Freddy", 3366),
            ("Frédéric", 14920),
            ("Félix", 1221),
            ("Gabin", 527),
            ("Gabriel", 5711),
            ("Gaetano", 986),
            ("Gary", 559),
            ("Gaspard", 1260),
            ("Gaston", 1055),
            ("Gauthier", 2049),
            ("Gaël", 964),
            ("Gaëtan", 2955),
            ("Geoffrey", 4406),
            ("Geoffroy", 678),
            ("Georges", 6122),
            ("Germain", 598),
            ("Ghislain", 1183),
            ("Giacomo", 512),
            ("Gianni", 927),
            ("Gilbert", 2359),
            ("Gilles", 3374),
            ("Gino", 562),
            ("Giovanni", 3174),
            ("Giuliano", 656),
            ("Giuseppe", 5859),
            ("Gregory", 1739),
            ("Grégoire", 1059),
            ("Grégory", 6539),
            ("Guido", 745),
            ("Guillaume", 7306),
            ("Gustave", 541),
            ("Guy", 9812),
            ("Gérald", 801),
            ("Gérard", 4072),
            ("Hadrien", 495),
            ("Hamza", 937),
            ("Harry", 444),
            ("Hasan", 579),
            ("Hassan", 853),
            ("Hector", 425),
            ("Henri", 5349),
            ("Henry", 576),
            ("Hervé", 1785),
            ("Hicham", 590),
            ("Hubert", 1971),
            ("Hugo", 7597),
            ("Hugues", 1463),
            ("Ibrahim", 1262),
            ("Igor", 553),
            ("Ilias", 458),
            ("Ilyas", 596),
            ("Isaac", 620),
            ("Ismail", 567),
            ("Ismaël", 557),
            ("Ivan", 822),
            ("Jacky", 693),
            ("Jacques", 13781),
            ("Jamal", 419),
            ("James", 801),
            ("Jan", 868),
            ("Jason", 2454),
            ("Jean", 34246),
            ("Jean-Baptiste", 587),
            ("Jean-Charles", 460),
            ("Jean-Christophe", 1144),
            ("Jean-Claude", 4959),
            ("Jean-François", 5180),
            ("Jean-Jacques", 1282),
            ("Jean-Louis", 2300),
            ("Jean-Luc", 4387),
            ("Jean-Marc", 3793),
            ("Jean-Marie", 5615),
            ("Jean-Michel", 2395),
            ("Jean-Paul", 2455),
            ("Jean-Philippe", 1880),
            ("Jean-Pierre", 6407),
            ("Jean-Pol", 1068),
            ("Jean-Yves", 1025),
            ("Jeremy", 1283),
            ("Jerôme", 588),
            ("Jessy", 503),
            ("Jimmy", 1661),
            ("Joachim", 1336),
            ("Joffrey", 525),
            ("Johan", 2594),
            ("Johann", 634),
            ("John", 1860),
            ("Johnny", 1580),
            ("Jonas", 945),
            ("Jonathan", 9513),
            ("Jordan", 3695),
            ("Joris", 505),
            ("Joseph", 6964),
            ("Joshua", 832),
            ("José", 5468),
            ("Joël", 3696),
            ("Juan", 999),
            ("Jules", 4826),
            ("Julian", 1320),
            ("Julien", 13669),
            ("Justin", 1953),
            ("Jérémie", 988),
            ("Jérémy", 4478),
            ("Jérôme", 6403),
            ("Karim", 1216),
            ("Karl", 1013),
            ("Kenny", 599),
            ("Kenzo", 502),
            ("Kevin", 7018),
            ("Khalid", 678),
            ("Killian", 569),
            ("Kylian", 1312),
            ("Kyllian", 460),
            ("Kévin", 2224),
            ("Laurent", 12306),
            ("Leandro", 675),
            ("Lenny", 932),
            ("Leo", 470),
            ("Leonardo", 519),
            ("Liam", 2521),
            ("Lionel", 3249),
            ("Livio", 842),
            ("Loan", 660),
            ("Logan", 2745),
            ("Lorenzo", 2291),
            ("Loris", 1519),
            ("Louis", 10770),
            ("Louka", 693),
            ("Loïc", 5339),
            ("Loïs", 503),
            ("Luc", 7990),
            ("Luca", 3543),
            ("Lucas", 7721),
            ("Luciano", 742),
            ("Lucien", 2714),
            ("Ludovic", 5288),
            ("Luigi", 1664),
            ("Luis", 831),
            ("Luka", 749),
            ("Lukas", 794),
            ("Lyam", 644),
            ("Léo", 2403),
            ("Léon", 3478),
            ("Léonard", 417),
            ("Léopold", 1224),
            ("Malik", 429),
            ("Mamadou", 534),
            ("Manuel", 2223),
            ("Marc", 15879),
            ("Marcel", 6458),
            ("Marco", 1636),
            ("Mario", 2384),
            ("Marius", 1551),
            ("Martial", 500),
            ("Martin", 5618),
            ("Marvin", 931),
            ("Massimo", 1121),
            ("Mathias", 2595),
            ("Mathieu", 5517),
            ("Mathis", 2650),
            ("Mathys", 2054),
            ("Mathéo", 3009),
            ("Matteo", 2172),
            ("Matthew", 564),
            ("Matthias", 805),
            ("Matthieu", 1560),
            ("Mattia", 451),
            ("Mattéo", 1416),
            ("Matéo", 580),
            ("Maurice", 3021),
            ("Maurizio", 472),
            ("Max", 769),
            ("Maxence", 2814),
            ("Maxim", 846),
            ("Maxime", 12119),
            ("Maximilien", 1125),
            ("Maël", 1768),
            ("Mehdi", 1848),
            ("Mehmet", 980),
            ("Michael", 2227),
            ("Michaël", 11573),
            ("Michel", 27759),
            ("Michele", 1275),
            ("Mickaël", 1383),
            ("Miguel", 1427),
            ("Mike", 1379),
            ("Milan", 524),
            ("Milo", 887),
            ("Mirko", 427),
            ("Mohamed", 6607),
            ("Mohammad", 535),
            ("Mohammed", 2726),
            ("Morgan", 744),
            ("Mustafa", 1012),
            ("Mustapha", 714),
            ("Nabil", 493),
            ("Nassim", 559),
            ("Nathan", 7980),
            ("Nathanaël", 747),
            ("Naël", 763),
            ("Naïm", 428),
            ("Nicola", 1044),
            ("Nicolas", 19140),
            ("Noa", 2223),
            ("Noah", 6029),
            ("Noam", 932),
            ("Nolan", 1956),
            ("Norbert", 611),
            ("Noé", 1992),
            ("Noël", 1119),
            ("Olivier", 16168),
            ("Omar", 895),
            ("Oscar", 1897),
            ("Owen", 827),
            ("Pablo", 642),
            ("Paolo", 1064),
            ("Pascal", 12529),
            ("Pasquale", 583),
            ("Patrice", 3068),
            ("Patrick", 13876),
            ("Paul", 6243),
            ("Pedro", 483),
            ("Peter", 1095),
            ("Philippe", 26431),
            ("Pierre", 19716),
            ("Pierre-Yves", 691),
            ("Pietro", 1273),
            ("Pol", 1604),
            ("Quentin", 7318),
            ("Rachid", 1081),
            ("Rafael", 755),
            ("Rafaël", 457),
            ("Raoul", 641),
            ("Raphael", 512),
            ("Raphaël", 7174),
            ("Rayan", 1389),
            ("Raymond", 3293),
            ("Remy", 624),
            ("Renato", 413),
            ("Renaud", 2589),
            ("René", 6163),
            ("Ricardo", 455),
            ("Richard", 2617),
            ("Robert", 7389),
            ("Roberto", 1129),
            ("Robin", 3999),
            ("Rodolphe", 435),
            ("Rodrigue", 427),
            ("Roger", 6103),
            ("Roland", 3544),
            ("Romain", 7493),
            ("Roman", 542),
            ("Romuald", 684),
            ("Roméo", 588),
            ("Ronald", 718),
            ("Ronny", 416),
            ("Rosario", 746),
            ("Ruben", 524),
            ("Rudi", 1063),
            ("Rudy", 2707),
            ("Ryan", 1187),
            ("Régis", 1032),
            ("Rémi", 1014),
            ("Rémy", 1477),
            ("Sacha", 3962),
            ("Said", 587),
            ("Salvatore", 4515),
            ("Sam", 1096),
            ("Sami", 743),
            ("Samir", 724),
            ("Samuel", 5213),
            ("Samuël", 637),
            ("Samy", 591),
            ("Sandro", 786),
            ("Sasha", 882),
            ("Saïd", 535),
            ("Sean", 494),
            ("Sebastien", 1131),
            ("Serge", 7494),
            ("Sergio", 803),
            ("Simon", 6624),
            ("Sofiane", 609),
            ("Stefan", 654),
            ("Stefano", 688),
            ("Stephan", 745),
            ("Stephane", 468),
            ("Steve", 3336),
            ("Steven", 1583),
            ("Stéphan", 823),
            ("Stéphane", 9458),
            ("Sylvain", 2342),
            ("Sébastien", 12494),
            ("Tanguy", 1129),
            ("Thiago", 537),
            ("Thibaud", 524),
            ("Thibault", 2665),
            ("Thibaut", 2628),
            ("Thierry", 11453),
            ("Thomas", 13916),
            ("Théo", 6139),
            ("Théodore", 416),
            ("Tiago", 1423),
            ("Tim", 1028),
            ("Timothy", 612),
            ("Timothée", 648),
            ("Timéo", 1989),
            ("Tiziano", 508),
            ("Tom", 5839),
            ("Tommy", 444),
            ("Tony", 1200),
            ("Tristan", 1909),
            ("Téo", 760),
            ("Ugo", 1133),
            ("Valentin", 4223),
            ("Valentino", 631),
            ("Victor", 4988),
            ("Vincent", 12875),
            ("Vincenzo", 2410),
            ("Vito", 450),
            ("Walid", 546),
            ("Walter", 957),
            ("William", 4072),
            ("Willy", 3376),
            ("Xavier", 6475),
            ("Yanis", 2199),
            ("Yann", 717),
            ("Yannick", 2075),
            ("Yannis", 488),
            ("Yassin", 487),
            ("Yassine", 1035),
            ("Yoann", 450),
            ("Yohan", 735),
            ("Younes", 792),
            ("Youri", 515),
            ("Youssef", 1375),
            ("Yusuf", 487),
            ("Yvan", 1552),
            ("Yves", 7626),
            ("Yvon", 1293),
            ("Zakaria", 915),
        )
    )

    # 500 most common female first names in Wallonia in 2022 as per StatBel (with the same simplification as above)
    #
    # Data sourced from sheet 'Women' columns H-I ('Wallonia') rows 3..502 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.2%20Voornamen%20van%20vrouwen%20en%20mannen/First%20names_Population_2022.xlsx
    # and then sorted by name
    first_names_female = OrderedDict(
        (
            ("Adeline", 1996),
            ("Adriana", 664),
            ("Adèle", 1294),
            ("Agathe", 625),
            ("Agnès", 2980),
            ("Alberte", 806),
            ("Alessia", 1206),
            ("Alexandra", 3873),
            ("Alexia", 1851),
            ("Alice", 6134),
            ("Alicia", 3281),
            ("Aline", 3827),
            ("Alison", 1567),
            ("Alisson", 799),
            ("Alix", 1766),
            ("Alizée", 764),
            ("Alyssa", 871),
            ("Amandine", 4726),
            ("Ambre", 1509),
            ("Amina", 1036),
            ("Amélia", 804),
            ("Amélie", 4034),
            ("Ana", 1303),
            ("Anastasia", 563),
            ("Anaëlle", 985),
            ("Anaïs", 5221),
            ("Andrea", 599),
            ("Andréa", 533),
            ("Andrée", 4461),
            ("Angela", 2190),
            ("Angelina", 921),
            ("Angèle", 749),
            ("Angélique", 4440),
            ("Anissa", 903),
            ("Anita", 1960),
            ("Ann", 612),
            ("Anna", 6389),
            ("Annabelle", 521),
            ("Anne", 18637),
            ("Anne-Catherine", 804),
            ("Anne-Françoise", 760),
            ("Anne-Marie", 5244),
            ("Anne-Sophie", 2074),
            ("Annette", 1688),
            ("Annick", 4519),
            ("Annie", 6979),
            ("Anny", 770),
            ("Antoinette", 635),
            ("Antonella", 551),
            ("Antonia", 852),
            ("Antonietta", 736),
            ("Antonina", 579),
            ("Apolline", 677),
            ("Ariane", 1591),
            ("Arlette", 4082),
            ("Assia", 702),
            ("Astrid", 1434),
            ("Aude", 1332),
            ("Audrey", 5495),
            ("Aurore", 4784),
            ("Aurélie", 8447),
            ("Axelle", 1363),
            ("Aya", 894),
            ("Aïcha", 777),
            ("Barbara", 2302),
            ("Bernadette", 7995),
            ("Berthe", 761),
            ("Betty", 716),
            ("Brigitte", 8729),
            ("Béatrice", 5819),
            ("Bénédicte", 2971),
            ("Bérengère", 538),
            ("Bérénice", 537),
            ("Camille", 7894),
            ("Capucine", 781),
            ("Carine", 8290),
            ("Carla", 975),
            ("Carmela", 1442),
            ("Carmen", 1171),
            ("Carole", 2773),
            ("Caroline", 9799),
            ("Cassandra", 1510),
            ("Caterina", 571),
            ("Catherine", 12270),
            ("Cathy", 2670),
            ("Chantal", 12383),
            ("Charlie", 560),
            ("Charline", 2336),
            ("Charlotte", 7281),
            ("Chiara", 1372),
            ("Chloé", 6538),
            ("Christel", 2878),
            ("Christelle", 7450),
            ("Christiane", 10377),
            ("Christianne", 587),
            ("Christina", 944),
            ("Christine", 14924),
            ("Cindy", 5453),
            ("Claire", 5014),
            ("Clara", 5701),
            ("Clarisse", 950),
            ("Claude", 845),
            ("Claudette", 1016),
            ("Claudia", 1540),
            ("Claudine", 8230),
            ("Cloé", 756),
            ("Clémence", 2553),
            ("Clémentine", 1438),
            ("Colette", 4420),
            ("Coline", 1171),
            ("Concetta", 958),
            ("Constance", 710),
            ("Coralie", 2770),
            ("Coraline", 704),
            ("Corine", 877),
            ("Corinne", 3399),
            ("Cristina", 654),
            ("Cynthia", 1434),
            ("Cécile", 6406),
            ("Célestine", 683),
            ("Célia", 3355),
            ("Céline", 9865),
            ("Daisy", 762),
            ("Daniela", 792),
            ("Danielle", 6363),
            ("Danièle", 1362),
            ("Daphné", 836),
            ("Deborah", 1443),
            ("Delphine", 5035),
            ("Denise", 3330),
            ("Diana", 1049),
            ("Diane", 799),
            ("Dina", 591),
            ("Dominique", 9714),
            ("Dorothée", 1009),
            ("Déborah", 1059),
            ("Edith", 1742),
            ("Elena", 2748),
            ("Eliane", 2556),
            ("Elina", 602),
            ("Eline", 1319),
            ("Elisa", 3679),
            ("Elisabeth", 4183),
            ("Elise", 5004),
            ("Elodie", 5667),
            ("Eloïse", 2015),
            ("Elsa", 1769),
            ("Eléa", 856),
            ("Eléna", 777),
            ("Eléonore", 1571),
            ("Emeline", 2078),
            ("Emelyne", 537),
            ("Emilia", 808),
            ("Emilie", 8324),
            ("Emma", 6227),
            ("Emmanuelle", 2171),
            ("Emy", 1101),
            ("Erika", 1027),
            ("Estelle", 2654),
            ("Esther", 997),
            ("Eugénie", 517),
            ("Eva", 4415),
            ("Eve", 690),
            ("Eveline", 1080),
            ("Evelyne", 2949),
            ("Fabienne", 9206),
            ("Fanny", 3891),
            ("Fatiha", 725),
            ("Fatima", 3382),
            ("Fatma", 969),
            ("Fernande", 1077),
            ("Fiona", 1216),
            ("Flavie", 722),
            ("Flora", 637),
            ("Flore", 721),
            ("Florence", 5651),
            ("Floriane", 577),
            ("Florine", 1127),
            ("Franca", 517),
            ("France", 1363),
            ("Francesca", 1413),
            ("Francine", 6224),
            ("Françoise", 15074),
            ("Frédérique", 1548),
            ("Gabrielle", 2245),
            ("Gaëlle", 2586),
            ("Geneviève", 5125),
            ("Georgette", 2987),
            ("Germaine", 1269),
            ("Ghislaine", 2128),
            ("Gilberte", 1685),
            ("Ginette", 2049),
            ("Giovanna", 1389),
            ("Gisèle", 2577),
            ("Giulia", 2225),
            ("Giuseppa", 1139),
            ("Giuseppina", 1526),
            ("Gladys", 517),
            ("Gloria", 669),
            ("Grazia", 537),
            ("Graziella", 844),
            ("Gwenaëlle", 809),
            ("Gwendoline", 1155),
            ("Géraldine", 2160),
            ("Helena", 621),
            ("Henriette", 952),
            ("Huguette", 1721),
            ("Héloïse", 1042),
            ("Hélène", 5090),
            ("Ilona", 840),
            ("Imane", 722),
            ("Inaya", 848),
            ("Ines", 1063),
            ("Ingrid", 3140),
            ("Inès", 2893),
            ("Irina", 641),
            ("Iris", 918),
            ("Irène", 2103),
            ("Isabelle", 18724),
            ("Isaline", 623),
            ("Jacqueline", 12361),
            ("Jade", 3076),
            ("Jamila", 536),
            ("Janine", 898),
            ("Jeanine", 2047),
            ("Jeanne", 5843),
            ("Jeannine", 7623),
            ("Jennifer", 4762),
            ("Jenny", 1488),
            ("Jessica", 6083),
            ("Joanna", 556),
            ("Jocelyne", 3275),
            ("Johanna", 1072),
            ("Josette", 3632),
            ("Josiane", 5824),
            ("Josianne", 635),
            ("Josée", 1676),
            ("Joséphine", 1827),
            ("Joëlle", 5374),
            ("Judith", 934),
            ("Julia", 3002),
            ("Julie", 11779),
            ("Juliette", 4813),
            ("Justine", 4599),
            ("Karima", 628),
            ("Karin", 1183),
            ("Karine", 2031),
            ("Kathleen", 815),
            ("Kathy", 920),
            ("Katia", 1209),
            ("Katty", 1595),
            ("Kelly", 1879),
            ("Kenza", 877),
            ("Khadija", 1269),
            ("Kimberley", 1197),
            ("Laetitia", 5479),
            ("Laly", 838),
            ("Laora", 560),
            ("Lara", 2110),
            ("Laura", 10919),
            ("Laure", 2241),
            ("Laurence", 10175),
            ("Laurie", 1961),
            ("Laurine", 892),
            ("Laëtitia", 526),
            ("Lea", 568),
            ("Leila", 771),
            ("Lena", 1193),
            ("Leslie", 1005),
            ("Leïla", 654),
            ("Lila", 648),
            ("Liliane", 6088),
            ("Lilou", 1593),
            ("Lily", 1275),
            ("Lina", 2597),
            ("Linda", 2499),
            ("Lindsay", 1778),
            ("Line", 777),
            ("Lisa", 3602),
            ("Lise", 1723),
            ("Lisette", 560),
            ("Livia", 814),
            ("Lola", 3350),
            ("Loredana", 990),
            ("Lorena", 752),
            ("Lou", 1446),
            ("Louane", 604),
            ("Louisa", 914),
            ("Louise", 7194),
            ("Louna", 1000),
            ("Lucette", 833),
            ("Lucia", 1484),
            ("Lucie", 7397),
            ("Lucienne", 1948),
            ("Lucile", 686),
            ("Lucy", 890),
            ("Ludivine", 1817),
            ("Luna", 1713),
            ("Lydia", 1374),
            ("Lydie", 1040),
            ("Léa", 8281),
            ("Léana", 657),
            ("Léna", 1948),
            ("Léonie", 1049),
            ("Madeleine", 3116),
            ("Madison", 899),
            ("Maeva", 668),
            ("Magali", 3148),
            ("Malika", 1266),
            ("Mandy", 647),
            ("Manon", 8633),
            ("Manuela", 825),
            ("Marcelle", 2535),
            ("Margaux", 3346),
            ("Margot", 1392),
            ("Marguerite", 2981),
            ("Maria", 17999),
            ("Mariam", 550),
            ("Marianne", 4787),
            ("Marie", 41370),
            ("Marie-Ange", 1183),
            ("Marie-Anne", 1153),
            ("Marie-Christine", 3831),
            ("Marie-Claire", 2731),
            ("Marie-Claude", 782),
            ("Marie-France", 2406),
            ("Marie-Françoise", 795),
            ("Marie-Hélène", 950),
            ("Marie-Jeanne", 1786),
            ("Marie-José", 1114),
            ("Marie-Laure", 724),
            ("Marie-Line", 786),
            ("Marie-Louise", 1895),
            ("Marie-Madeleine", 546),
            ("Marie-Noëlle", 724),
            ("Marie-Paule", 1888),
            ("Marie-Pierre", 753),
            ("Marie-Rose", 1553),
            ("Marie-Thérèse", 2860),
            ("Marielle", 608),
            ("Mariette", 1580),
            ("Marina", 2257),
            ("Marine", 4568),
            ("Marion", 2755),
            ("Marjorie", 2057),
            ("Marlène", 672),
            ("Marthe", 1048),
            ("Martine", 14396),
            ("Mary", 642),
            ("Maryline", 2068),
            ("Marylène", 913),
            ("Maryse", 2299),
            ("Mathilde", 2814),
            ("Maud", 1898),
            ("Maude", 739),
            ("Maureen", 1135),
            ("Mauricette", 633),
            ("Maurine", 577),
            ("Maya", 734),
            ("Maëlle", 1768),
            ("Maëlys", 890),
            ("Maëva", 905),
            ("Maïté", 1545),
            ("Melissa", 1922),
            ("Mia", 1911),
            ("Micheline", 4151),
            ("Michelle", 3334),
            ("Michèle", 5819),
            ("Mila", 1700),
            ("Mireille", 5018),
            ("Monika", 595),
            ("Monique", 14460),
            ("Morgane", 3021),
            ("Muriel", 4542),
            ("Murielle", 2550),
            ("Mya", 699),
            ("Myriam", 6727),
            ("Mégane", 1148),
            ("Mélanie", 5610),
            ("Mélina", 760),
            ("Mélissa", 3038),
            ("Mélodie", 563),
            ("Nadia", 3929),
            ("Nadine", 8525),
            ("Nadège", 1314),
            ("Nancy", 4071),
            ("Naomi", 828),
            ("Natacha", 2113),
            ("Nathalie", 20537),
            ("Nell", 727),
            ("Nelly", 2947),
            ("Nicole", 12422),
            ("Nina", 2274),
            ("Nora", 2030),
            ("Norah", 634),
            ("Nour", 595),
            ("Noémie", 3730),
            ("Noëlla", 681),
            ("Noëlle", 544),
            ("Océane", 3687),
            ("Odette", 2075),
            ("Odile", 958),
            ("Olga", 894),
            ("Olivia", 4201),
            ("Ophélie", 1764),
            ("Ornella", 956),
            ("Paola", 784),
            ("Pascale", 6004),
            ("Pascaline", 574),
            ("Patricia", 11883),
            ("Patrizia", 575),
            ("Paula", 1199),
            ("Paule", 637),
            ("Paulette", 2759),
            ("Pauline", 6347),
            ("Perrine", 1090),
            ("Priscilla", 754),
            ("Rachel", 2193),
            ("Rachida", 553),
            ("Rania", 582),
            ("Raymonde", 1870),
            ("Rebecca", 872),
            ("Renée", 3228),
            ("Rita", 3593),
            ("Rolande", 639),
            ("Romane", 2596),
            ("Romy", 533),
            ("Rosa", 2546),
            ("Rosalia", 656),
            ("Rosalie", 812),
            ("Rosaria", 618),
            ("Rose", 3145),
            ("Rose-Marie", 1135),
            ("Roseline", 530),
            ("Roxane", 1280),
            ("Régine", 1824),
            ("Sabine", 6458),
            ("Sabrina", 5356),
            ("Salma", 576),
            ("Salomé", 826),
            ("Samantha", 847),
            ("Samira", 967),
            ("Sandra", 4896),
            ("Sandrine", 5810),
            ("Sandy", 963),
            ("Sara", 2684),
            ("Sarah", 10424),
            ("Serena", 547),
            ("Shana", 521),
            ("Shirley", 521),
            ("Silvana", 685),
            ("Silvia", 688),
            ("Simone", 2189),
            ("Simonne", 845),
            ("Sofia", 2471),
            ("Solange", 1585),
            ("Soline", 588),
            ("Solène", 663),
            ("Sonia", 3555),
            ("Sophia", 808),
            ("Sophie", 12241),
            ("Soraya", 629),
            ("Stephanie", 630),
            ("Stéphanie", 11825),
            ("Suzanne", 3805),
            ("Sylvia", 1405),
            ("Sylviane", 1678),
            ("Sylvianne", 553),
            ("Sylvie", 8635),
            ("Séverine", 3219),
            ("Tamara", 956),
            ("Tania", 842),
            ("Tatiana", 1193),
            ("Teresa", 884),
            ("Thi", 755),
            ("Théa", 832),
            ("Thérèse", 2834),
            ("Tiffany", 1391),
            ("Tiziana", 628),
            ("Valentina", 887),
            ("Valentine", 2976),
            ("Valérie", 12193),
            ("Vanessa", 5847),
            ("Victoire", 541),
            ("Victoria", 3961),
            ("Vincenza", 762),
            ("Vinciane", 2062),
            ("Violette", 882),
            ("Virginie", 8533),
            ("Viviane", 4710),
            ("Véronique", 14214),
            ("Wendy", 1885),
            ("Yasmina", 916),
            ("Yasmine", 1405),
            ("Yolande", 1370),
            ("Ysaline", 738),
            ("Yvette", 4769),
            ("Yvonne", 2432),
            ("Zoé", 4697),
            ("Zélie", 969),
        )
    )

    # combination of *almost* 1000 common first names (not necesarilly *the most* common)
    # with totalled frequencies (for duplicate 'unisex'/gender-neutral first names, which cause < 1000 total names)
    first_names = OrderedDict(Counter(first_names_male) + Counter(first_names_female))

    # 500 most common last names in Wallonia in 2022 as per StatBel (with the same simplification as above)
    #
    # Data sourced from sheet 'top10000_2022' columns H-I ('Walloon Region') rows 2..501 (top-500 names incl. frequency)
    # of Excel workbook at https://statbel.fgov.be/sites/default/files/files/documents/bevolking/
    # 5.10%20Namen%20en%20voornamen/5.10.1%20Familienamen/Family_names_2022.xlsx
    # and then sorted by name
    last_names = OrderedDict(
        (
            ("Abrassart", 723),
            ("Adam", 3488),
            ("Aerts", 784),
            ("Albert", 1241),
            ("Alexandre", 1249),
            ("Allard", 1609),
            ("Anciaux", 1011),
            ("André", 3474),
            ("Antoine", 2984),
            ("Arnould", 1822),
            ("Bah", 816),
            ("Bailly", 1125),
            ("Barbier", 1361),
            ("Barbieux", 592),
            ("Barry", 817),
            ("Bastien", 659),
            ("Bastin", 3310),
            ("Baudoux", 910),
            ("Bauduin", 1097),
            ("Bauwens", 1013),
            ("Beaujean", 758),
            ("Becker", 1250),
            ("Beckers", 1410),
            ("Beguin", 633),
            ("Benoit", 795),
            ("Berger", 1561),
            ("Bernard", 3890),
            ("Bertrand", 4339),
            ("Blaise", 984),
            ("Blondiau", 722),
            ("Bodart", 2436),
            ("Bodson", 1835),
            ("Bogaert", 722),
            ("Boland", 621),
            ("Bonnet", 633),
            ("Bouchat", 1112),
            ("Boucher", 604),
            ("Bouchez", 811),
            ("Boudart", 777),
            ("Bouillon", 987),
            ("Boulanger", 2227),
            ("Bourgeois", 1534),
            ("Bourguignon", 1553),
            ("Bovy", 1147),
            ("Brasseur", 1970),
            ("Brichard", 656),
            ("Bruno", 769),
            ("Bruyère", 808),
            ("Bultot", 700),
            ("Burton", 1097),
            ("Bury", 837),
            ("Callens", 736),
            ("Cambier", 1614),
            ("Capelle", 618),
            ("Cardon", 645),
            ("Carlier", 3185),
            ("Carpentier", 1254),
            ("Carton", 595),
            ("Cassart", 612),
            ("Caudron", 825),
            ("Chapelle", 971),
            ("Charles", 1572),
            ("Charlier", 4656),
            ("Chevalier", 1558),
            ("Christiaens", 867),
            ("Claes", 1854),
            ("Claessens", 740),
            ("Claeys", 718),
            ("Claude", 642),
            ("Claus", 878),
            ("Close", 765),
            ("Closset", 904),
            ("Clément", 1426),
            ("Colin", 1672),
            ("Colinet", 767),
            ("Collard", 3974),
            ("Collart", 918),
            ("Collet", 1564),
            ("Collette", 1270),
            ("Collignon", 2295),
            ("Collin", 2840),
            ("Colot", 636),
            ("Colson", 1244),
            ("Compère", 608),
            ("Constant", 1027),
            ("Coppens", 1062),
            ("Cordier", 1393),
            ("Corman", 678),
            ("Cornet", 2727),
            ("Cornez", 847),
            ("Cornil", 666),
            ("Coulon", 992),
            ("Courtois", 1026),
            ("Couvreur", 684),
            ("Crahay", 596),
            ("Cremer", 656),
            ("Culot", 1023),
            ("Cuvelier", 1930),
            ("Dardenne", 1552),
            ("David", 1588),
            ("De Backer", 822),
            ("De Clercq", 794),
            ("De Coster", 683),
            ("De Smet", 983),
            ("De Vos", 828),
            ("Debroux", 1018),
            ("Decamps", 662),
            ("Dechamps", 1124),
            ("Declercq", 1252),
            ("Degueldre", 612),
            ("Dehon", 951),
            ("Dejardin", 797),
            ("Delannoy", 679),
            ("Delattre", 962),
            ("Delcourt", 1753),
            ("Delcroix", 1032),
            ("Delforge", 1459),
            ("Delfosse", 2058),
            ("Delhaye", 2192),
            ("Delhez", 1112),
            ("Delmotte", 1462),
            ("Delporte", 1004),
            ("Deltour", 635),
            ("Delvaux", 3124),
            ("Delvigne", 768),
            ("Demanet", 601),
            ("Demaret", 1321),
            ("Demir", 615),
            ("Demoulin", 1811),
            ("Denis", 5205),
            ("Deprez", 2034),
            ("Descamps", 2249),
            ("Desmet", 1576),
            ("Dessart", 861),
            ("Dessy", 754),
            ("Detaille", 649),
            ("Dethier", 2057),
            ("Detournay", 598),
            ("Detry", 842),
            ("Devaux", 698),
            ("Deville", 645),
            ("Devillers", 1019),
            ("Devos", 2042),
            ("Dewez", 804),
            ("Diallo", 1936),
            ("Dieu", 1081),
            ("Dieudonné", 629),
            ("Docquier", 960),
            ("Donnay", 775),
            ("Dony", 606),
            ("Doumont", 664),
            ("Doyen", 1117),
            ("Druart", 1534),
            ("Dubois", 9160),
            ("Dubuisson", 1159),
            ("Duchesne", 1099),
            ("Duchêne", 681),
            ("Duez", 609),
            ("Dufour", 1734),
            ("Dufrane", 664),
            ("Dufrasne", 832),
            ("Dujardin", 1679),
            ("Dumont", 5757),
            ("Dumortier", 818),
            ("Dumoulin", 1217),
            ("Dupont", 6185),
            ("Dupuis", 1573),
            ("Duquesne", 865),
            ("Durant", 795),
            ("Durieux", 1622),
            ("Dury", 622),
            ("Dussart", 768),
            ("Dutrieux", 635),
            ("Duval", 596),
            ("Duvivier", 1191),
            ("Eloy", 806),
            ("Englebert", 1130),
            ("Ernst", 701),
            ("Etienne", 2150),
            ("Evrard", 4220),
            ("Fabry", 675),
            ("Fauconnier", 653),
            ("Feron", 784),
            ("Flament", 1286),
            ("Fontaine", 4156),
            ("Foucart", 1119),
            ("Franck", 822),
            ("Franssen", 795),
            ("François", 4779),
            ("Frère", 895),
            ("Gabriel", 973),
            ("Gaillard", 637),
            ("Gallez", 930),
            ("Gallo", 604),
            ("Gaspard", 1128),
            ("Gauthier", 917),
            ("Genot", 794),
            ("George", 733),
            ("Georges", 2503),
            ("Gerard", 1264),
            ("Ghislain", 1092),
            ("Gilbert", 879),
            ("Gillain", 1018),
            ("Gillard", 1729),
            ("Gilles", 2228),
            ("Gillet", 3330),
            ("Gilson", 2516),
            ("Glineur", 679),
            ("Gobert", 1732),
            ("Goblet", 775),
            ("Godart", 1622),
            ("Godefroid", 1654),
            ("Godfroid", 1129),
            ("Godin", 669),
            ("Goffaux", 775),
            ("Goffin", 2664),
            ("Goffinet", 951),
            ("Goossens", 1661),
            ("Gosset", 751),
            ("Gossiaux", 594),
            ("Graindorge", 709),
            ("Grandjean", 1230),
            ("Greco", 850),
            ("Grosjean", 961),
            ("Grégoire", 2866),
            ("Guillaume", 3266),
            ("Guiot", 880),
            ("Gustin", 1407),
            ("Gérard", 5196),
            ("Hainaut", 662),
            ("Halleux", 1061),
            ("Hannecart", 649),
            ("Hardy", 2007),
            ("Heinen", 1003),
            ("Hendrick", 773),
            ("Hendrickx", 971),
            ("Henin", 1003),
            ("Henrard", 1426),
            ("Henrion", 691),
            ("Henrotte", 610),
            ("Henry", 3357),
            ("Herman", 1954),
            ("Hermans", 1224),
            ("Honoré", 781),
            ("Hougardy", 597),
            ("Huart", 1229),
            ("Hubert", 3047),
            ("Huberty", 617),
            ("Hubin", 665),
            ("Huet", 713),
            ("Humblet", 945),
            ("Jacob", 1848),
            ("Jacobs", 2001),
            ("Jacquemart", 593),
            ("Jacquemin", 1768),
            ("Jacques", 2939),
            ("Jacquet", 1935),
            ("Jadot", 1514),
            ("Jadoul", 770),
            ("Jamar", 719),
            ("Janssen", 1396),
            ("Janssens", 2781),
            ("Joly", 1091),
            ("Joris", 1030),
            ("Kaya", 743),
            ("Kinet", 663),
            ("Klein", 872),
            ("Lacroix", 2320),
            ("Lahaye", 1258),
            ("Lallemand", 1440),
            ("Laloux", 1105),
            ("Lambert", 8375),
            ("Lambot", 872),
            ("Lambotte", 1056),
            ("Lambrechts", 750),
            ("Lamy", 837),
            ("Lange", 780),
            ("Lardinois", 1269),
            ("Latour", 603),
            ("Laurent", 5463),
            ("Lebeau", 747),
            ("Leblanc", 978),
            ("Lebon", 737),
            ("Lebrun", 2897),
            ("Leclercq", 5862),
            ("Lecocq", 1916),
            ("Lecomte", 2778),
            ("Lecoq", 650),
            ("Ledent", 811),
            ("Ledoux", 981),
            ("Leduc", 890),
            ("Lefebvre", 3263),
            ("Lefevre", 1086),
            ("Lefèbvre", 1178),
            ("Lefèvre", 2335),
            ("Legrain", 769),
            ("Legrand", 3524),
            ("Legros", 1723),
            ("Lejeune", 5316),
            ("Leleux", 779),
            ("Leloup", 640),
            ("Lemaire", 4304),
            ("Lemmens", 1155),
            ("Lemoine", 1298),
            ("Lempereur", 838),
            ("Lenaerts", 772),
            ("Lenoir", 1438),
            ("Leonard", 633),
            ("Lepage", 851),
            ("Leroy", 5162),
            ("Leruth", 608),
            ("Lheureux", 599),
            ("Lhoir", 755),
            ("Libert", 2432),
            ("Limbourg", 713),
            ("Liégeois", 1212),
            ("Liénard", 1054),
            ("Lombardo", 703),
            ("Lorent", 936),
            ("Louis", 3131),
            ("Lucas", 1095),
            ("Léonard", 3464),
            ("Maes", 2492),
            ("Mahieu", 2123),
            ("Mahy", 1030),
            ("Malherbe", 787),
            ("Maquet", 1023),
            ("Marchal", 3699),
            ("Marchand", 1353),
            ("Marechal", 623),
            ("Marion", 623),
            ("Marlier", 1005),
            ("Marquet", 796),
            ("Martens", 875),
            ("Martin", 6603),
            ("Mary", 875),
            ("Maréchal", 2673),
            ("Massart", 2283),
            ("Masset", 941),
            ("Massin", 673),
            ("Masson", 1964),
            ("Matagne", 1057),
            ("Materne", 754),
            ("Mathieu", 4723),
            ("Mathot", 830),
            ("Mathy", 1223),
            ("Maton", 816),
            ("Matthys", 611),
            ("Mercier", 1814),
            ("Mertens", 2483),
            ("Messina", 747),
            ("Meunier", 2564),
            ("Meurant", 692),
            ("Meurisse", 647),
            ("Meyer", 1205),
            ("Michaux", 2562),
            ("Michel", 4456),
            ("Michiels", 1712),
            ("Mignon", 870),
            ("Minet", 1237),
            ("Minne", 773),
            ("Monfort", 809),
            ("Moreau", 3287),
            ("Mortier", 693),
            ("Motte", 1067),
            ("Mottet", 637),
            ("Moulin", 1025),
            ("Moureau", 732),
            ("Mouton", 707),
            ("Muller", 1258),
            ("Müller", 789),
            ("Navez", 889),
            ("Neuville", 647),
            ("Nguyen", 980),
            ("Nicaise", 893),
            ("Nicolas", 1977),
            ("Nicolay", 786),
            ("Nizet", 699),
            ("Noel", 915),
            ("Noël", 4424),
            ("Nys", 754),
            ("Nyssen", 748),
            ("Oger", 694),
            ("Olivier", 1676),
            ("Orban", 1578),
            ("Paquay", 1354),
            ("Paquet", 1683),
            ("Paquot", 654),
            ("Parent", 1295),
            ("Parmentier", 1922),
            ("Paul", 989),
            ("Paulus", 1136),
            ("Pauwels", 1069),
            ("Peeters", 2527),
            ("Peters", 1305),
            ("Petit", 4547),
            ("Petitjean", 795),
            ("Philippart", 958),
            ("Philippe", 1313),
            ("Picard", 693),
            ("Pierard", 1192),
            ("Pierrard", 634),
            ("Pierre", 2059),
            ("Pierret", 843),
            ("Pierson", 718),
            ("Piette", 2415),
            ("Pirard", 1786),
            ("Pire", 1262),
            ("Piret", 1706),
            ("Pirlot", 954),
            ("Piron", 1949),
            ("Pirotte", 1917),
            ("Pirson", 1814),
            ("Piérard", 769),
            ("Pochet", 909),
            ("Polet", 689),
            ("Pollet", 597),
            ("Poncelet", 2599),
            ("Poncin", 1361),
            ("Raskin", 622),
            ("Remacle", 1741),
            ("Remy", 2715),
            ("Renard", 5209),
            ("Renier", 1057),
            ("Renson", 1120),
            ("Reuter", 647),
            ("Richard", 1322),
            ("Rigaux", 611),
            ("Rizzo", 922),
            ("Robert", 3519),
            ("Robin", 701),
            ("Roger", 650),
            ("Roland", 2283),
            ("Romain", 1173),
            ("Rossi", 722),
            ("Rousseau", 2630),
            ("Rousseaux", 660),
            ("Ruelle", 1040),
            ("Russo", 1002),
            ("Sacré", 1151),
            ("Salmon", 1055),
            ("Sauvage", 1222),
            ("Schmit", 996),
            ("Schmitz", 2588),
            ("Schoonbroodt", 674),
            ("Schumacher", 720),
            ("Schyns", 933),
            ("Segers", 693),
            ("Servais", 3367),
            ("Sevrin", 606),
            ("Simon", 6022),
            ("Simonis", 835),
            ("Simons", 713),
            ("Smets", 1236),
            ("Staquet", 737),
            ("Stas", 672),
            ("Stassin", 617),
            ("Stevens", 977),
            ("Theys", 709),
            ("Thibaut", 1402),
            ("Thirion", 1659),
            ("Thiry", 3688),
            ("Thomas", 3751),
            ("Thonon", 859),
            ("Thys", 1317),
            ("Timmermans", 1081),
            ("Toubeau", 632),
            ("Tournay", 683),
            ("Toussaint", 2986),
            ("Urbain", 1986),
            ("Valentin", 642),
            ("Van Damme", 823),
            ("Vandamme", 705),
            ("Vandenberghe", 861),
            ("Vanderheyden", 628),
            ("Vanderlinden", 634),
            ("Vandevelde", 772),
            ("Verhaeghe", 638),
            ("Vermeulen", 1207),
            ("Vigneron", 773),
            ("Vilain", 1027),
            ("Vincent", 1869),
            ("Vrancken", 632),
            ("Warnier", 739),
            ("Wathelet", 925),
            ("Wattiez", 713),
            ("Wauters", 1109),
            ("Wauthier", 928),
            ("Weber", 865),
            ("Wilkin", 729),
            ("Willame", 622),
            ("Willem", 816),
            ("Willems", 2515),
            ("Wilmart", 593),
            ("Wilmet", 841),
            ("Wilmotte", 695),
            ("Wouters", 1216),
            ("Wéry", 1094),
            ("Yildirim", 623),
            ("Yilmaz", 1096),
        )
    )
