#!/bin/python
# -*- coding: utf-8 -*-

# Fenrir TTY screen reader
# By Chrys, Storm Dragon, and contributors.

import re
import string
import time

from fenrirscreenreader.core import debug
from fenrirscreenreader.core.i18n import _
from fenrirscreenreader.utils import line_utils


class OutputManager:
    def __init__(self):
        self.last_echo = ""

    def initialize(self, environment):
        self.env = environment
        self.env["runtime"]["SettingsManager"].load_driver(
            self.env["runtime"]["SettingsManager"].get_setting(
                "speech", "driver"
            ),
            "SpeechDriver",
        )
        self.env["runtime"]["SettingsManager"].load_driver(
            self.env["runtime"]["SettingsManager"].get_setting(
                "sound", "driver"
            ),
            "SoundDriver",
        )

    def shutdown(self):
        self.env["runtime"]["SettingsManager"].shutdown_driver("SoundDriver")
        self.env["runtime"]["SettingsManager"].shutdown_driver("SpeechDriver")

    def present_text(
        self,
        text,
        interrupt=True,
        sound_icon="",
        ignore_punctuation=False,
        announce_capital=False,
        flush=True,
    ):
        if text == "":
            return
        if (
            self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "speech", "readNumbersAsDigits"
            )
            and len(text.strip()) > 1
        ):
            text = re.sub(r"(\d)", r"\1 ", text).rstrip()
        self.env["runtime"]["DebugManager"].write_debug_out(
            "present_text:\nsoundIcon:'" + sound_icon + "'\nText:\n" + text,
            debug.DebugLevel.INFO,
        )
        if self.play_sound_icon(sound_icon, interrupt):
            self.env["runtime"]["DebugManager"].write_debug_out(
                "sound_icon found", debug.DebugLevel.INFO
            )
            return
        if (len(text) > 1) and (text.strip(string.whitespace) == ""):
            return
        to_announce_capital = announce_capital and text[0].isupper()
        if to_announce_capital:
            if self.play_sound_icon("capital", False):
                to_announce_capital = False
        self.last_echo = text
        self.speak_text(
            text, interrupt, ignore_punctuation, to_announce_capital
        )

    def get_last_echo(self):
        return self.last_echo

    def speak_text(
        self,
        text,
        interrupt=True,
        ignore_punctuation=False,
        announce_capital=False,
    ):
        if not self.env["runtime"]["SettingsManager"].get_setting_as_bool(
            "speech", "enabled"
        ):
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Speech disabled in OutputManager.speak_text",
                debug.DebugLevel.INFO,
            )
            return
        if self.env["runtime"]["SpeechDriver"] is None:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "No SpeechDriver in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            return
        if interrupt:
            self.interrupt_output()
        try:
            self.env["runtime"]["SpeechDriver"].set_language(
                self.env["runtime"]["SettingsManager"].get_setting(
                    "speech", "language"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "setting speech language in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            self.env["runtime"]["SpeechDriver"].set_voice(
                self.env["runtime"]["SettingsManager"].get_setting(
                    "speech", "voice"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Error while setting speech voice in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            if announce_capital:
                self.env["runtime"]["SpeechDriver"].set_pitch(
                    self.env["runtime"][
                        "SettingsManager"
                    ].get_setting_as_float("speech", "capitalPitch")
                )
            else:
                self.env["runtime"]["SpeechDriver"].set_pitch(
                    self.env["runtime"][
                        "SettingsManager"
                    ].get_setting_as_float("speech", "pitch")
                )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "setting speech pitch in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            self.env["runtime"]["SpeechDriver"].set_rate(
                self.env["runtime"]["SettingsManager"].get_setting_as_float(
                    "speech", "rate"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "setting speech rate in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            self.env["runtime"]["SpeechDriver"].set_module(
                self.env["runtime"]["SettingsManager"].get_setting(
                    "speech", "module"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "setting speech module in OutputManager.speak_text",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            self.env["runtime"]["SpeechDriver"].set_volume(
                self.env["runtime"]["SettingsManager"].get_setting_as_float(
                    "speech", "volume"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "setting speech volume in OutputManager.speak_text ",
                debug.DebugLevel.ERROR,
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

        try:
            if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
                "general", "newLinePause"
            ):
                clean_text = text.replace("\n", " , ")
            else:
                clean_text = text.replace("\n", " ")

            clean_text = self.env["runtime"]["TextManager"].replace_head_lines(
                clean_text
            )
            clean_text = self.env["runtime"][
                "PunctuationManager"
            ].proceed_punctuation(clean_text, ignore_punctuation)
            clean_text = re.sub(" +$", " ", clean_text)
            self.env["runtime"]["SpeechDriver"].speak(
                clean_text, True, ignore_punctuation
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Speak: " + clean_text, debug.DebugLevel.INFO
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                '"speak" in OutputManager.speak_text ', debug.DebugLevel.ERROR
            )
            self.env["runtime"]["DebugManager"].write_debug_out(
                str(e), debug.DebugLevel.ERROR
            )

    def interrupt_output(self):
        try:
            self.env["runtime"]["SpeechDriver"].cancel()
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Interrupt speech", debug.DebugLevel.INFO
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager interrupt_output: Error canceling speech: "
                + str(e),
                debug.DebugLevel.ERROR,
            )

    def play_sound_icon(self, sound_icon="", interrupt=True):
        if sound_icon == "":
            return False
        sound_icon = sound_icon.upper()
        if not self.env["runtime"]["SettingsManager"].get_setting_as_bool(
            "sound", "enabled"
        ):
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Sound disabled in OutputManager.play_sound_icon",
                debug.DebugLevel.INFO,
            )
            return False

        try:
            e = self.env["soundIcons"][sound_icon]
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager play_sound_icon: SoundIcon does not exist "
                + sound_icon
                + ": "
                + str(e),
                debug.DebugLevel.WARNING,
            )
            return False

        if self.env["runtime"]["SoundDriver"] is None:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "No SoundDriver in OutputManager.play_sound_icon:  SoundDriver not loaded",
                debug.DebugLevel.ERROR,
            )
            return False

        try:
            self.env["runtime"]["SoundDriver"].set_volume(
                self.env["runtime"]["SettingsManager"].get_setting_as_float(
                    "sound", "volume"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager.play_sound_icon::set_volume: " + str(e),
                debug.DebugLevel.ERROR,
            )

        try:
            self.env["runtime"]["SoundDriver"].play_sound_file(
                self.env["soundIcons"][sound_icon], interrupt
            )
            return True
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager.play_sound_icon::play_sound_file: " + str(e),
                debug.DebugLevel.ERROR,
            )
            return False

        return False

    def play_frequence(self, frequence, duration, interrupt=True):
        if not self.env["runtime"]["SettingsManager"].get_setting_as_bool(
            "sound", "enabled"
        ):
            self.env["runtime"]["DebugManager"].write_debug_out(
                "Sound disabled in OutputManager.play_frequence",
                debug.DebugLevel.INFO,
            )
            return False

        if frequence < 1 or frequence > 20000:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager.play_frequence::Filefrequence is out of range:"
                + str(frequence),
                debug.DebugLevel.INFO,
            )
            return False

        if self.env["runtime"]["SoundDriver"] is None:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "No SoundDriver in OutputManager.play_frequence: SoundDriver not loaded",
                debug.DebugLevel.ERROR,
            )
            return False

        try:
            self.env["runtime"]["SoundDriver"].set_volume(
                self.env["runtime"]["SettingsManager"].get_setting_as_float(
                    "sound", "volume"
                )
            )
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager.play_sound_icon::set_volume: " + str(e),
                debug.DebugLevel.ERROR,
            )
        adjust_volume = 0.0
        try:
            adjust_volume = 1.0 - (frequence / 20000)
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager play_frequence: Error calculating adjust volume: "
                + str(e),
                debug.DebugLevel.ERROR,
            )
        if adjust_volume > 9.0:
            adjust_volume = 9.0

        try:
            self.env["runtime"]["SoundDriver"].play_frequence(
                frequence, duration, adjust_volume, interrupt
            )
            return True
        except Exception as e:
            self.env["runtime"]["DebugManager"].write_debug_out(
                "OutputManager.play_sound_icon::play_sound_file: " + str(e),
                debug.DebugLevel.ERROR,
            )
            return False

        return False

    def temp_disable_speech(self):
        if self.env["runtime"]["SettingsManager"].get_setting_as_bool(
            "speech", "enabled"
        ):
            self.present_text(
                _("speech temporary disabled"),
                sound_icon="SpeechOff",
                interrupt=True,
            )
            self.env["commandBuffer"]["enableSpeechOnKeypress"] = True
            # Also enable prompt watching for automatic speech restoration
            if "silenceUntilPrompt" not in self.env["commandBuffer"]:
                self.env["commandBuffer"]["silenceUntilPrompt"] = False
            self.env["commandBuffer"]["silenceUntilPrompt"] = True
            self.env["runtime"]["SettingsManager"].set_setting(
                "speech",
                "enabled",
                str(
                    not self.env["runtime"][
                        "SettingsManager"
                    ].get_setting_as_bool("speech", "enabled")
                ),
            )
            self.interrupt_output()

    def announce_active_cursor(self, interrupt_p=False):
        if self.env["runtime"]["CursorManager"].is_review_mode():
            self.present_text(" review cursor ", interrupt=interrupt_p)
        else:
            self.present_text(" text cursor ", interrupt=interrupt_p)

    def reset_SpeechDriver(self):
        """Reset speech driver to clean state - called by settings_manager"""
        if (
            "SpeechDriver" in self.env["runtime"]
            and self.env["runtime"]["SpeechDriver"]
        ):
            try:
                self.env["runtime"]["SpeechDriver"].reset()
                self.env["runtime"]["DebugManager"].write_debug_out(
                    "Speech driver reset successfully", debug.DebugLevel.INFO
                )
            except Exception as e:
                self.env["runtime"]["DebugManager"].write_debug_out(
                    f"reset_SpeechDriver error: {e}", debug.DebugLevel.ERROR
                )
